﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Drawing;
using Editor.DefaultParser;

namespace Editor
{
    class PythonTextBoxController
    {
        Form _ownerForm;

        IPythonTextBox _textBox;
        ITextOutout _textOutput;
        
        Keys _lastKey;

        internal event EventHandler<TextChangedEventArgs> TextChanged;
        internal event KeyPressEventHandler KeyPress;
        internal event KeyEventHandler KeyDown;
        internal event EventHandler Initialized;
        internal event EventHandler LocationChangedAndWindowLocationChanged;

        CommandListHistory _history = new CommandListHistory();
        CommandListHistoryIterator _iterator;
        bool _isUpdating;

        internal PythonTextBoxController(IPythonTextBox textBox, ITextOutout textOutput)
        {
            _textBox = textBox;
            _textOutput = textOutput;
            _textBox.KeyDown += new KeyEventHandler(_textBox_KeyDown);
            _textBox.KeyPress += new KeyPressEventHandler(_textBox_KeyPress);
            _textBox.TextChanged += new EventHandler(_textBox_TextChanged);
            _textBox.LocationChangedAndWindowLocationChanged += new EventHandler(_textBox_LocationChangedAndWindowLocationChanged);
            _iterator = _history.GetEndIterator();
        }

        void _textBox_LocationChangedAndWindowLocationChanged(object sender, EventArgs e)
        {
            OnLocationChangedAndWindowLocationChanged();
        }

        void OnLocationChangedAndWindowLocationChanged()
        {
            if (LocationChangedAndWindowLocationChanged != null)
            {
                LocationChangedAndWindowLocationChanged(this, new EventArgs());
            }
        }

        void _textBox_TextChanged(object sender, EventArgs e)
        {
            if (_isUpdating)
            {
                return;
            }
            string text;
            int cursorIndex;
            _textBox.GetText(out text, out cursorIndex);

            OnTextChanged(text, _lastKey, cursorIndex);
        }

        private void OnTextChanged(string text, Keys lastKey, int cursorIndex)
        {
            if (TextChanged != null)
            {
                TextChangedEventArgs e = new TextChangedEventArgs();
                e.Text = text;
                e.LastKey = lastKey;
                e.CursorIndex = cursorIndex;
                TextChanged(this, e);
            }
        }
        private void OnKeyPress(KeyPressEventArgs e)
        {
            if (KeyPress != null)
            {
                KeyPress(this, e);
            }
        }
        private void OnKeyDown(KeyEventArgs e)
        {
            if (KeyDown != null)
            {
                KeyDown(this, e);
            }
        }
        private void OnInitialized()
        {
            if (Initialized != null)
            {
                Initialized(this, new EventArgs());
            }
        }

        void _textBox_KeyPress(object sender, KeyPressEventArgs e)
        {


            OnKeyPress(e);
            return;
        }

        void _textBox_KeyDown(object sender, KeyEventArgs e)
        {
            _lastKey = e.KeyCode;

            OnKeyDown(e);
            if (e.Handled == false)
            {
                if (_lastKey == Keys.Up)
                {
                    string text = _iterator.GetValue();

                    SetText(text, text.Length);
                    _textBox.SelectAll();

                    _iterator.Decrement();
                    e.Handled = true;
                }
                else if (_lastKey == Keys.Down)
                {
                    string text = _iterator.GetValue();

                    SetText(text, text.Length);
                    _textBox.SelectAll();

                    _iterator.Increment();
                    e.Handled = true;
                }
            }
            return;
        }

        internal void SetText(string text, int currentIndex)
        {
            _isUpdating = true;
            _textBox.SetText(text, currentIndex);
            _isUpdating = false;
        }

        internal Point GetBasePositoin()
        {
            return _textBox.PointToScreen(new Point(0, 0));
        }

        internal void GetText(out string input, out int cursor)
        {
            _textBox.GetText(out input, out cursor);
        }

        internal void SetCommandHistory(string input)
        {
            _history.Add(input);
            _iterator = _history.GetEndIterator();
        }
    }

    class TextChangedEventArgs : EventArgs
    {
        internal string Text;
        internal Keys LastKey;
        internal int CursorIndex;
    }
}
