﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Reflection;
using System.Text.RegularExpressions;

namespace Editor
{
    class XmlHelpCache
    {
        Dictionary<Assembly, XmlHelpCacheContent> _cache = new Dictionary<Assembly, XmlHelpCacheContent>();
        internal void AddXml(Assembly asm, string xmlFile)
        {
            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(xmlFile);
            var content = new XmlHelpCacheContent(xmlDoc);
            _cache[asm] = content;
        }

        internal string GetHelp(MethodBase method)
        {
            XmlHelpCacheContent content = null;
            bool result = _cache.TryGetValue(method.DeclaringType.Assembly, out content);
            if (result == false)
            {
                return "";
            }
            return content.GetHelp(method);
        }

        internal bool Contains(Assembly assembly)
        {
            XmlHelpCacheContent content;
            return _cache.TryGetValue(assembly, out content);
        }


    }

    class XmlHelpCacheContent
    {
        XmlDocument _xDoc = new XmlDocument();
        Regex regex1 = new Regex("<see cref=\"T:(?<cls>[^\"]+)\" */>", RegexOptions.Compiled);
        Regex regex2 = new Regex("<see cref=\"T:(?<cls>[^\"]+)\" *> *</see>", RegexOptions.Compiled);
        internal XmlHelpCacheContent(XmlDocument xDoc)
        {
            _xDoc = xDoc;
        }

        internal string GetHelp(MethodBase method)
        {
            StringBuilder str = new StringBuilder();
            string key = CreateKey(method);
            XmlNode node = _xDoc.SelectSingleNode(string.Format("/doc/members/member[@name=\"{0}\"]/summary", key));
            if (node == null)
            {
                return "";
            }
            string text = node.InnerXml.Trim();
            text = ReplaceXmlToText(text);

            str.Append(text + "\r\n");
            
            return str.ToString();
        }

        private string ReplaceXmlToText(string text)
        {
            if (regex1.IsMatch(text))
            {
                MatchCollection matches = regex1.Matches(text);
                foreach (Match match in matches)
                {
                    string replaceText = match.Groups[1].Value;
                    replaceText = TypeNameService.ToSimpleTypeName(replaceText);
                    text = text.Replace(match.Value, replaceText);
                }
            }
            if (regex2.IsMatch(text))
            {
                MatchCollection matches = regex2.Matches(text);
                foreach (Match match in matches)
                {
                    string replaceText = match.Groups[1].Value;
                    replaceText = TypeNameService.ToSimpleTypeName(replaceText);
                    text = text.Replace(match.Value, replaceText);
                }
            }
            return text;
        }

        private string CreateKey(MethodBase method)
        {
           
            StringBuilder str = new StringBuilder();
            str.Append("M:");
            str.Append(method.DeclaringType.Namespace+ "." + method.DeclaringType.Name);
            str.Append(".");
            if (method.IsConstructor)
            {
                str.Append("#ctor");
            }
            else
            {
                str.Append(method.Name);
            }
            var parameters = method.GetParameters();
            if (parameters.Length != 0)
            {
                str.Append("(");
                for (int i = 0; i < parameters.Length; i++)
                {
                    if (i != 0)
                    {
                        str.Append(",");
                    }
                    
                    str.Append(CreateGenericParameterKey(parameters[i].ParameterType));
                }
                str.Append(")");
            }
            return str.ToString();
        }

        private string CreateGenericParameterKey(Type type)
        {
            Type[] ts = type.GetGenericArguments();
            if (ts.Length == 0)
            {
                return type.Namespace + "." + type.Name;
            }
            else
            {
                Regex regex = new Regex(@"`.+");
                string typeNameWithoutBackCotation = regex.Replace(type.Name, "");
                StringBuilder str = new StringBuilder();
                str.Append(type.Namespace);
                str.Append(".");
                str.Append(typeNameWithoutBackCotation);
                str.Append("{");
                for (int i = 0; i < ts.Length; i++)
                {
                    if (i != 0)
                    {
                        str.Append(",");
                    }
                    str.Append("`" + i);
                }
                str.Append("}");

                return str.ToString();
            }
            
        }
    }
}
