unit tn_app;

interface

uses
    windows, messages, tn_classes;

type
    PMessage        =   ^TMessage;

    TNUniqueID       =   class
        private
            IDs : TNIntList;
            Pointers : TNIntList;
        public
            constructor Create;
            destructor Destroy; override;
            function RequestID(P : Pointer) : Integer;
            function GetPointerFromID(ID : Integer) : Pointer;
            procedure DeleteID(ID : Integer);
            procedure Add(ID : Integer ; P : Pointer);
    end;

    TNItem          =   class
        private
            procedure WindowProc(var Message : TMessage); virtual;
        public
            Handle : THandle;
    end;

    TNControl       =   class
        private
            procedure WMNCDestroy(var Message: TMessage); message WM_NCDESTROY;
            procedure WMCommand(var Message: TMessage); message WM_COMMAND;
            //procedure WMMenuSelect(var Message : TMessage); message WM_MENUSELECT;
        public
            Handle : THandle;
            ParentHandle : THandle;
            MainWindow : Boolean;
            constructor Create(Name : String; Parent : THandle);
            destructor Destroy; override;
            procedure WindowProc(var Message: TMessage);
            procedure DefaultHandler(var Message); override;
    end;

    TNMenu          =   class(TNItem)
        private
            FOnClick : TNotifyEvent;
        public
            Items : array of TNMenu;
            Count : Integer;
            constructor Create;
            procedure Clear;
            procedure WindowProc(var Message : TMessage); override;
            property OnClick : TNotifyEvent read FOnClick write FOnClick;
    end;

    TNMenuItem      =   class(TNMenu)
        private
            mif : MENUITEMINFO;
            FCaption : String;
            Parent : TNMenu;
            ID : Integer;
            procedure SetCaption(Value : String);
        public
            Index : Integer;
            constructor Create(Owner : TNMenu); overload;
            constructor Create(Owner : TNMenu ; Index : Integer); overload;
            destructor Destroy; override;
            function Add : TNMenuItem;
            property Caption : String read FCaption write SetCaption;
    end;

    TNPopupMenu     =   class(TNMenu)
        public
            constructor Create;
            destructor Destroy; override;
            function Add : TNMenuItem;
    end;

    TNMainMenu     =   class(TNMenu)
        public
            constructor Create;
            destructor Destroy; override;
            function Add : TNMenuItem;
    end;

    TNResourceMenu  =   class(TNMenu)
        public
            constructor Create;
            destructor Destroy; override;
            procedure AssignMenu(Value : THandle);
    end;
var
    UniqueID : TNUniqueID;

implementation

function StdWndProc(Handle: HWND; Message, WParam: Longint; LParam: Longint) : Longint; stdcall;
var
    P : Pointer;
    Msg : TMessage;
begin
    P := Pointer(GetProp(Handle,'Object'));
    Msg.LParam := LParam;
    Msg.Msg := Message;
    Msg.WParam := WParam;
    TNControl(P).WindowProc(Msg);
    Result := Msg.Result;
end;

//TNUniqueID

constructor TNUniqueID.Create;
begin
    IDs := TNIntList.Create;
    Pointers := TNIntList.Create;
end;

destructor TNuniqueID.Destroy;
begin
    IDs.Free;
    Pointers.Free;
    inherited Destroy;
end;

procedure TNUniqueID.Add(ID : Integer ; P : Pointer);
var
    Index : Integer;
begin
    Index := IDs.IndexOf(ID);

    if Index = -1 then
    begin
        IDs.Add(ID);
        Pointers.Add(Integer(P));
    end;
end;

function TNUniqueID.RequestID(P : Pointer) : Integer;
var
    n : Integer;
    Index : Integer;
begin
    Result := -1;
    
    for n := 1000 to 65535 do
    begin
        Index := IDs.IndexOf(n);
        if Index = -1 then
        begin
            IDs.Add(n);
            Pointers.Add(Integer(P));
            Result := n;
            Break;
        end;
    end;
end;

function TNUniqueID.GetPointerFromID(ID : Integer) : Pointer;
var
    Index : Integer;
begin
    Result := nil;
    Index := IDs.IndexOf(ID);
    if Index <> -1 then
        Result := Pointer(Pointers.Values[Index]);
end;

procedure TNUniqueID.DeleteID(ID : Integer);
var
    Index : Integer;
begin
    Index := IDs.IndexOf(ID);
    if Index <> -1 then
    begin
        IDs.Delete(Index);
        Pointers.Delete(Index);
    end;
end;

//TNItem

procedure TNItem.WindowProc(var Message : TMessage);
begin

end;

//TNControl

constructor TNControl.Create(Name : String ; Parent : THandle);
const
    WINNAME     =   'TNWINDOW';
var
    TempClass: WNDCLASS;
    Reg : Boolean;
begin
    if Name = '' then
    begin
        Reg := GetClassInfo(HInstance, WINNAME, TempClass);
        if not Reg or (TempClass.lpfnWndProc <> @DefWindowProc) then
        begin
            FillChar((@TempClass)^,SizeOf(WNDCLASS),#0);
            TempClass.hInstance := hInstance;
            TempClass.lpfnWndProc := @DefWindowProc;
            TempClass.lpszClassName := WINNAME;
            TempClass.hbrBackground := COLOR_WINDOW;
            TempClass.style := CS_OWNDC or CS_VREDRAW or CS_HREDRAW;
            TempClass.hCursor := LoadCursor(0, IDC_ARROW);

            if Reg then
                Windows.UnregisterClass(WINNAME, hInstance);

            RegisterClass(TempClass);
        end;

        Handle := CreateWindowEx(WS_EX_ACCEPTFILES or WS_EX_APPWINDOW or WS_EX_WINDOWEDGE
                                , WINNAME, nil
                                , WS_SYSMENU or WS_BORDER or WS_MAXIMIZEBOX or WS_MINIMIZEBOX or WS_SIZEBOX
                                , 0, 0
                                , 0, 0, Parent, 0, hInstance, nil);

        SetProp(Handle,'Object',Cardinal(Self));
        SetWindowLong(Handle, GWL_WNDPROC, Longint(@StdWndProc));
    end
    else
    begin
        Reg := GetClassInfo(HInstance, PChar(Name), TempClass);
        if not Reg then Exit;

        Handle := CreateWindow(PChar(Name)
                               ,'Push Button'
                                ,WS_CHILD or WS_VISIBLE or BS_NOTIFY or BS_PUSHBUTTON
                                ,40,10,120,32,Parent,1,hInstance,nil);
    end;

    ParentHandle := Parent;
end;

destructor TNControl.Destroy;
begin
    if Handle <> 0 then
        DestroyWindow(Handle);
    inherited Destroy;
end;

procedure TNControl.WindowProc(var Message: TMessage);
begin
    Message.Result := 0;
    if Message.Msg = WM_WINDOWPOSCHANGED then
        DefWindowProc(Handle, Message.Msg, Message.WParam, Message.LParam)
    else
        Dispatch(Message);
end;

procedure TNControl.DefaultHandler(var Message);
var
    Msg : PMessage;
begin
    Msg := @Message;
    Msg^.Result := DefWindowProc(Handle
                                    ,Msg^.Msg
                                    ,Msg^.WParam
                                    ,Msg^.LParam);
    inherited DefaultHandler(Message);
end;

procedure TNControl.WMNCDestroy(var Message: TMessage);
begin
    RemoveProp(Handle,'Object');
    if MainWindow then
        PostQuitMessage(0);
    Message.Result := 1;
    inherited;
end;

procedure TNControl.WMCommand(var Message: TMessage);
var
    P : Pointer;
begin
    P := Pointer(GetProp(Message.LParam,'Object'));
    if P <> nil then
        TNItem(P).WindowProc(Message)
    else
    begin
        P := UniqueID.GetPointerFromID(Message.WParamLo);
        if P <> nil then
            TNItem(P).WindowProc(Message);
    end;

    inherited;
end;

//TNMenu

constructor TNMenu.Create;
begin
    inherited Create;
    Handle := 0;
end;

procedure TNMenu.Clear;
var
    n : Integer;
begin
    for n := 0 to Count - 1 do
        Items[n].Free;

    Count := 0;
    SetLength(Items, Count);
end;

procedure TNMenu.WindowProc(var Message : TMessage);
begin
    if Assigned(FOnClick) then
        FOnClick(Self);
end;


//TNMenuItem

constructor TNMenuItem.Create(Owner : TNMenu);
begin
    inherited Create;

    if Owner = nil then Exit;

    Handle := 0;
    Parent := Owner;
    Index := Parent.Count;

    FillChar((@mif)^, SizeOf(MENUITEMINFO), #0);
    FCaption := '';
    ID := UniqueID.RequestID(Self);

    mif.cbSize := SizeOf(MENUITEMINFO);
    mif.cch := Length(FCaption);
    mif.dwTypeData := PChar(FCaption);
    mif.fMask := MIIM_CHECKMARKS or MIIM_DATA or MIIM_ID or
                    MIIM_STATE or MIIM_SUBMENU or MIIM_TYPE;
    mif.fType := MFT_STRING;
    mif.wID := ID;

    InsertMenuItem(Parent.Handle, Index, True, mif);
end;

constructor TNMenuItem.Create(Owner : TNMenu ; Index : Integer);
var
    n : Integer;
    cnt : Integer;
    Item : TNMenuItem;
begin
    inherited Create;
    if Owner = nil then Exit;

    Handle := 0;
    cnt := 0;

    FillChar((@mif)^, SizeOf(MENUITEMINFO), #0);

    SetLength(FCaption, 513);
    n := GetMenuString(Owner.Handle
                                , Index
                                , PChar(FCaption)
                                , 512
                                ,MF_BYPOSITION);
    SetLength(FCaption, n);

    ID := GetMenuItemID(Owner.Handle, Index);
    if ID <> -1 then
        UniqueID.Add(ID, Self)
    else
    begin
        Handle := GetSubMenu(Owner.Handle, Index);
        cnt := GetMenuItemCount(Handle);
    end;

    if Handle <> 0 then
    begin
        for n := 0 to cnt - 1 do
        begin
            Item := TNMenuItem.Create(Self, n);

            SetLength(Items, Count+1);
            Items[Count] := Item;
            Count := Count + 1;
        end;
    end;

    mif.cbSize := SizeOf(MENUITEMINFO);
    mif.cch := Length(FCaption);
    mif.dwTypeData := PChar(FCaption);
    mif.hSubMenu := Handle;
    mif.wID := ID;
end;

destructor TNMenuItem.Destroy;
begin
    Clear;
    if Handle <> 0 then
        DestroyMenu(Handle);

    UniqueID.DeleteID(ID);
    inherited Destroy;
end;

function TNMenuItem.Add : TNMenuItem;
begin
    if Handle = 0 then
    begin
        Handle := CreatePopupMenu;
        mif.hSubMenu := Handle;
        SetMenuItemInfo(Parent.Handle, Index, True, mif);
    end;

    Result := TNMenuItem.Create(Self);

    SetLength(Items, Count+1);
    Items[Count] := Result;
    Count := Count + 1;
end;

procedure TNMenuItem.SetCaption(Value : String);
begin
    FCaption := Value;
    mif.cch := Length(FCaption);
    mif.dwTypeData := PChar(FCaption);
    SetMenuItemInfo(Parent.Handle,ID,False,mif);
end;

//TNPopupMenu

constructor TNPopupMenu.Create;
begin
    inherited Create;
    Count := 0;
    Handle := CreatePopupMenu;
end;

destructor TNPopupMenu.Destroy;
begin
    Clear;
    DestroyMenu(Handle);
    inherited Destroy;
end;

function TNPopupMenu.Add : TNMenuItem;
begin
    Result := TNMenuItem.Create(Self);
    SetLength(Items, Count+1);
    Items[Count] := Result;
    Count := Count + 1;
end;

//TNMainMenu

constructor TNMainMenu.Create;
begin
    inherited Create;
    Count := 0;
    Handle := CreateMenu;
end;

destructor TNMainMenu.Destroy;
begin
    Clear;
    DestroyMenu(Handle);
    inherited Destroy;
end;

function TNMainMenu.Add : TNMenuItem;
begin
    Result := TNMenuItem.Create(Self);
    SetLength(Items, Count+1);
    Items[Count] := Result;
    Count := Count + 1;
end;

//TNResourceMenu

constructor TNResourceMenu.Create;
begin
    inherited Create;
    Count := 0;
    Handle := 0;
end;

destructor TNResourceMenu.Destroy;
begin
    Clear;
    if Handle <> 0 then
        DestroyMenu(Handle);
    inherited Destroy;
end;

procedure TNResourceMenu.AssignMenu(Value : THandle);
var
    n : Integer;
    cnt : Integer;
    mif : MENUITEMINFO;
    Item : TNMenuItem;
begin
    if Handle <> 0 then Exit;

    FillChar((@mif)^, SizeOf(MENUITEMINFO), #0);
    Handle := Value;
    cnt := GetMenuItemCount(Handle);

    for n := 0 to cnt - 1 do
    begin
        Item := TNMenuItem.Create(Self, n);

        SetLength(Items, Count+1);
        Items[Count] := Item;
        Count := Count + 1;
    end;
end;

initialization
    UniqueID := TNUniqueID.Create;

finalization
    UniqueID.Free;
end.
