unit tn_lang;

interface

uses
    windows, tn_utils, tn_classes, tn_internet;

type
    TLangItem           =   record
        Name : String;
        Instance : HINST;
    end;

    TLanguageDriver     =   class
        private
            Count : Integer;
            Langs : array of TLangItem;
            LibFiles : TNIntList;
            procedure Install(const Path : String);
            procedure AddLanguage(const Name : String ; Inst : HINST);
            function GetLangIndex(const Name : String) : Integer;
        public
            constructor Create(const Path : String);
            destructor Destroy; override;
            function EncodeString(const Language, Source : String) : String;
            function DecodeString(const Language, Source : String) : String;
            function BEncode(const Source, Charset : String) : String;
            function BDecode(const Source : String) : String;
            function HeaderBEncode(const Source, Charset : String) : String;
    end;

    TRegistLanguage     =   function(Index : Integer ; Name : PChar) : BOOL; stdcall;
    TSystemToEncode     =   function(Src : PChar ; SrcSize : Integer ; var Dest : PChar) : Integer; stdcall;
    TDecodeToSystem     =   function(Src : PChar ; SrcSize : Integer ; var Dest : PChar) : Integer; stdcall;
    TFreeMemory         =   procedure (Buffer : PChar ; Size : Integer); stdcall;

implementation

constructor TLanguageDriver.Create(const Path : String);
begin
    Count := 0;
    SetLength(Langs, 0);
    LibFiles := TNIntList.Create;
    Install(Path);
end;

destructor TLanguageDriver.Destroy;
var
    n : Integer;
begin
    for n := 0 to LibFiles.Count - 1 do
        FreeLibrary(HINST(IntCard(LibFiles.GetValue(n))));

    Count := 0;
    SetLength(Langs, 0);
end;

procedure TLanguageDriver.Install(const Path : String);
var
    Lists : TNStringList;
    m : Integer;
    n : Integer;
    P : Pointer;
    Inst : HINST;
    Name : array [0..255] of Char;
    GetLangName : TRegistLanguage;
begin
    Lists := SearchFile(Path + '*.drv');

    for n := 0 to Lists.Count - 1 do
    begin
        Inst := LoadLibrary(PChar(Path + Lists.Strings[n]));
        LibFiles.Add(CardInt(Inst));
        P := GetProcAddress(Inst, 'RegistLanguage');

        if P = nil then Break;

        for m := 0 to 10 do
        begin
            FillChar(Name, 256, #0);
            GetLangName := TRegistLanguage(P);
            if not GetLangName(m, Name) then Break;
            AddLanguage(UpperCase(Name), Inst);
        end;
    end;

    Lists.Free;
end;

procedure TLanguageDriver.AddLanguage(const Name : String ; Inst : HINST);
begin
    SetLength(Langs, Count + 1);
    Langs[Count].Name := Name;
    Langs[Count].Instance := Inst;
    Count := Count + 1;
end;

function TLanguageDriver.GetLangIndex(const Name : String) : Integer;
var
    n : Integer;
begin
    Result := -1;

    for n := 0 to Count - 1 do
    begin
        if UpperCase(Langs[n].Name) = UpperCase(Trim(Name)) then
        begin
            Result := n;
            Break;
        end;
    end;

end;

function TLanguageDriver.EncodeString(const Language, Source : String) : String;
var
    temp : String;
    Index : Integer;
    P : Pointer;
    EncodeData : TSystemToEncode;
    FreeMemory : TFreeMemory;
    Buf : PChar;
    Size : Integer;
begin
    //GR[h
    Result := '';
    if Source = '' then Exit;
    temp := UpperCase(Language);
    Index := GetLangIndex(Language);

    if Index <> -1 then
    begin
        P := GetProcAddress(Langs[Index].Instance, 'SystemToEncode');
        if P = nil then Exit;
        EncodeData := TSystemToEncode(P);
        Size := EncodeData(PChar(Source), Length(Source), Buf);

        if Size = 0 then
            Result := ''
        else
            Result := String(Buf);

        P := GetProcAddress(Langs[Index].Instance, 'FreeMemory');
        if P = nil then Exit;
        FreeMemory := TFreeMemory(P);
        FreeMemory(Buf, Size);
    end
    else
        Result := Source;
end;

function TLanguageDriver.DecodeString(const Language, Source : String) : String;
var
    temp : String;
    Index : Integer;
    P : Pointer;
    DecodeData : TDecodeToSystem;
    FreeMemory : TFreeMemory;
    Buf : PChar;
    Size : Integer;
begin
    //fR[h

    Result := '';
    if Source = '' then Exit;
    temp := UpperCase(Language);
    Index := GetLangIndex(temp);

    if Index <> -1 then
    begin
        P := GetProcAddress(Langs[Index].Instance, 'DecodeToSystem');
        if P = nil then Exit;
        DecodeData := TSystemToEncode(P);
        Size := DecodeData(PChar(Source), Length(Source), Buf);

        if Size = 0 then
            Result := ''
        else
            Result := String(Buf);

        P := GetProcAddress(Langs[Index].Instance, 'FreeMemory');
        if P = nil then Exit;
        FreeMemory := TFreeMemory(P);
        FreeMemory(Buf, Size);
    end
    else
        Result := Source;
end;

function TLanguageDriver.BEncode(const Source, Charset : String) : String;
begin
    if (CharSet <> '') and (CharSet <> 'US-ASCII')
        and (Source <> '') and (IsIncludeNonAscii(Source)) then
    begin
        Result := '=?' + Charset + '?B?'
                + Base64Encode(EncodeString(Charset, Source))
                + '?=';
    end
    else
        Result := Source;
end;

function TLanguageDriver.BDecode(const Source : String) : String;
var
    temp : String;
    stemp : String;
    src : String;
    lang : String;
    ps : Integer;
    sps : Integer;
    n : Integer;
begin
    Result := '';

    temp := Trim(Source);
    n := 1;

    while n <= Length(temp) do
    begin
        if (Copy(temp, n, 2) = '=?') then
        begin
            stemp := Copy(temp, n + 2, Length(temp) - 2);
            ps := Pos('?B?', UpperCase(stemp));
            lang := Copy(stemp, 1, ps - 1);
            src := Copy(stemp, ps + 3, Length(stemp) - ps);
            sps := Pos('?=', src);

            if sps = 0 then
                Result := Result + Source[n]
            else
            begin
                stemp := Copy(src, 1, sps - 1);
                Result := Result + DecodeString(lang, Base64Decode(stemp));
                n := n + ps + sps + 4;
            end;
        end
        else
            Result := Result + Source[n];

        n := n + 1;
    end;
end;

function TLanguageDriver.HeaderBEncode(const Source, Charset : String) : String;
var
    temp : String;
    l , m , n : Integer;
begin
    temp := '';

    for n := 1 to Length(Source) do
    begin
        if Source[n]=';' then
            temp := temp + ','
        else
            temp := temp + Source[n];
    end;

    Result := '';
    if Source = '' then Exit;
    temp := Trim(temp);

    while temp<>'' do
    begin
        l := Pos(',' , temp );
        m := Pos('<' , temp );
        n := Pos('>' , temp );

        if l<>0 then
        begin
            if (m=0) or (n=0) then
            begin
                Result := Result + temp;
                temp := '';
            end
            else if l < m then
            begin
                Result := Result + Copy(temp , 1 , l );
                temp := Copy(temp , l + 1 , Length(temp) - l );
            end
            else
            begin
                Result := Result + BEncode(Copy(temp , 1 , m - 1 ) , Charset )
                    + Copy(temp , m , n - m + 1);
                temp := Copy(temp , n + 1 , Length(temp) - n );
            end;
        end
        else
        begin
            if (m=0) or (n=0) then
            begin
                Result := Result + temp;
                temp := '';
            end
            else
            begin
                Result := Result + BEncode(Copy(temp , 1 , m - 1 ) , Charset )
                            + Copy(temp , m , n - m + 1);
                temp := Copy(temp , n + 1 , Length(temp) - n );
            end;
        end;
    end;
end;

end.
