/*
 * @(#)ExecutionException.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.exception;


/**
 * 実行例外処理.
 *  
 * @version 1.0.0 2003/10/24
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class ExecutionException extends BaseException
{
    /**
     * 検知レベル : ノーマル.
     * <BR><BR>
     * 通常の検知レベルです.
     * (デフォルトでは、この情報となります.)
     */
    public static final int LEVEL_NORMAL =          0x00000000 ;
    
    /**
     * 検知レベル : ワーニング.
     * <BR><BR>
     * 注意を検知させるためのレベルです.
     */
    public static final int LEVEL_WARNING =         0x00000001 ;
    
    /**
     * 検知レベル : エラー通知.
     * <BR><BR>
     * エラーを検知させるためのレベルです.
     */
    public static final int LEVEL_ERROR =           0x00000002 ;
    
    /**
     * 検知レベル : 致命的エラー通知.
     * <BR><BR>
     * 致命的なエラーを検知させるためのレベルです.
     * 通常このレベルが通知された場合、OS,利用ミドルウェア等が
     * 何らかの障害により利用できない場合に通知されます.
     */
    public static final int LEVEL_FATAL =           0x00000003 ;
    
    /**
     * 検知レベル : プログラム不具合検知.
     * <BR><BR>
     * プログラム等が何らかの不具合によりエラーが発生した場合のレベルです.
     */
    public static final int LEVEL_DETECTION_BUG =   0x00000010 ;
    
    /**
     * 検知レベル : 設定不備.
     * <BR><BR>
     * オブジェクトの呼び出し順序等による設定不備により、処理を中断した場合.
     */
    public static final int LEVEL_NOT_SETTING =     0x00000011 ;
    
    /**
     * 検知レベル : デッドロック検知.
     * <BR><BR>
     * 指定タイマー値以上のロックを検出した場合.
     */
    public static final int LEVEL_DEAD_LOCK =       0x00000020 ;
    
    /**
     * 検知レベル : 動作停止.
     * <BR><BR>
     * 何らかの動作停止を通知するレベルです.
     * たとえば、動作対象のプロセス停止や、対象スレッドの停止等を検知した
     * 場合に通知されます.
     */
    public static final int LEVEL_STOP =            0x000000fe ;
    
    /**
     * 検知レベル : 不明(その他).
     * <BR><BR>
     * 何らかの理由で、処理が中断しました.
     */
    public static final int LEVEL_UNKNOWN =         0x000000ff ;
    
    /**
     * エラーレベル.
     */
    private volatile int m_level = ExecutionException.LEVEL_NORMAL ;
    
    /**
     * コンタストラクタ.
     */
    public ExecutionException()
    {
        super() ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     */
    public ExecutionException(Throwable e)
    {
        super(e) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     */
    public ExecutionException(BaseException e)
    {
        super(e) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時にメッセージを設定します.
     * <BR>
     * @param   message スローエラー生成時のメッセージを設定します。
     */
    public ExecutionException(String message)
    {
        super(message) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のエラーレベルを
     * 設定します.
     * <BR>
     * @param level エラー検知レベルを設定します.
     */
    public ExecutionException(int level)
    {
        super( "動作エラーが検知されました" ) ;
        this.setLevel( level ) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     * @param level エラー検知レベルを設定します.
     */
    public ExecutionException(Throwable e,int level)
    {
        super(e) ;
        this.setLevel( level ) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     * @param level エラー検知レベルを設定します.
     */
    public ExecutionException(BaseException e,int level)
    {
        super(e) ;
        this.setLevel( level ) ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時にメッセージを設定します.
     * <BR>
     * @param   message スローエラー生成時のメッセージを設定します。
     * @param level エラー検知レベルを設定します.
     */
    public ExecutionException(String message,int level)
    {
        super(message) ;
        this.setLevel( level ) ;
    }
    
    
    
    /**
     * 検知レベルセット.
     * <BR><BR>
     * 例外設定時の検知レベルを設定します.
     * <BR>
     * @param level 検知レベルを設定します.
     */
    public final void setLevel( int level )
    {
        switch( level ){
            case ExecutionException.LEVEL_NORMAL :
            case ExecutionException.LEVEL_WARNING :
            case ExecutionException.LEVEL_ERROR :
            case ExecutionException.LEVEL_FATAL :
            case ExecutionException.LEVEL_DETECTION_BUG :
            case ExecutionException.LEVEL_NOT_SETTING :
            case ExecutionException.LEVEL_DEAD_LOCK :
            case ExecutionException.LEVEL_STOP :
                break ;
            default :
                level = ExecutionException.LEVEL_UNKNOWN ;
        }
        
        m_level = level ;
    }
    
    /**
     * 検知レベルの取得.
     * <BR><BR>
     * 設定検知レベルを取得します.
     * <BR>
     * @return int 検知レベルが返されます.
     */
    public final int getLevel()
    {
        int ret ;
        
        ret = ( int )m_level ;
        
        return ret ;
    }
    
    /**
     * 検知レベルを文字情報で取得.
     * <BR><BR>
     * 現在設定されている検知レベルを文字情報で取得します.
     * <BR>
     * @return String 検知レベル情報が返されます.
     */
    public final String getLevelToString()
    {
        int level ;
        String ret = null ;
        
        level = ( int )m_level ;
        switch( level ){
            case ExecutionException.LEVEL_NORMAL :
                ret = "ノーマル検知" ;
                break ;
            case ExecutionException.LEVEL_WARNING :
                ret = "警告検知" ;
                break ;
            case ExecutionException.LEVEL_ERROR :
                ret = "エラー検知" ;
                break ;
            case ExecutionException.LEVEL_FATAL :
                ret = "致命的エラー検知" ;
                break ;
            case ExecutionException.LEVEL_DETECTION_BUG :
                ret = "プログラム不具合検知" ;
                break ;
            case ExecutionException.LEVEL_NOT_SETTING :
                ret = "必須設定値不足" ;
                break ;
            case ExecutionException.LEVEL_DEAD_LOCK :
                ret = "デッドロックタイムアウト検知" ;
                break ;
            case ExecutionException.LEVEL_STOP :
                ret = "ストップ検知" ;
                break ;
            case ExecutionException.LEVEL_UNKNOWN :
                ret = "その他例外" ;
                break ;
            default :
                ret = "不明" ;
                break ;
        }
        
        return ret ;
    }
    
}
