/** 
 * @(#)CatCSV.java.
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.io;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.convert.ConvertToIndexNo;
import org.maachang.commons.util.read.ReadCsv;
import org.maachang.commons.util.read.ReadElement;


/**
 * CSVファイルをカット.
 * <BR><BR>
 * CSVファイルをカットします.
 *
 * @version 1.00, 2004/05/23
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class CatCSV
{
    
    /**
     * １行データ解析先.
     */
    private final ReadElement m_oneLine = new ReadElement() ;
    
    /**
     * 改行単位で読み込むファイルオブジェクト.
     */
    private ReadEnterIndex m_index = null ;
    
    /**
     * CSV読み込み処理.
     */
    private ReadCsv m_csv = null ;
    
    /**
     * 現在の読み込み位置.
     */
    private int m_next = -1 ;
    
    /**
     * コンストラクタ.
     */
    private CatCSV(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルを指定して、読み込みます.
     * <BR>
     * @param name 読み込み対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public CatCSV( String name )
        throws InputException,FileAccessException
    {
        this( name,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルを指定して、読み込みます.
     * <BR>
     * @param name 読み込み対象のファイル名を指定します.
     * @param catCd 対象の区切りコードを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public CatCSV( String name,String catCd )
        throws InputException,FileAccessException
    {
        m_index = new ReadEnterIndex( true,name ) ;
        m_csv = new ReadCsv( m_index,catCd ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルを指定して、読み込みます.
     * <BR>
     * @param name 読み込み対象のファイル名を指定します.
     * @param catCd 対象の区切りコードを設定します.
     * @param charset 読み込み対象のキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public CatCSV( String name,String catCd,String charset )
        throws InputException,FileAccessException
    {
        m_index = new ReadEnterIndex( true,name,charset ) ;
        m_csv = new ReadCsv( m_index,catCd ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクトクローズ.
     * <BR><BR>
     * オブジェクトをクローズします.
     */
    public final void close()
    {
        if( m_csv != null ){
            m_csv.clear() ;
        }
        m_csv = null ;
        
        if( m_index != null ){
            m_index.close() ;
        }
        m_index = null ;
        
        m_oneLine.clear() ;
        m_next = -1 ;
    }
    
    /**
     * 次行情報を読み込みます.
     * <BR><BR>
     * 現在より、１つ先の行位置の情報を読み込みます.
     * <BR>
     * @return boolean 読み込み結果が返されます.<BR>
     *                 [true]が返された場合、読み込みに成功しました.<BR>
     *                 [false]が返された場合、読み込みに失敗しました.
     */
    public final boolean next()
    {
        boolean ret ;
        
        try{
            m_next ++ ;
            m_oneLine.clear() ;
            m_csv.getIndex( m_next,m_oneLine ) ;
            ret = true ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 指定行の情報を読み込む.
     * <BR><BR>
     * 対象の指定情報を読み込みます.
     * <BR>
     * @param no 読み込み先の項番を指定します.
     * @return boolean 読み込み結果が返されます.<BR>
     *                 [true]が返された場合、読み込みに成功しました.<BR>
     *                 [false]が返された場合、読み込みに失敗しました.
     */
    public final boolean setLine( int no )
    {
        boolean ret ;
        
        try{
            m_next = no ;
            m_oneLine.clear() ;
            m_csv.getIndex( m_next,m_oneLine ) ;
            ret = true ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 反映先のオブジェクトを取得.
     * <BR><BR>
     * 反映先のオブジェクト情報を取得します.
     * <BR>
     * @return ConvertToIndexNo 読み込みデータ反映先のオブジェクトが
     *                          返されます.
     */
    public final ConvertToIndexNo getResult()
    {
        return m_oneLine ;
    }
    
    /**
     * 現在の読み込み位置を取得.
     * <BR><BR>
     * 現在読み込まれている位置を取得します.
     * <BR>
     * @return int 現在読み込まれている位置が返されます.
     */
    public final int getReadLine()
    {
        return m_next ;
    }
    
    /**
     * 読み込みできる、行数を取得.
     * <BR><BR>
     * 読み込むことができる、行数を取得します.
     * <BR>
     * @return int 読み込み可能な行数が返されます.
     */
    public final int getMaxLine()
    {
        int ret ;
        
        try{
            ret = m_index.size() ;
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在、オブジェクトが有効かチェック.
     * <BR><BR>
     * 現在、オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public final boolean isUse()
    {
        return ( m_csv == null ) ? false : true ;
    }

}
