/*
 * @(#)ReadEnterIndex.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.io;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.ReadIndex;


/**
 * 改行単位でインデックスを設定して、情報取得.
 * <BR><BR>
 * 改行単位でインデックスを設定して情報を取得します.
 * <BR>
 *
 * @version 1.00, 2003/12/20
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class ReadEnterIndex implements ReadIndex
{
    /**
     * 1度に取得するファイル情報数.
     */
    private static final int READ_FILE_LENGTH = 1024 ;
    
    /**
     * 改行タイプ : Windows系.
     */
    private static final int TYPE_ENTER_WINDOWS = 1 ;
    
    /**
     * 改行タイプ : Unix系.
     */
    private static final int TYPE_ENTER_UNIX = 2 ;
    
    /**
     * 改行タイプ : Macintosh系.
     */
    private static final int TYPE_ENTER_MACINTOSH = 3 ;
    
    /**
     * 改行タイプ : なし.
     */
    private static final int TYPE_ENTER_NOT = -1 ;
    
    
    /**
     * 改行に対するインデックス.
     */
    private long[] m_index = null ;
    
    /**
     * インデックスに対する１データ長.
     */
    private int[] m_indexLength = null ;
    
    /**
     * オープンファイル名.
     */
    private String m_name = null ;
    
    /**
     * 読み込み対象キャラクタセット.
     */
    private String m_charset = null ;
    
    /**
     * ファイルモード.
     */
    private boolean m_mode = false ;
    
    /**
     * データ取得タイプ.
     * [false]バイナリ.
     * [true]文字列.
     */
    private boolean m_getType = true ;
    
    /**
     * ランダムIO.
     */
    private RandomIO m_fp = null ;
    
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ReadEnterIndex()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ReadEnterIndex( String name )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( false,true,name,BaseDef.THIS_CHARSET ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param name オープン対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ReadEnterIndex( boolean mode,String name )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( mode,true,name,BaseDef.THIS_CHARSET ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ReadEnterIndex( String name,String charset )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( false,true,name,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ReadEnterIndex( boolean mode,String name,String charset )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( mode,true,name,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param getType メソッド[getIndex]の戻り値の条件を指定します.<BR>
     *                [true]を設定した場合、Stringオブジェクトで返されます.<BR>
     *                [false]を設定した場合、byte[]配列で返されます.<BR>
     *                また、[false]を設定した場合、引数[charset]は意味を持ちません.
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ReadEnterIndex( boolean mode,boolean getType,String name,String charset )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( mode,getType,name,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 対象のファイルをオープン.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void open( String name )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( false,true,name,BaseDef.THIS_CHARSET ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 対象のファイルをオープン.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param name オープン対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void open( boolean mode,String name )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( mode,true,name,BaseDef.THIS_CHARSET ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 対象のファイルをオープン.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void open( String name,String charset )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( false,true,name,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 対象のファイルをオープン.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void open( boolean mode,String name,String charset )
        throws InputException,FileAccessException
    {
        
        try{
            this.open( mode,true,name,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 対象のファイルをオープン.
     * <BR><BR>
     * 対象のファイルをオープンして、対象のインデックスを設定します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を指定した場合、オープン時のファイルを引き続き利用します.<BR>
     *             [false]を指定した場合、オープン時のファイルを一旦破棄します.<BR>
     *             基本的に[true]を設定した方がI/O処理が早くなります.
     * @param getType メソッド[getIndex]の戻り値の条件を指定します.<BR>
     *                [true]を設定した場合、Stringオブジェクトで返されます.<BR>
     *                [false]を設定した場合、byte[]配列で返されます.<BR>
     *                また、[false]を設定した場合、引数[charset]は意味を持ちません.
     * @param name オープン対象のファイル名を設定します.
     * @param charset 読み込み対象のキャラクタセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void open( boolean mode,boolean getType,String name,String charset )
        throws InputException,FileAccessException
    {
        
        if( name == null || charset == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.close() ;
        
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                
                try{
                    
                    m_fp = new RandomIO( name,false ) ;
                    
                    this.readIndex( m_fp,name,charset ) ;
                    
                    if( mode == false ){
                        m_fp.close() ;
                        m_fp = null ;
                    }
                    
                    m_mode = mode ;
                    m_getType = getType ;
                    
                }catch( FileAccessException fa ){
                    throw fa ;
                }
                
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * オープン対象のファイルをクローズ.
     * <BR><BR>
     * オープン対象のファイルをクローズします.
     */
    public final void close()
    {
        m_sync.clear() ;
        
        try{
            m_fp.close() ;
        }catch( Exception t ){
        }
        
        m_index = null ;
        m_indexLength = null ;
        m_name = null ;
        m_charset = null ;
        m_fp = null ;
        m_mode = false ;
        m_getType = true ;
        
    }
    
    /**
     * インデックス情報の取得.
     * <BR><BR>
     * インデックス情報を取得します.
     * <BR>
     * @param no 取得対象のインデックス位置を設定します.
     * @return Object インデックス位置に対する情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final Object getIndex( int no )
        throws InputException,AccessException
    {
        Object ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( no < 0 || no >= m_index.length ){
                    throw new InputException( "引数は不正です" ) ;
                }
                
                try{
                    
                    ret = this.getIndexByData(
                        m_mode,m_getType,m_index[ no ],m_indexLength[ no ]
                    ) ;
                    
                }catch( IndexOutOfBoundsException io ){
                    throw new AccessException( io ) ;
                }catch( FileAccessException fa ){
                    throw fa ;
                }
                
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 管理インデックス数の取得.
     * <BR><BR>
     * 管理しているインデックス数を取得します.
     * <BR>
     * @return int 管理インデックス数が返されます.<BR>
     *             情報が存在しない場合[ReadIndex.NOT_INDEX]が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_index.length ;
            }
        }catch( NullPointerException nul ){
            ret = ReadIndex.NOT_INDEX ;
        }
        
        return ret ;
    }
    
    /**
     * インデックス利用可能状態チェック.
     * <BR><BR>
     * インデックス情報が利用可能かチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、インデックスは利用可能です.
     *                 [false]が返された場合、インデックスは利用不可能です.
     */
    public final boolean isIndex()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                ret = ( m_name == null ) ? false : true ;
                
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 対象ファイルに対するインデックス情報および
     * インデックスに対する情報長を取得.
     */
    private final void readIndex( RandomIO fp,String name,String charset )
        throws FileAccessException
    {
        int i,k ;
        int len ;
        int fixEnter ;
        int enterCnt ;
        int enterType ;
        int entLength ;
        
        int winEnterLen ;
        int unixEnterLen ;
        int macEnterLen ;
        
        long j ;
        long seekFirst ;
        long fileCount ;
        
        boolean flg ;
        
        InputStream input = null ;
        ArrayList list = null ;
        
        byte[] code = null ;
        
        byte[] winEnter = null ;
        byte[] unixEnter = null ;
        byte[] macEnter = null ;
        
        long[] index = null ;
        int[] indexLen = null ;
        
        list = new ArrayList() ;
        
        code = new byte[ ReadEnterIndex.READ_FILE_LENGTH ] ;
        
        
        try{
            
            winEnter = BaseDef.ENTER_WINDOWS.getBytes( charset ) ;
            unixEnter = BaseDef.ENTER_UNIX.getBytes( charset ) ;
            macEnter = BaseDef.ENTER_MACINTOSH.getBytes( charset ) ;
            
            winEnterLen = winEnter.length ;
            unixEnterLen = unixEnter.length ;
            macEnterLen = macEnter.length ;
            
            input = new BufferedInputStream( fp.getInputStream( 0L ) ) ;
            
            list.add( new Long( 0 ) ) ;
            
            enterType = ReadEnterIndex.TYPE_ENTER_NOT ;
            
            // 改行コード単位で情報取得.
            flg = false ;
            fileCount = 0 ;
            entLength = 0 ;
            
            // 改行コードチェック.
            for( seekFirst = 0,fixEnter = 0,enterCnt = 0 ;; ){
                
                // ファイルからデータを読み込む.
                if( ( len = input.read( code ) ) == -1 ){
                    break ;
                }
                
                // 読み込んだ内容でチェック.
                for( i = 0,j = seekFirst ; i < len ; i ++,j ++ ){
                    
                    flg = false ;
                    fileCount ++ ;
                    
                    // MACで認識していたが、ウィンドウ改行条件である場合.
                    if(
                        enterType == ReadEnterIndex.TYPE_ENTER_MACINTOSH &&
                        code[ i ] == winEnter[ macEnterLen ]
                    )
                    {
                        // Windows改行条件でセット.
                        fixEnter = winEnterLen ;
                        entLength = winEnterLen ;
                        enterType = ReadEnterIndex.TYPE_ENTER_WINDOWS ;
                        flg = true ;
                        
                        continue ;
                        
                    }
                    // 改行条件が検知されている場合.
                    else if( enterType != ReadEnterIndex.TYPE_ENTER_NOT ){
                        
                        list.add( new Long( (j-fixEnter) ) ) ;
                        list.add( new Long( j ) ) ;
                        
                        enterCnt = 0 ;
                        fixEnter = 0 ;
                        enterType = ReadEnterIndex.TYPE_ENTER_NOT ;
                        flg = true ;
                        
                    }
                    
                    // 指定コードがMac(MacOs9まで)[￥ｒ]の場合.
                    if( code[ i ] == macEnter[ enterCnt ] ) {
                        
                        enterCnt += 1 ;
                        
                        // mac改行コードが終了する場合.
                        if( macEnterLen == enterCnt ){
                            
                            fixEnter = macEnterLen ;
                            entLength = macEnterLen ;
                            enterType = ReadEnterIndex.TYPE_ENTER_MACINTOSH ;
                            flg = true ;
                            
                        }
                        
                        continue ;
                        
                    }
                    
                    // 指定コードがUnix(Linux)[￥ｎ]の場合.
                    if( code[ i ] == unixEnter[ enterCnt ] ){
                        
                        enterCnt += 1 ;
                        
                        // unix改行コードが終了する場合.
                        if( unixEnterLen == enterCnt ){
                            
                            fixEnter = unixEnterLen ;
                            entLength = unixEnterLen ;
                            enterType = ReadEnterIndex.TYPE_ENTER_UNIX ;
                            flg = true ;
                            
                        }
                        
                        continue ;
                        
                    }
                    
                    // 指定コードがWindows系[￥ｒ￥ｎ]の場合.
                    if( code[ i ] == winEnter[ enterCnt ] ){
                        
                        enterCnt += 1 ;
                        
                        // windows改行コードが終了する場合.
                        if( winEnterLen == enterCnt ){
                            
                            fixEnter = winEnterLen ;
                            entLength = winEnterLen ;
                            enterType = ReadEnterIndex.TYPE_ENTER_WINDOWS ;
                            flg = true ;
                            
                        }
                        
                        continue ;
                    }
                    
                    enterCnt = 0 ;
                    fixEnter = 0 ;
                    enterType = ReadEnterIndex.TYPE_ENTER_NOT ;
                    
                }
                
                seekFirst += len ;
                
            }
            
            // 取得されたインデックス情報を整える.
            try{
                input.close() ;
            }catch( Exception t1 ){
            }
            
            if( flg == true ){
                list.add( new Long( (fileCount-entLength) ) ) ;
            }else{
                list.add( new Long( fileCount ) ) ;
            }
            
            // 取得されたインデックス情報をセット.
            len = list.size() / 2 ;
            index = new long[ len ] ;
            indexLen = new int[ len ] ;
            
            for( i = 0,k = 0 ; i < len ; i ++, k += 2 ){
                
                index[ i ] = ( ( Long )list.get( k ) ).longValue() ;
                indexLen[ i ] = ( int )(
                    ( ( Long )list.get( k+1 ) ).longValue() - index[ i ]
                ) ;
                
            }
            
            m_index = index ;
            m_indexLength = indexLen ;
            m_name = name ;
            m_charset = charset ;
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
            throw new FileAccessException( t ) ;
        }finally{
            
            try{
                input.close() ;
            }catch( Exception t1 ){
            }
            
            try{
                list.clear() ;
            }catch( Exception t2 ){
            }
            
            input = null ;
            list = null ;
            
            code = null ;
            
            winEnter = null ;
            unixEnter = null ;
            macEnter = null ;
            
            index = null ;
            indexLen = null ;
            
        }
    }
    
    /**
     * 指定インデックス位置の内容を取得.
     */
    private final Object getIndexByData( boolean mode,boolean getType,long addr,int length )
        throws FileAccessException
    {
        
        Object ret = null ;
        byte[] binary = null ;
        RandomIO fp = null ;
        
        if( length <= 0L ){
            if( getType == true ){
                ret = "" ;
            }
            else{
                ret = new byte[ 0 ] ;
            }
        }
        else{
            
            try{
                
                if( mode == false ){
                    fp = new RandomIO( m_name,false ) ;
                }
                else{
                    fp = m_fp ;
                }
                
                binary = new byte[ length ] ;
                
                fp.read( binary,addr ) ;
                if( getType == true ){
                    ret = new String( binary,m_charset ) ;
                }
                else{
                    ret = binary ;
                }
                
            }catch( Exception t ){
                throw new FileAccessException( t ) ;
            }finally{
                if( mode == false ){
                    if( fp != null ){
                        fp.close() ;
                    }
                }
                fp = null ;
                binary = null ;
            }
            
        }
        
        return ret ;
        
    }
}


