/*
 * @(#)BaseMultiCast.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;


/**
 * マルチキャストインターフェイス.
 * <BR><BR>
 * マルチキャストUDPプロトコルインターフェイスです.
 *
 * @version 1.00, 2005/02/07
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public interface BaseMultiCast
{
    /**
     * 受信バッファ長.
     * <BR><BR>
     * 受信時のバッファ長です.
     */
    public static final int BUFFER = 65535 ;
    
    /**
     * 受信デフォルトバッファ長.
     * <BR><BR>
     * 受信時のデフォルトバッファ長です.
     */
    public static final int DEF_BUFFER = 65535 ;
    
    /**
     * デフォルトポート番号.
     * <BR><BR>
     * デフォルトのポート番号です.
     */
    public static final int DEF_PORT = 5500 ;
    
    /**
     * デフォルトTTL(TimeToLive).
     * <BR><BR>
     * デフォルトのTTL値を設定します.
     */
    public static final byte DEF_TTL = 1 ;
    
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @exception NotBindException バインド失敗.
     */
    public void open() throws NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param buf 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void open( int buf )
        throws InputException,NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号を指定して、オープンします.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param buf 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void open( int port,int buf )
        throws InputException,NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号を指定して、オープンします.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param buf 送受信バッファ長を設定します.
     * @param defTTL デフォルトのTTL(TimeToLive)を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void open( int port,int buf,byte defTTL )
        throws InputException,NotBindException ;
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を行います.
     */
    public void close() ;
    
    /**
     * 利用インターフェイスを設定.
     * <BR><BR>
     * マルチキャストを行う利用インターフェイスを設定します.<BR>
     * また、ここで言うインターフェイスは接続先のネットワークを意味する
     * ものであり、例えばLANが２つあった時の利用先を指定する場合に
     * 利用します.
     * <BR>
     * @param inf 対象のインターフェイス先を設定します.
     * @exception InputException 入力例外.
     * @exception NotInterfaceException インターフェイス非存在例外.
     */
    public void setInterface( InetAddress inf )
        throws InputException,NotInterfaceException ;
    
    /**
     * 現在参加しているグループから離脱.
     * <BR><BR>
     * 現在参加しているグループから離脱します.
     */
    public void leaveGroup() ;
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] binary )
        throws InputException,UndefineBindException ;
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param ttl マルチキャストパケット有効期間を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] binary,byte ttl )
        throws InputException,UndefineBindException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public byte[] receive( ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public byte[] receive( ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public int receive( byte[] out,ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public int receive( byte[] out,ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * 有効期限を設定.
     * <BR><BR>
     * 有効期限を設定します.
     * <BR>
     * @return ttl 現在参加しているグループに対する有効期限を設定します.
     * @exception InputException 入力例外.
     */
    public void setTTL( int ttl ) throws InputException ;
    
    /**
     * 利用インターフェイスを取得.
     * <BR><BR>
     * マルチキャストを行う利用インターフェイスを取得します.
     * <BR>
     * @return InetAddress 対象のインターフェイス先が返されます.
     */
    public InetAddress getInterface() ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public void getLocal( ConnectAddress addr ) ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @return ConnectAddress 対象のローカルアドレスが返されます.
     */
    public ConnectAddress getLocal() ;
    
    /**
     * ローカルアドレス情報を取得.
     * <BR><BR>
     * ローカルアドレス情報を取得します.
     * <BR>
     * @return InetAddress ローカルアドレス情報が返されます.
     */
    public InetAddress getLocalAddress() ;
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public int getLocalPort() ;
    
    /**
     * 設定バッファ長を取得.
     * <BR><BR>
     * 設定されているバッファ長を取得します.
     * <BR>
     * @return int 設定バッファ長が返されます.
     */
    public int getBuffer() ;
    
    /**
     * 設定されている有効期限を取得.
     * <BR><BR>
     * 設定されている有効期限が返されます.
     * <BR>
     * @return int 現在参加しているグループに対する有効期限が返されます.
     */
    public int getTTL() ;
    
    /**
     * グループ参加チェック.
     * <BR><BR>
     * 現在グループに参加しているかチェックします.
     * <BR>
     * @return boolean グループ参加状態が返されます.<BR>
     *                 [true]が返された場合グループに参加しています.<BR>
     *                 [false]が返された場合グループに参加していません.
     */
    public boolean isGroup() ;
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープン状態が返されます.<BR>
     *                 [true]が返された場合、ソケットはオープンされています.<BR>
     *                 [false]が返された場合、ソケットはオープンされていません.
     */
    public boolean isOpen() ;
    
}

