/*
 * @(#)NetAgentThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;


/**
 * ネットワークエージェントスレッド処理.
 * <BR><BR>
 * ネットワークの切断/復帰を管理するエージェント用スレッドです.
 *  
 * @version 1.0.0 2003/12/21
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class NetAgentThread extends ExecutionThread
{
    
    /**
     * 最小インターバル値.
     */
    private static final int MIN_INTERVAL = 3500 ;
    
    /**
     * インターバル値.
     */
    private int m_interval = 0 ;
    
    /**
     * 前回実施タイミング.
     */
    private long m_beforeTime = 0L ;
    
    /**
     * 切断/復帰対象要素群.
     */
    private final ObjectArray m_man = new ObjectArray() ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public NetAgentThread()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public final void create()
        throws AccessException
    {
        this.create( 0 ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param interval チェック間隔を設定します.<BR>
     *                 設定可能な最小値は[1000]です.<BR>
     *                 それ以下の値を設定した場合は、最小値となります.
     * @exception AccessException アクセス例外.
     */
    public final void create( int interval )
        throws AccessException
    {
        this.clear() ;
        m_sync.create() ;
        
        try{
            
            NetConfig.getInstance().flush() ;
            
            synchronized( m_sync.get() ){
                
                m_interval = ( interval < NetAgentThread.MIN_INTERVAL ) ?
                    NetAgentThread.MIN_INTERVAL : interval ;
                m_thread.create( this,null ) ;
                m_thread.startThread() ;
                
            }
            
        }catch( BaseException be ){
            this.clear() ;
            throw new AccessException( be ) ;
        }catch( Exception t ){
            this.clear() ;
            throw new AccessException( t ) ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        try{
            synchronized( m_sync.get() ){
                
                m_thread.clear() ;
                this.clearElements() ;
                m_man.clear() ;
                m_interval = 0 ;
                m_beforeTime = 0L ;
                
            }
        }catch( Exception t ){
            
            m_thread.clear() ;
            this.clearElements() ;
            m_man.clear() ;
            m_interval = 0 ;
            m_beforeTime = 0L ;
            
        }finally{
            m_sync.clear() ;
        }
        
    }
    
    /**
     * 管理内容を破棄.
     * <BR><BR>
     * 管理されている内容を破棄します.
     */
    public final void destroyElements()
    {
        try{
            synchronized( m_sync.get() ){
                this.clearElements() ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * ネットワークエージェント管理要素を追加.
     * <BR><BR>
     * ネットワークエージェント管理要素を追加します.
     * <BR>
     * @param value 対象のネットワークエージェント管理要素を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void add( NetAgentElement value )
        throws InputException,AccessException
    {
        String addr = null ;
        String adapter = null ;
        
        if(
            value == null ||
            value.isUse() == false ||
            ( addr = value.getLocalAddress() ) == null
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            // 対象アドレスがlocalhostの場合.
            if( addr.equals( NetDef.LOCAL_HOST ) == true ){
                
                ///////////////////////////////
                // エージェント監視下にしない.
                ///////////////////////////////
                
                // ネットワークインターフェイスがサポートされている場合.
                if( NetConfig.getInstance().useObject() == true ){
                    
                    // 対象アダプタ名を設定.
                    value.setAdapter(
                        NetConfig.getInstance().getAddressByAdapter( addr )
                    ) ;
                    
                }
                
                // 対象エレメントを実行.
                value.restore( addr ) ;
                
            }
            // 対象アドレスがlocalhostで無い場合.
            else{
                
                /////////////////////////////
                // エージェント監視下にする.
                /////////////////////////////
                
                synchronized( m_sync.get() ){
                    
                    // ネットワークアドレスが割り当てられていない場合.
                    if( addr.equals( NetDef.NOT_ADDR ) == true ){
                        m_man.add( value ) ;
                    }
                    else{
                        
                        // ネットワークインターフェイスがサポートされている場合.
                        if( NetConfig.getInstance().useObject() == true ){
                            
                            // 対象アダプタ名を取得.
                            adapter = NetConfig.getInstance().getAddressByAdapter( addr ) ;
                            
                            // 対象IPアドレスに対するアダプタ名が取得された場合.
                            if( adapter != null ){
                                
                                value.setAdapter( adapter ) ;
                                value.restore( addr ) ;
                                
                            }
                            
                            m_man.add( value ) ;
                            
                        }
                        // ネットワークインターフェイスがサポートされていない場合.
                        else{
                            
                            value.restore( addr ) ;
                            m_man.add( value ) ;
                            
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( InputException in ){
            throw new AccessException( in ) ;
        }catch( Exception t ){
        }finally{
            addr = null ;
        }
        
    }
    
    /**
     * 実施インターバル値を設定.
     * <BR><BR>
     * 実施インターバル値を設定します.
     * <BR>
     * @param interval チェック間隔を設定します.<BR>
     *                 設定可能な最小値は[1000]です.<BR>
     *                 それ以下の値を設定した場合は、最小値となります.
     */
    public final void setInterval( int interval )
    {
        try{
            
            synchronized( m_sync.get() ){
                
                m_interval = ( interval < NetAgentThread.MIN_INTERVAL ) ?
                    NetAgentThread.MIN_INTERVAL : interval ;
            }
            
        }catch( Exception t ){
        }
    }
    
    /**
     * 実施インターバル値を取得.
     * <BR><BR>
     * 実施インターバル値を取得します.
     * <BR>
     * @return int 実施インターバル値が返されます.
     */
    public final int getInterval()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_interval ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * ネットワークエージェント管理要素数を取得.
     * <BR><BR>
     * 対象のネットワークエージェント管理要素数を取得します.
     * <BR>
     * @return int 管理されているネットワークエージェント管理要素数が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_man.size() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施チェック.
     * <BR><BR>
     * このオブジェクトのスレッドが動作しているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、スレッドは動作中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド処理 : 初期処理.
     */
    protected final void init( Object obj )throws ExecutionException{}
    
    /**
     * スレッド処理 : 終了処理.
     */
    protected final void exit( Object obj )throws ExecutionException{}
    
    /**
     * スレッド処理 : 停止時処理.
     */
    protected final void stop( Object obj )throws ExecutionException{}
    
    /**
     * スレッド処理 : 実行処理.
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        long beforeTime ;
        
        try{
            
            UtilCom.idleTime() ;
            
            synchronized( m_sync.get() ){
                beforeTime = m_beforeTime + ( long )( m_interval & 0x00000000ffffffff ) ;
            }
            
            // 実行時間
            if( beforeTime <= System.currentTimeMillis() ){
                
                // ネットワークインターフェイスがサポートされている場合.
                if( NetConfig.getInstance().useObject() == true ){
                    
                    this.useNetInterface() ;
                    
                }
                // ネットワークインターフェイスがサポートされていない場合.
                else{
                    
                    this.notNetInterface() ;
                    
                }
                
                // 処理終了後更新.
                synchronized( m_sync.get() ){
                    m_beforeTime = System.currentTimeMillis() ;
                }
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }
    }
    
    /**
     * 管理要素クリア.
     */
    private final void clearElements()
    {
        int i ;
        int len ;
        
        ObjectArray ary = null ;
        NetAgentElement pause = null ;
        
        ary = m_man ;
        
        try{
            
            len = ary.size() ;
            for( i = 0 ; i < len ; i ++ ){
                
                pause = ( NetAgentElement )ary.get( i ) ;
                pause.down() ;
                pause.exitAgent() ;
                
            }
            
        }catch( Exception t ){
        }finally{
            if( ary != null ){
                ary.clear() ;
            }
            
            ary = null ;
        }
    }
    
    /**
     * ネットワークインターフェイス対応版処理.
     */
    private final void useNetInterface()
        throws ExecutionException
    {
        int i ;
        int len ;
        
        String addr = null ;
        String adapterName = null ;
        ObjectArray ary = null ;
        NetAgentElement element = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ary = m_man ;
            }
            
            len = ary.size() ;
            NetConfig.getInstance().flush() ;
            
            // 管理テーブル分処理.
            for( i = len-1 ; i >= 0 ; i -- ){
                
                try{
                    
                    element = ( NetAgentElement )ary.get( i ) ;
                    
                    // 対象アドレスを取得.
                    addr = element.getLocalAddress() ;
                    
                    // 対象アダプタ名を取得.
                    adapterName = element.getAdapter() ;
                    
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがクローズしている場合.
                    // 対象のアダプタ名が存在しない場合.
                    ////////////////////////////////////////////////////////////
                    if( element.isUse() == false ){
                        
                        // 情報をクリア.
                        ary.remove( i ) ;
                        element.down() ;
                        element.exitAgent() ;
                        
                    }
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがダウンしている場合.
                    ////////////////////////////////////////////////////////////
                    else if( element.isDown() == true ){
                        
                        // 対象アダプタ名が存在しない場合.
                        if( adapterName == null ){
                            
                            // ローカルアドレスとなる情報を取得.
                            if( ( addr = NetAgentThread.getUseNetInterface() ) != null ){
                                
                                // アドレス管理条件を更新.
                                NowAddress.getInstance().flush() ;
                                
                                // 対象アダプタ名を取得.
                                adapterName = NetConfig.getInstance().getAddressByAdapter( addr ) ;
                                
                                // 対象IPアドレスに対するアダプタ名が取得された場合.
                                if( adapterName != null ){
                                    
                                    element.setAdapter( adapterName ) ;
                                    
                                    // 対象アドレスが更新されていない場合.
                                    if( NowAddress.getInstance().getLocalAddress().equals( NetDef.NOT_ADDR ) == true ){
                                        
                                        UtilCom.idleTime() ;
                                        continue ;
                                        
                                    }
                                    
                                    element.restore( addr ) ;
                                    
                                }
                                // 対象IPアドレスに対するアダプタ名が取得されない場合.
                                else{
                                    
                                    // 情報をクリア.
                                    element.down() ;
                                    element.exitAgent() ;
                                    
                                }
                                
                            }
                            
                        }
                        // 対象アダプタ名に対する要素条件が存在する場合.
                        else if( NetConfig.getInstance().isUseAdapter( adapterName ) == true ){
                            
                            // アドレス管理条件を更新.
                            NowAddress.getInstance().flush() ;
                            
                            // 対象アドレスを取得.
                            addr = NetConfig.getInstance().getAdapterByAddress( adapterName,0 ) ;
                            
                            // 対象アドレスが更新されていない場合.
                            if( NowAddress.getInstance().getLocalAddress().equals( NetDef.NOT_ADDR ) == true ){
                                
                                UtilCom.idleTime() ;
                                continue ;
                                
                            }
                            
                            // 対象エレメントを復帰.
                            element.restore( addr ) ;
                            
                        }
                        
                    }
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがダウンしていない場合.
                    // 対象アダプタ名に対する要素条件が存在しない場合.
                    ////////////////////////////////////////////////////////////
                    else if(
                        element.isDown() == false &&
                        NetConfig.getInstance().isUseAddress( addr ) == false
                    )
                    {
                        
                        // アドレス管理条件を更新.
                        NowAddress.getInstance().flush() ;
                        
                        // 対象エレメントをダウン.
                        element.down() ;
                        
                    }
                    
                }catch( Exception tt ){
                    
                    try{
                        
                        // 情報をクリア.
                        element.down() ;
                        element.exitAgent() ;
                        
                    }catch( Exception ttt ){
                    }
                    
                }finally{
                    addr = null ;
                    element = null ;
                    addr = null ;
                    adapterName = null ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }finally{
            addr = null ;
            adapterName = null ;
            ary = null ;
            element = null ;
        }
        
    }
    
    /**
     * ネットワークインターフェイス非対応版処理.
     */
    private final void notNetInterface()
        throws ExecutionException
    {
        int i ;
        int len ;
        
        String addr = null ;
        ObjectArray ary = null ;
        NetAgentElement element = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ary = m_man ;
            }
            
            len = ary.size() ;
            
            // 管理テーブル分処理.
            for( i = len-1 ; i >= 0 ; i -- ){
                
                element = ( NetAgentElement )ary.get( i ) ;
                
                // 対象アドレスを取得.
                addr = element.getLocalAddress() ;
                
                try{
                    
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがクローズしている場合.
                    // 対象のアダプタ名が存在しない場合.
                    ////////////////////////////////////////////////////////////
                    if( element.isUse() == false ){
                        
                        // 情報をクリア.
                        ary.remove( i ) ;
                        element.down() ;
                        element.exitAgent() ;
                        
                    }
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがダウンしている場合.
                    // 対象アダプタ名に対する要素条件が存在する場合.
                    ////////////////////////////////////////////////////////////
                    else if( element.isDown() == true ){
                        
                        // 現在のアドレスが指定されていない場合.
                        if( addr.equals( NetDef.NOT_ADDR ) == true ){
                            
                            // 現在のネットワークローカルアドレスを取得.
                            addr = NowAddress.getInstance().getLocalAddress() ;
                            
                            // 取得されたローカルアドレスが有効である場合.
                            if( addr.equals( NetDef.NOT_ADDR ) == false ){
                                element.restore( addr ) ;
                            }
                            
                        }
                        // 対象アドレスが存在する場合.
                        else if( NowAddress.getInstance().isUse( addr ) == true ){
                            
                            // アドレス管理条件を更新.
                            NowAddress.getInstance().flush() ;
                            
                            // 対象エレメントを復帰.
                            element.restore( addr ) ;
                            
                        }
                        
                    }
                    ////////////////////////////////////////////////////////////
                    // 対象エレメントがダウンしていない場合.
                    // 対象アダプタ名に対する要素条件が存在しない場合.
                    ////////////////////////////////////////////////////////////
                    else if(
                        element.isDown() == false &&
                        NowAddress.getInstance().isUse( addr ) == false
                    )
                    {
                        
                        // アドレス管理条件を更新.
                        NowAddress.getInstance().flush() ;
                        
                        // 対象エレメントをダウン.
                        element.down() ;
                        
                    }
                    
                }catch( Exception tt ){
                    
                    try{
                        
                        // 情報をクリア.
                        element.down() ;
                        element.exitAgent() ;
                        
                    }catch( Exception ttt ){
                    }
                    
                }finally{
                    element = null ;
                    addr = null ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }finally{
            addr = null ;
            ary = null ;
            element = null ;
        }
        
    }
    
    /**
     * ネットワークインターフェイス管理から、有効IPアドレスを取得.
     */
    private static final String getUseNetInterface()
    {
        int i ;
        int len ;
        
        String[] addrs = null ;
        String ret = null ;
        
        if( ( addrs = NetConfig.getInstance().getAddress() ) != null ){
            
            len = addrs.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                if(
                    addrs[ i ].equals( NetDef.NOT_ADDR ) == false &&
                    addrs[ i ].equals( NetDef.LOCAL_HOST ) == false
                )
                {
                    ret = addrs[ i ] ;
                    break ;
                }
                
            }
        }
        
        addrs = null ;
        return ret ;
    }
    
    
}

