/*
 * @(#)NetConfig.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.UnsupportException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.CharTable;


/**
 * ネットワークインターフェイス取得.
 * <BR><BR>
 * ネットワークインターフェイスを取得します.
 *  
 * @version 1.0.0 2004/10/13
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class NetConfig
{
    
    /**
     * アドレス情報格納.
     */
    class NetPause{
        String[] ipAddr = null ;
        String subNet = null ;
        String macAddr = null ;
    } ;
    
    /**
     * 非アドレス.
     */
    private static final String NOT_ADDR = NetDef.LOCAL_HOST ;
    
    
    
    /**
     * アドレス格納条件.
     */
    private final CharTable m_addr = new CharTable() ;
    
    /**
     * 媒体格納条件.
     */
    private final CharTable m_adapter = new CharTable() ;
    
    /**
     * ネットワークアダプタオブジェクト.
     */
    private NetAdapter m_adapterObject = null ;
    
    /**
     * 利用可能フラグ.
     */
    private boolean m_useFlg = false ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * シングルトン.
     */
    private static final NetConfig SNGL = new NetConfig() ;
    
    
    /**
     * コンストラクタ.
     */
    private NetConfig()
    {
        m_sync.create() ;
        this.flush() ;
    }
    
    /**
     * オブジェクト情報を取得.
     * <BR><BR>
     * オブジェクト情報を取得します.
     * <BR>
     * @return NetConfig オブジェクト情報が返されます.
     */
    public static final NetConfig getInstance()
    {
        return SNGL ;
    }
    
    /**
     * オブジェクト利用可能フラグ.
     * <BR><BR>
     * このオブジェクト[org.maachang.commons.net.NetConfig]が利用可能であるかチェックします.
     * このメソッドで[false]が返された場合はこのオブジェクトは利用できません.
     * <BR>
     * @return boolean オブジェクト利用可能チェックが返されます.<BR>
     *                 [true]が返された場合利用可能です.<BR>
     *                 [false]が返された場合利用不可能です.
     */
    public final boolean useObject()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_useFlg ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 情報更新.
     * <BR><BR>
     * 対象の条件を更新します.
     */
    public final void flush()
    {
        int i,j ;
        int len ;
        int lenJ ;
        
        String addr = null ;
        NetPause pause = null ;
        String[] adapterNames = null ;
        NetAdapter adapter = null ;
        
        try{
            
            adapter = NetAdapterImpl.getAdapter() ;
            adapterNames = adapter.getNames() ;
            
            if( adapterNames != null ){
                
                len = adapterNames.length ;
                synchronized( m_sync.get() ){
                    
                    if( m_adapterObject != null ){
                        m_adapterObject.clear() ;
                    }
                    
                    m_adapterObject = adapter ;
                    m_useFlg = true ;
                    
                    m_addr.clear() ;
                    m_adapter.clear() ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        pause = new NetPause() ;
                        m_adapter.add( adapterNames[ i ],pause ) ;
                        
                        // IPアドレスをセット.
                        lenJ = adapter.getElement( adapterNames[ i ],NetAdapter.CATEGORY_IPADDR ) ;
                        if( lenJ > 0 ){
                            
                            pause.ipAddr = new String[ lenJ ] ;
                            
                            for( j = 0 ; j < lenJ ; j ++ ){
                                
                                addr = adapter.getCategory( adapterNames[ i ],NetAdapter.CATEGORY_IPADDR,j ) ;
                                m_addr.add( addr,adapterNames[ i ] ) ;
                                pause.ipAddr[ j ] = addr ;
                                
                            }
                            
                        }
                        else{
                            pause.ipAddr = new String[ 1 ] ;
                            pause.ipAddr[ 0 ] = NetDef.LOCAL_HOST ;
                        }
                        
                        // サブネットアドレスをセット.
                        lenJ = adapter.getElement( adapterNames[ i ],NetAdapter.CATEGORY_SUBNET ) ;
                        if( lenJ > 0 ){
                            
                            pause.subNet = adapter.getCategory( adapterNames[ i ],NetAdapter.CATEGORY_SUBNET,0 ) ;
                            
                        }
                        
                        // MACアドレスをセット.
                        lenJ = adapter.getElement( adapterNames[ i ],NetAdapter.CATEGORY_MACADDR ) ;
                        if( lenJ > 0 ){
                            
                            pause.macAddr = adapter.getCategory( adapterNames[ i ],NetAdapter.CATEGORY_MACADDR,0 ) ;
                            
                        }
                        
                        adapterNames[ i ] = null ;
                        
                    }
                    
                }
                
            }
            else{
                
                synchronized( m_sync.get() ){
                    
                    if( adapter != null ){
                        adapter.clear() ;
                    }
                    if( m_adapterObject != null ){
                        m_adapterObject.clear() ;
                    }
                    
                    m_adapterObject = null ;
                    m_addr.clear() ;
                    m_adapter.clear() ;
                    m_useFlg = false ;
                    
                }
                
            }
        }catch( Exception t ){
            try{
                synchronized( m_sync.get() ){
                    
                    if( adapter != null ){
                        adapter.clear() ;
                    }
                    if( m_adapterObject != null ){
                        m_adapterObject.clear() ;
                    }
                    
                    m_adapterObject = null ;
                    m_addr.clear() ;
                    m_adapter.clear() ;
                    m_useFlg = false ;
                    
                }
            }catch( Exception tt ){
            }
        }finally{
            
            addr = null ;
            pause = null ;
            adapterNames = null ;
            adapter = null ;
        }
        
    }
    
    /**
     * 媒体情報を取得.
     * <BR><BR>
     * ネットワーク媒体情報を取得します.
     * <BR>
     * @return NetAdapter ネットワーク媒体情報が返されます.
     * @exception UnsupportException サポート外例外.
     */
    public final NetAdapter getNetAdapter()
        throws UnsupportException
    {
        NetAdapter ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                this.flush() ;
                
                if( this.useObject() == false ){
                    throw new UnsupportException(
                        "媒体名取得はこのOSではサポートされていません"
                    ) ;
                }
                
                ret = m_adapterObject ;
                
            }
        }catch( UnsupportException ue ){
            ret = null ;
            throw ue ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納IPアドレス群を取得.
     * <BR><BR>
     * 格納IPアドレス群を取得します.
     * <BR>
     * @return String[] 格納アドレス群が返されます.
     */
    public final String[] getAddress()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_addr.getNames() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納媒体名群を取得.
     * <BR><BR>
     * 格納媒体名群を取得します.
     * <BR>
     * @return String[] 格納媒体名が返されます.
     */
    public final String[] getAdapters()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_adapter.getNames() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象アドレスに対する媒体名を取得.
     * <BR><BR>
     * 対象アドレスに対する媒体名を取得します.
     * <BR>
     * @param addr アドレス情報を指定します.
     * @return String 対象の媒体名が返されます.
     */
    public final String getAddressByAdapter( InetAddress addr )
    {
        String ret = null ;
        
        try{
            ret = this.getAddressByAdapter( addr.getHostAddress() ) ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象アドレスに対する媒体名を取得.
     * <BR><BR>
     * 対象アドレスに対する媒体名を取得します.
     * <BR>
     * @param addr アドレス情報を指定します.
     * @return String 対象の媒体名が返されます.
     */
    public final String getAddressByAdapter( String addr )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( String )m_addr.get( addr ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象媒体名に対する格納アドレス数を取得.
     * <BR><BR>
     * 対象媒体名に対する格納アドレス数を取得します.
     * <BR>
     * @param adp 媒体名を設定します.
     * @return int 格納アドレス数が返されます.
     */
    public final int getAdapterByAddressLength( String adp )
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( ( NetPause )m_adapter.get( adp ) ).ipAddr.length ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象媒体名に対する格納アドレスを取得.
     * <BR><BR>
     * 対象媒体名に対する格納アドレスを取得します.
     * <BR>
     * @param adp 媒体名を設定します.
     * @param no 媒体名に対する取得項番を設定します.
     * @return String 格納アドレスが返されます.
     */
    public final String getAdapterByAddress( String adp,int no )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( ( NetPause )m_adapter.get( adp ) ).ipAddr[ no ] ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象媒体名に対するサブネットマスクを取得.
     * <BR><BR>
     * 対象媒体名に対するサブネットマスクを取得します.
     * <BR>
     * @param adp 媒体名を設定します.
     * @return String 格納サブネットマスクが返されます.
     */
    public final String getAdapterBySubnetMask( String adp )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( ( NetPause )m_adapter.get( adp ) ).subNet  ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象媒体名に対するMACアドレスを取得.
     * <BR><BR>
     * 対象媒体名に対するMACアドレスを取得します.
     * <BR>
     * @param adp 媒体名を設定します.
     * @return String 格納MACアドレスが返されます.
     */
    public final String getAdapterByMacAddress( String adp )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( ( NetPause )m_adapter.get( adp ) ).subNet  ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定アドレス情報が存在するかチェック.
     * <BR><BR>
     * 指定アドレス情報が存在するかチェックします.
     * <BR>
     * @param addr チェック対象のアドレスを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public final boolean isUseAddress( InetAddress addr )
    {
        boolean ret ;
        
        try{
            ret = this.isUseAddress( addr.getHostAddress() ) ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 指定アドレス情報が存在するかチェック.
     * <BR><BR>
     * 指定アドレス情報が存在するかチェックします.
     * <BR>
     * @param addr チェック対象のアドレスを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public final boolean isUseAddress( String addr )
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_addr.isData( addr ) ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 指定アドレス情報が存在するかチェック.
     * <BR><BR>
     * 指定アドレス情報が存在するかチェックします.
     * <BR>
     * @param adp チェック対象の媒体名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public final boolean isUseAdapter( String adp )
    {
        int i ;
        int len ;
        
        boolean ret ;
        String addr = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( ( len = this.getAdapterByAddressLength( adp ) ) != 0 ){
                    
                    ret = false ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        addr = this.getAdapterByAddress( adp,i ) ;
                        
                        if(
                            addr.equals( NetDef.LOCAL_HOST ) == false &&
                            addr.equals( NetDef.NOT_ADDR ) == false
                        )
                        {
                            ret = true ;
                            break ;
                        }
                        
                    }
                }
                else{
                    ret = false ;
                }
            }
        }catch( Exception t ){
            ret = false ;
        }finally{
            addr = null ;
        }
        
        return ret ;
    }
    
}

