/*
 * @(#)NetDef.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;
import java.net.UnknownHostException;

import org.maachang.commons.def.BaseDef;


/**
 * net定義.
 *  
 * @version 1.0.0 2003/12/21
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class NetDef
{
    
    /**
     * コンストラクタ.
     */
    private NetDef()
    {
    }
    
    /**
     * デフォルト文字列変換.
     */
    public static final String DEF_CHARSET = BaseDef.DEF_CHARSET ;
    
    /**
     * ローカルホストアドレス.
     * <BR><BR>
     * ローカルホストアドレスを示す番号が定義されています.
     */
    public static final String LOCAL_HOST = "127.0.0.1" ;
    
    /**
     * 非アドレス情報.
     * <BR><BR>
     * 無効なアドレスである場合の定義です.
     */
    public static final String NOT_ADDR = "0.0.0.0" ;
    
    /**
     * ポート番号最小値.
     * <BR><BR>
     * ポート番号の最小値です.
     */
    public static final int PORT_MIN = 0 ;
    
    /**
     * ポート番号最大値.
     * <BR><BR>
     * ポート番号の最大値です.
     */
    public static final int PORT_MAX = 65535 ;
    
    /**
     * ポート番号レジスタット領域最小値.
     * <BR><BR>
     * ポート番号レジスタット領域最小値です.
     */
    public static final int PORT_REG_MIN = 1024 ;
    
    /**
     * ポート番号レジスタット領域最大値.
     * <BR><BR>
     * ポート番号レジスタット領域最大値です.
     */
    public static final int PORT_REG_MAX = 49151 ;
    
    /**
     * ポート番号プライベート領域最小値.
     * <BR><BR>
     * ポート番号プライベート領域最小値です.
     */
    public static final int PORT_PRI_MIN = 49152 ;
    
    /**
     * ポート番号プライベート領域最大値.
     * <BR><BR>
     * ポート番号プライベート領域最大値です.
     */
    public static final int PORT_PRI_MAX = 65535 ;
    
    /**
     * デフォルトバッファ長.
     * <BR><BR>
     * 送受信のデフォルトバッファ長を示します.
     */
    public static final int DEF_BUFLEN = 65535 ;
    
    /**
     * UDP最大データ長.
     */
    public static final int UDP_MAXBYTE = 65535 ;
    
    /**
     * 送信用ネットワークバッファ係数.
     */
    public static final double NETWORK_SEND_BUFFER = 0.75 ;
    
    /**
     * 受信用ネットワークバッファ係数.
     */
    public static final double NETWORK_RECEIVE_BUFFER = 1.25 ;
    
    /**
     * マックアドレスサイズ.
     */
    public static final int MAC_ADDRESS_LENGTH = 6 ;
    
    
    
    /**
     * InetAddress.getAddress()で生成されたバイナリ情報をIPアドレスに変換します.
     * <BR><BR>
     * InetAddress.getAddress()で生成されたバイナリ情報をIPアドレスに変換.
     * <BR>
     * @param addr アドレスを示すバイナリ情報を設定します.
     * @return InetAddress 変換されたInetAddressが返されます.
     * @exception UnknownHostException 対象アドレスの変換が失敗した場合に返されます.
     */
    public static final InetAddress getByAddress( byte[] addr )
        throws UnknownHostException
    {
        int i ;
        int len ;
        int lenJ ;
        
        StringBuffer buf = null ;
        InetAddress ret = null ;
        
        if( addr == null || ( len = addr.length ) <= 0 ){
            throw new UnknownHostException( "対象の条件は不正です" ) ;
        }
        // IPV4の場合.
        else if( len == 4 ){
            
            buf = new StringBuffer() ;
            buf.append( ( addr[ 0 ] & 0x000000ff ) ) ;
            buf.append( "." ) ;
            buf.append( ( addr[ 1 ] & 0x000000ff ) ) ;
            buf.append( "." ) ;
            buf.append( ( addr[ 2 ] & 0x000000ff ) ) ;
            buf.append( "." ) ;
            buf.append( ( addr[ 3 ] & 0x000000ff ) ) ;
            ret = InetAddress.getByName( buf.toString() ) ;
            buf = null ;
            
        }
        // IPV6の場合.
        else if( len == 16 ){
            
            len /= 2 ;
            lenJ = len-1 ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                buf.append(
                    Integer.toHexString(
                        ( (addr[ i << 1 ] << 8 ) & 0x0000ff00 ) |
                        ( addr[ ( i << 1 ) + 1 ] & 0x000000ff )
                    )
                ) ;
                
                if ( i < lenJ ){
                    buf.append( ":" );
                }
                
            }
            
            ret = InetAddress.getByName( buf.toString() ) ;
            buf = null ;
            
        }
        // その他.
        else{
            throw new UnknownHostException(
                "対象の条件はアドレスではありません"
            ) ;
        }
        
        return ret ;
    }
}

