/*
 * @(#)NetIpPortV4.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.UtilCom;


/**
 * ネットIPPortコンバート.
 * <BR><BR>
 * IPPort(IPV4)でのコンバート処理をサポートします.
 *  
 * @version 2005/02/07
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class NetIpPortV4
{
    
    /**
     * コンストラクタ.
     */
    private NetIpPortV4()
    {
    }
    
    /**
     * IP区切り文字.
     */
    public static final char IP_PAUSE = '.' ;
    
    /**
     * IPポート番号存在なし.
     */
    public static final long NOT_SETIPPORT = 0x0000000000000000L ;
    
    /**
     * ネットワークアドレスサイズ.
     * <BR><BR>
     * ネットワークアドレス数(IPV4).
     */
    public static final int NET_LENGTH = 4 ;
    
    
    
    /**
     * 指定情報から指定long値へ変換.
     * <BR><BR>
     * 指定情報から指定long値へ変換します.
     * <BR>
     * @param host 変換対象のホスト名を設定します.
     * @param port 変換対象のポート番号を設定します.
     * @return 変換された指定long値が返されます.
     * @exception InputException 入力例外.
     */
    public static final long convertIpPortByLong( String host,int port )
        throws InputException
    {
        long ret ;
        
        if( host == null || port < NetDef.PORT_MIN || port >= NetDef.PORT_MAX ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            ret = NetIpPortV4.convertIpPortByLong(
                InetAddress.getByName( host ),port
            ) ;
        }catch( Exception t ){
            ret = NetIpPortV4.NOT_SETIPPORT ;
        }
        
        return ret ;
    }
    
    /**
     * 指定情報から指定long値へ変換.
     * <BR><BR>
     * 指定情報から指定long値へ変換します.
     * <BR>
     * @param addrPort 変換対象のConnectAddressオブジェクトを設定します.
     * @return 変換された指定long値が返されます.
     * @exception InputException 入力例外.
     */
    public static final long convertIpPortByLong( ConnectAddress addrPort )
        throws InputException
    {
        long ret ;
        
        if( addrPort == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            ret = NetIpPortV4.convertIpPortByLong(
                addrPort.getAddress(),addrPort.getPort()
            ) ;
        }catch( Exception t ){
            ret = NetIpPortV4.NOT_SETIPPORT ;
        }
        
        return ret ;
    }
    
    /**
     * 指定情報から指定long値へ変換.
     * <BR><BR>
     * 指定情報から指定long値へ変換します.
     * <BR>
     * @param host 変換対象のInetAddressオブジェクトを設定します.
     * @param port 変換対象のポート番号を設定します.
     * @return 変換された指定long値が返されます.
     * @exception InputException 入力例外.
     */
    public static final long convertIpPortByLong( InetAddress host,int port )
        throws InputException
    {
        long ret ;
        
        byte[] ip = null ;
        
        if( host == null || port < NetDef.PORT_MIN || port >= NetDef.PORT_MAX ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            ip = host.getAddress() ;
            ret = NetIpPortV4.NOT_SETIPPORT ;
            
            ret = ( long )(
                ( ip[ 0 ] & 0x00000000000000ffL ) |
                ( ( ip[ 1 ] & 0x00000000000000ffL ) << 8 ) |
                ( ( ip[ 2 ] & 0x00000000000000ffL ) << 16 ) |
                ( ( ip[ 3 ] & 0x00000000000000ffL ) << 24 ) |
                ( ( port & 0x000000000000ffffL ) << 32 )
            ) ;
            
        }catch( Exception t ){
            ret = NetIpPortV4.NOT_SETIPPORT ;
        }finally{
            ip = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定long値をIPアドレスとポート番号に変換.
     * <BR><BR>
     * 指定long値をIPアドレスとポート番号に変換します.
     * <BR>
     * @param port 変換結果のポート番号が返されます.
     * @param code 変換対象のlong値を設定します.
     * @return InetAddress 変換されたIPアドレスが返されます.
     * @exception InputException 入力例外.
     */
    public static final InetAddress convertLongByIpPort( int[] port,long code )
        throws InputException
    {
        
        InetAddress ret = null ;
        StringBuffer buf = null ;
        
        if( port == null || port.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        buf.append( ( int )( code & 0x00000000000000ffL ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x000000000000ff00L ) >> 8 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x0000000000ff0000L ) >> 16 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x00000000ff000000L ) >> 24 ) ) ;
        
        try{
            ret = InetAddress.getByName( buf.toString() ) ;
            port[ 0 ] = ( int )( ( code & 0x0000ffff00000000L ) >> 32 ) ;
        }catch( Exception t ){
            ret = null ;
            port[ 0 ] = 0 ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定long値からConnectAddressオブジェクト変換.
     * <BR><BR>
     * 指定long値からConnectAddressオブジェクトに変換します.
     * <BR>
     * @param code 変換対象の指定long値を設定します.
     * @return ConnectAddress 変換されたオブジェクトが返されます.
     * @exception InputException 入力例外.
     */
    public static final ConnectAddress convertLongByIpPort( long code )
        throws InputException
    {
        int port ;
        
        StringBuffer buf = null ;
        ConnectAddress ret = null ;
        
        if( code == NetIpPortV4.NOT_SETIPPORT ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        buf.append( ( int )( code & 0x00000000000000ffL ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x000000000000ff00L ) >> 8 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x0000000000ff0000L ) >> 16 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x00000000ff000000L ) >> 24 ) ) ;
        
        port  =  ( int )( ( code & 0x0000ffff00000000L ) >> 32 ) ;
        
        try{
            ret.create( InetAddress.getByName( buf.toString() ),port ) ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定long値をIPアドレスに変換.
     * <BR><BR>
     * 指定long値をIPアドレスに変換します.
     * <BR>
     * @param code 変換対象のlong値を設定します.
     * @return InetAddress 変換されたIPアドレスが返されます.
     */
    public static final InetAddress convertLongByIpAddr( long code )
    {
        
        StringBuffer buf = null ;
        InetAddress ret = null ;
        
        buf = new StringBuffer() ;
        
        buf.append( ( int )( code & 0x00000000000000ffL ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x000000000000ff00L ) >> 8 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x0000000000ff0000L ) >> 16 ) ) ;
        buf.append( NetIpPortV4.IP_PAUSE ) ;
        buf.append( ( int )( ( code & 0x00000000ff000000L ) >> 24 ) ) ;
        
        try{
            ret = InetAddress.getByName( buf.toString() ) ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定long値をポート番号に変換.
     * <BR><BR>
     * 指定long値をポート番号に変換します.
     * <BR>
     * @param code 変換対象のlong値を設定します.
     * @return long 変換されたポート番号が返されます.
     */
    public static final int convertLongByPort( long code )
    {
        return ( int )( ( code & 0x0000ffff00000000L ) >> 32 ) ;
    }
    
    
    /**
     * IPポート番号を暗号化.
     * <BR><BR>
     * 指定されたIPポート番号を暗号化します.
     * <BR>
     * @param srcIpPort 暗号対象のIPポート番号を設定します.
     * @return long 暗号化されたIPポート番号が返されます.
     */
    public static final long setSrcIpPort( long srcIpPort )
    {
        return ( long )(
            NetIpPortV4.cnvIpPortA( NetIpPortV4.cnvIpPort( srcIpPort ) )
        ) ;
    }
    
    /**
     * 暗号化されたIPポート番号を元に戻す.
     * <BR><BR>
     * 指定対象の暗号化されたIPポート番号を元に戻します.
     * <BR>
     * @param srcIpPort 対象の暗号化されたIPポート番号を設定します.
     * @return long 元に戻されたIPポート番号が返されます.
     */
    public static final long getDestIpPort( long srcIpPort )
    {
        return ( long )(
            NetIpPortV4.cnvIpPort( NetIpPortV4.cnvIpPortB( srcIpPort ) ) & 0x0000ffffffffffffL
        ) ;
    }
    
    
    
    
    /**
     * srcIpPort コンバート処理.
     */
    private static final long cnvIpPort( long srcIpPort )
    {
        long nSrcIpPort = ( ~ srcIpPort ) ;
        return ( long )(
            ( srcIpPort & 0xffff000000000000L ) |
                
            ( ( srcIpPort & 0x0000000000000001L ) << 47 ) |
            ( ( srcIpPort & 0x0000800000000000L ) >> 47 ) |
            ( ( nSrcIpPort & 0x0000000000000002L ) << 42 ) |
            ( ( nSrcIpPort & 0x0000080000000000L ) >> 42 ) |
            ( ( srcIpPort & 0x0000000000000004L ) << 37 ) |
            ( ( srcIpPort & 0x0000008000000000L ) >> 37 ) |
            ( ( nSrcIpPort & 0x0000000000000008L ) << 32 ) |
            ( ( nSrcIpPort & 0x0000000800000000L ) >> 32 ) |
                
            ( ( nSrcIpPort & 0x0000000000000010L ) << 42 ) |
            ( ( nSrcIpPort & 0x0000400000000000L ) >> 42 ) |
            ( ( srcIpPort & 0x0000000000000020L ) << 37 ) |
            ( ( srcIpPort & 0x0000040000000000L ) >> 37 ) |
            ( ( nSrcIpPort & 0x0000000000000040L ) << 32 ) |
            ( ( nSrcIpPort & 0x0000004000000000L ) >> 32 ) |
            ( ( srcIpPort & 0x0000000000000080L ) << 27 ) |
            ( ( srcIpPort & 0x0000000400000000L ) >> 27 ) |
                
            ( ( srcIpPort & 0x0000000000000100L ) << 37 ) |
            ( ( srcIpPort & 0x0000200000000000L ) >> 37 ) |
            ( ( nSrcIpPort & 0x0000000000000200L ) << 32 ) |
            ( ( nSrcIpPort & 0x0000020000000000L ) >> 32 ) |
            ( ( nSrcIpPort & 0x0000000000000400L ) << 27 ) |
            ( ( nSrcIpPort & 0x0000002000000000L ) >> 27 ) |
            ( ( srcIpPort & 0x0000000000000800L ) << 22 ) |
            ( ( srcIpPort & 0x0000000200000000L ) >> 22 ) |
                
            ( ( nSrcIpPort & 0x0000000000001000L ) << 32 ) |
            ( ( nSrcIpPort & 0x0000100000000000L ) >> 32 ) |
            ( ( srcIpPort & 0x0000000000002000L ) << 27 ) |
            ( ( srcIpPort & 0x0000010000000000L ) >> 27 ) |
            ( ( srcIpPort & 0x0000000000004000L ) << 22 ) |
            ( ( srcIpPort & 0x0000001000000000L ) >> 22 ) |
            ( ( nSrcIpPort & 0x0000000000008000L ) << 17 ) |
            ( ( nSrcIpPort & 0x0000000100000000L ) >> 17 ) |
                
            ( ( srcIpPort & 0x0000000000010000L ) << 15 ) |
            ( ( srcIpPort & 0x0000000080000000L ) >> 15 ) |
            ( ( nSrcIpPort & 0x0000000000020000L ) << 10 ) |
            ( ( nSrcIpPort & 0x0000000008000000L ) >> 10 ) |
            ( ( srcIpPort & 0x0000000000040000L ) << 12 ) |
            ( ( srcIpPort & 0x0000000040000000L ) >> 12 ) |
            ( ( nSrcIpPort & 0x0000000000080000L ) << 7 ) |
            ( ( nSrcIpPort & 0x0000000004000000L ) >> 7 ) |
                
            ( ( srcIpPort & 0x0000000000100000L ) << 9 ) |
            ( ( srcIpPort & 0x0000000020000000L ) >> 9 ) |
            ( ( nSrcIpPort & 0x0000000000200000L ) << 4 ) |
            ( ( nSrcIpPort & 0x0000000002000000L ) >> 4 ) |
            ( ( nSrcIpPort & 0x0000000000400000L ) << 6 ) |
            ( ( nSrcIpPort & 0x0000000010000000L ) >> 6 ) |
            ( ( srcIpPort & 0x0000000000800000L ) << 1 ) |
            ( ( srcIpPort & 0x0000000001000000L ) >> 1 )
        ) ;
    }
    
    /**
     * srcIpPort コンバート変換(A).
     */
    private static final long cnvIpPortA( long srcIpPort )
    {
        int pat ;
        
        srcIpPort = ( long )(
            (
                ( ( UtilCom.random( 0x00010000 ) & 0x000000000000ffffL ) << 48 ) &
                0xffff000000000000L
            ) | ( srcIpPort & 0x0000ffffffffffffL )
        ) ;
        
        pat = NetIpPortV4.getCnvPat( srcIpPort ) ;
        
        switch( pat ){
            case 0 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ~srcIpPort ) & 0x0000ffffffffffffL )
                ) ;
                break ;
            case 1 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( srcIpPort & 0x0000ffff00000000L ) >> 32 ) |
                    ( ( (~srcIpPort) & 0x00000000ffff0000L ) << 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16 )
                ) ;
                break ;
            case 2 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( ( ~srcIpPort ) & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
            case 3 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( (~srcIpPort) & 0x0000ffff00000000L ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16 )
                ) ;
                break ;
            case 4 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( srcIpPort & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( ( ~srcIpPort ) & 0x00000000ffff0000L ) << 16 ) |
                    ( ( ~srcIpPort ) & 0x000000000000ffffL )
                ) ;
                break ;
            case 5 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( srcIpPort & 0x0000ffff00000000L ) >> 32 ) |
                    ( srcIpPort & 0x00000000ffff0000L ) |
                    ( ( (~srcIpPort) & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
            case 6 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( (~srcIpPort) & 0x0000ffff00000000L ) >> 32 ) |
                    ( ( (~srcIpPort) & 0x00000000ffff0000L ) << 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16)
                ) ;
                break ;
            case 7 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( srcIpPort & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
        }
        
        return srcIpPort ;
    }
    
    /**
     * srcIpPort コンバート戻し変換(B).
     */
    private static final long cnvIpPortB( long srcIpPort )
    {
        int pat ;
        
        pat = NetIpPortV4.getCnvPat( srcIpPort ) ;
        
        switch( pat ){
            case 0 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ~srcIpPort ) & 0x0000ffffffffffffL )
                ) ;
                break ;
            case 1 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
            case 2 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 32 ) |
                    ( ( ( ~srcIpPort ) & 0x00000000ffff0000L ) << 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16 )
                ) ;
                break ;
            case 3 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ~srcIpPort ) & 0x0000ffff00000000L ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16 )
                ) ;
                break ;
            case 4 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) << 16 ) |
                    ( ( ~srcIpPort ) & 0x000000000000ffffL )
                ) ;
                break ;
            case 5 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 32 ) |
                    ( srcIpPort & 0x00000000ffff0000L ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
            case 6 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( ( ~srcIpPort ) & 0x0000ffff00000000L ) >> 16 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) >> 16 ) |
                    ( ( ( ~srcIpPort ) & 0x000000000000ffffL ) << 32 )
                ) ;
                break ;
            case 7 :
                srcIpPort = ( long )(
                    ( srcIpPort & 0xffff000000000000L ) |
                    ( ( srcIpPort & 0x0000ffff00000000L ) >> 32 ) |
                    ( ( srcIpPort & 0x00000000ffff0000L ) << 16 ) |
                    ( ( srcIpPort & 0x000000000000ffffL ) << 16 )
                ) ;
                break ;
        }
        
        return srcIpPort ;
    }
    
    /**
     * パターン暗号コードを取得.
     */
    private static final int getCnvPat( long srcIpPort )
    {
        int base ;
        int ret ;
        
        base = ( int )(
            ( ( srcIpPort & 0x2000000000000000L ) >> 61 ) |
            ( ( (~srcIpPort) & 0x0100000000000000L ) >> 55 ) |
            ( ( srcIpPort & 0x0040000000000000L ) >> 52 )
        ) ;
        
        switch( base ){
            case 0 :
                ret = ( int )(
                    ( ( (~srcIpPort) & 0x0020000000000000L ) >> 53 ) |
                    ( ( srcIpPort & 0x0400000000000000L ) >> 57 ) |
                    ( ( srcIpPort & 0x0001000000000000L ) >> 46 )
                ) ;
                break ;
            case 1 :
                ret = ( int )(
                    ( ( srcIpPort & 0x0040000000000000L ) >> 54 ) |
                    ( ( srcIpPort & 0x0800000000000000L ) >> 58 ) |
                    ( ( (~srcIpPort) & 0x0002000000000000L ) >> 47 )
                ) ;
                break ;
            case 2 :
                ret = ( int )(
                    ( ( srcIpPort & 0x0200000000000000L ) >> 57 ) |
                    ( ( (~srcIpPort) & 0x4000000000000000L ) >> 61 ) |
                    ( ( srcIpPort & 0x0010000000000000L ) >> 50 )
                ) ;
                break ;
            case 3 :
                ret = ( int )(
                    ( ( srcIpPort & 0x0002000000000000L ) >> 49 ) |
                    ( ( srcIpPort & 0x0040000000000000L ) >> 53 ) |
                    ( ( (~srcIpPort) & 0x1000000000000000L ) >> 58 )
                ) ;
                break ;
            case 4 :
                ret = ( int )(
                    ( ( (~srcIpPort) & 0x0010000000000000L ) >> 52 ) |
                    ( ( srcIpPort & 0x0200000000000000L ) >> 56 ) |
                    ( ( (~srcIpPort) & 0x1000000000000000L ) >> 58 )
                ) ;
                break ;
            case 5 :
                ret = ( int )(
                    ( ( srcIpPort & 0x0100000000000000L ) >> 56 ) |
                    ( ( (~srcIpPort) & 0x0010000000000000L ) >> 51 ) |
                    ( ( srcIpPort & 0x0001000000000000L ) >> 46 )
                ) ;
                break ;
            case 6 :
                ret = ( int )(
                    ( ( srcIpPort & 0x0001000000000000L ) >> 48 ) |
                    ( ( (~srcIpPort) & 0x0010000000000000L ) >> 51 ) |
                    ( ( (~srcIpPort) & 0x0100000000000000L ) >> 54 )
                ) ;
                break ;
            case 7 :
                ret = ( int )(
                    ( ( (~srcIpPort) & 0x4000000000000000L ) >> 62 ) |
                    ( ( srcIpPort & 0x0800000000000000L ) >> 58 ) |
                    ( ( srcIpPort & 0x0020000000000000L ) >> 51 )
                ) ;
                break ;
            default :
                ret = ( int )(
                    ( ( (~srcIpPort) & 0x2000000000000000L ) >> 61 ) |
                    ( ( srcIpPort & 0x0100000000000000L ) >> 55 ) |
                    ( ( (~srcIpPort) & 0x0040000000000000L ) >> 52 )
                ) ;
        }
        
        return ret ;
        
    }
    
}

