/*
 * @(#)NowAddressCore.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.ExecutionException;


/**
 * 現在のIPアドレス情報取得.
 * <BR><BR>
 * 現在のIPアドレス情報を取得します.
 *  
 * @version 1.0.0 2004/09/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class NowAddressCore
{
    
    /**
     * コマンド引数 : 現在IPアドレスを取得.
     */
    public static final String NOW_IPADDR = "-I" ;
    
    /**
     * コマンド引数 : 現在ドメイン名を取得.
     */
    public static final String NOW_DOMAIN = "-D" ;
    
    /**
     * コマンド引数 : 現在管理されている複数のアドレスを取得.
     */
    public static final String ALL_IPADDR = "-A" ;
    
    /**
     * コマンド引数 : 全ての条件の情報を取得.
     */
    public static final String ALL_DATA = "-X" ;
    
    /**
     * 複数アドレス取得時の区切り文字.
     */
    public static final String ALL_PAUSE = ":" ;
    
    /**
     * 全ての条件情報取得時の区切り文字.
     */
    public static final String ALL_DATA_PAUSE = "$" ;
    
    /**
     * 全アドレスに対するドメイン名区切り文字.
     */
    public static final String DOMAIN_PAUSE = "/" ;
    
    
    /**
     * コマンド実施 : 条件なし.
     */
    public static final int CODE_NOT = 0x00000000 ;
    
    /**
     * コマンド実施 : 現在IPアドレス取得.
     */
    public static final int CODE_NOW_IPADDR = 0x00000001 ;
    
    /**
     * コマンド実施 : 現在ドメイン名取得.
     */
    public static final int CODE_NOW_DOMAIN = 0x00000002 ;
    
    /**
     * コマンド実施 : 現在管理されている複数のアドレスを取得.
     */
    public static final int CODE_ALL_IPADDR = 0x00000003 ;
    
    /**
     * コマンド実施 : 現在管理されている複数のドメイン名を取得.
     */
    public static final int CODE_ALL_DOMAIN = 0x00000004 ;
    
    /**
     * コマンド実施 : 現在管理されている全ての条件の情報を取得.
     */
    public static final int CODE_ALL_DATA = 0x00000005 ;
    
    
    
    /**
     * メイン定義.
     * <BR><BR>
     * メイン処理の定義を行います.
     * @param args 実行処理のパラメータをセットします.
     */
    public static void main( String[] args )
    {
        String cmd = null ;
        NowAddressCore addr = null ;
        
        try{
            addr = new NowAddressCore() ;
            cmd = addr.execution( args ) ;
        }catch( Exception t ){
            cmd = NetDef.LOCAL_HOST ;
        }finally{
            addr = null ;
        }
        
        // 取得された情報を出力.
        System.out.println( cmd ) ;
    }
    
    /**
     * コンストラクタ.
     */
    protected NowAddressCore()
    {
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 対象条件を実行します.
     * <BR>
     * @param args 実行処理のパラメータをセットします.
     * @return String 取得された情報が返されます.
     * @exception ExecutionException 実施例外.
     */
    private final String execution( String[] args )
        throws ExecutionException
    {
        int code ;
        
        String ret = null ;
        StringBuffer buf = null ;
        
        code = this.getCommand( args ) ;
        
        try{
            
            if( code == NowAddressCore.CODE_ALL_DATA ){
                
                buf = new StringBuffer() ;
                buf.append( this.getCode( NowAddressCore.CODE_NOW_IPADDR ) ) ;
                buf.append( NowAddressCore.ALL_DATA_PAUSE ) ;
                buf.append( this.getCode( NowAddressCore.CODE_NOW_DOMAIN ) ) ;
                buf.append( NowAddressCore.ALL_DATA_PAUSE ) ;
                buf.append( this.getCode( NowAddressCore.CODE_ALL_IPADDR ) ) ;
                buf.append( NowAddressCore.ALL_DATA_PAUSE ) ;
                
                ret = buf.toString() ;
                
            }
            else{
                
                ret = this.getCode( code ) ;
                
            }
            
        }catch( ExecutionException ex ){
            throw ex ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 各情報を取得.
     * <BR><BR>
     * 指定コードに対する各情報を取得します.
     * <BR>
     * @param code 取得対象のコードを設定します.
     * @return String 取得された文字列が返されます.
     * @exception ExecutionException 実施例外.
     */
    private final String getCode( int code )
        throws ExecutionException
    {
        int i ;
        int len ;
        
        String addr = null ;
        String host = null ;
        String ret = null ;
        StringBuffer buf = null ;
        InetAddress[] addrs = null ;
        
        try{
            
            switch( code ){
                case NowAddressCore.CODE_NOW_IPADDR :
                    
                    addr = InetAddress.getLocalHost().getHostAddress().trim() ;
                    
                    if( addr.equals( NetDef.LOCAL_HOST ) == true ){
                        addr = NetDef.NOT_ADDR ;
                    }
                    
                    ret = addr ;
                    
                    break ;
                    
                case NowAddressCore.CODE_NOW_DOMAIN :
                    
                    ret = InetAddress.getLocalHost().getHostName().trim() ;
                    break ;
                    
                case NowAddressCore.CODE_ALL_IPADDR :
                    
                    addrs = InetAddress.getAllByName(
                        InetAddress.getLocalHost().getHostName()
                    ) ;
                    
                    if( addrs != null ){
                        
                        buf = new StringBuffer() ;
                        len = addrs.length ;
                        
                        if( len > 0 ){
                            
                            for( i = 0 ; i < len ; i ++ ){
                                
                                addr = addrs[ i ].getHostAddress().trim() ;
                                
                                if( i != 0 ){
                                    buf.append( NowAddressCore.ALL_PAUSE ) ;
                                }
                                
                                if( addr.equals( NetDef.LOCAL_HOST ) == true ){
                                    buf.append( NetDef.NOT_ADDR ) ;
                                }
                                else{
                                    
                                    buf.append( addr ) ;
                                    host = addrs[ i ].getHostName().trim() ;
                                    
                                    if( host.length() != 0 && addr.equals( host ) == false ){
                                        buf.append( NowAddressCore.DOMAIN_PAUSE ) ;
                                        buf.append( host ) ;
                                    }
                                    
                                }
                                
                            }
                            
                            ret = buf.toString() ;
                            
                        }
                        else{
                            ret = NetDef.NOT_ADDR ;
                        }
                        
                    }
                    else{
                        ret = NetDef.NOT_ADDR ;
                    }
                    
                    break ;
                    
                default :
                    
                    ret = NetDef.NOT_ADDR ;
                    break ;
                
            }
            
        }catch( Exception t ){
            throw new ExecutionException( t ) ;
        }finally{
            addr = null ;
            buf = null ;
            addrs = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * 指定コマンド情報に対するコマンドコードを取得.
     * <BR><BR>
     * 指定コマンド情報に対するコマンドコードを取得します.
     * <BR>
     * @param args 対象のコマンド群を設定します.
     * @return int 対象のコマンドコードが返されます.
     */
    private final int getCommand( String[] args )
    {
        int i ;
        int len ;
        int ret ;
        
        String cmd = null ;
        
        if( args == null || ( len = args.length ) <= 0 ){
            ret = NowAddressCore.CODE_NOW_IPADDR ;
        }
        else{
            
            ret = NowAddressCore.CODE_NOW_IPADDR ;
            for( i = 0 ; i < len ; i ++ ){
                
                cmd = args[ i ] ;
                
                if( cmd.equalsIgnoreCase( NowAddressCore.NOW_IPADDR ) == true ){
                    ret = NowAddressCore.CODE_NOW_IPADDR ;
                    break ;
                }
                else if( cmd.equalsIgnoreCase( NowAddressCore.NOW_DOMAIN ) == true ){
                    ret = NowAddressCore.CODE_NOW_DOMAIN ;
                    break ;
                }
                else if( cmd.equalsIgnoreCase( NowAddressCore.ALL_IPADDR ) == true ){
                    ret = NowAddressCore.CODE_ALL_IPADDR ;
                    break ;
                }
                else if( cmd.equalsIgnoreCase( NowAddressCore.ALL_DATA ) == true ){
                    ret = NowAddressCore.CODE_ALL_DATA ;
                    break ;
                }
                
            }
            
        }
        
        cmd = null ;
        return ret ;
    }
    
    
}

