/*
 * @(#)RollID.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.util.Arrays;

/**
 * IDロール管理.
 * <BR><BR>
 * IDロール管理オブジェクト.
 *
 * @version 1.00, 2005/06/02
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class RollID
{
    /**
     * デフォルト値.
     */
    private static final int DEF = 5 ;
    
    /**
     * 最大管理長.
     */
    private int m_max = -1 ;
    
    /**
     * ロール管理.
     */
    private int[] m_roll = null ;
    
    /**
     * 現在ポイント.
     */
    private int m_pnt = -1 ;
    
    /**
     * 最終受信時間.
     */
    private long m_lastTime = -1L ;
    
    /**
     * 現在取得ID.
     */
    private int m_nowID = -1 ;
    
    /**
     * 前回取得ID.
     */
    private int m_beforeID = -1 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public RollID()
    {
        this( -1 ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大管理長を設定してオブジェクトを生成します.
     * <BR>
     * @param max 最大管理長を設定します.
     */
    public RollID( int max )
    {
        if( max <= DEF ){
            max = DEF ;
        }
        
        m_max = max ;
        m_roll = new int[ max ] ;
        m_pnt = -1 ;
        m_nowID = -1 ;
        m_beforeID = -1 ;
        
        Arrays.fill( m_roll,( int )-1 ) ;
        
        m_lastTime = System.currentTimeMillis() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリア.
     */
    public final void clear()
    {
        m_roll = null ;
        m_max = -1 ;
        m_pnt = -1 ;
        m_lastTime = -1L ;
        m_nowID = -1 ;
        m_beforeID = -1 ;
    }
    
    /**
     * ID情報追加.
     * <BR><BR>
     * ID情報を追加します.<BR>
     * この処理の場合、メソッド[getID]や[getBeforeID]から取得する
     * 条件が変更されません.<BR>
     * そのため、受信されたIDとして設定する必要があるが、最新の
     * 状態を変更したくない場合に利用します.
     * <BR>
     * @param id 追加対象のIDを設定します.
     */
    public final void addID( int id )
    {
        
        m_pnt ++ ;
        
        if( m_pnt >= m_max ){
            m_pnt = 0 ;
        }
        
        m_roll[ m_pnt ] = id ;
        m_lastTime = System.currentTimeMillis() ;
        
    }
    
    /**
     * ID情報設定.
     * <BR><BR>
     * ID情報を設定します.<BR>
     * この処理の場合、メソッド[getID]や[getBeforeID]から取得する
     * 条件が変更されます.<BR>
     * 通常このメソッドで処理する場合、正常な流れで処理が成功した場合に
     * 呼び出します.例えばUDPなど、確実に到達しない場合の通信プロトコルを
     * 利用している場合、送信したデータが正常に到達したときなどで呼び出します.<BR>
     * そのため正常に到達しなくて、際装要求などで受信された場合は[addID]で
     * 処理する必要があります.
     * <BR>
     * @param id 対象のIDを設定します.
     */
    public final void setID( int id )
    {
        
        m_pnt ++ ;
        
        if( m_pnt >= m_max ){
            m_pnt = 0 ;
        }
        
        m_beforeID = m_nowID ;
        m_nowID = id ;
        m_roll[ m_pnt ] = id ;
        m_lastTime = System.currentTimeMillis() ;
        
    }
    
    /**
     * ID情報取得.
     * <BR><BR>
     * 現在のID情報を取得します.
     * <BR>
     * @return int 現在のID情報が返されます.
     */
    public final int getID()
    {
        return m_nowID ;
    }
    
    /**
     * 前回設定IDを取得.
     * <BR><BR>
     * 前回設定されたID情報を取得します.
     * <BR>
     * @return int 前回設定されたID情報を取得します.
     */
    public final int getBeforeID()
    {
        return m_beforeID ;
    }
    
    /**
     * 最終更新時間を取得.
     * <BR><BR>
     * 最終更新時間を取得します.
     * <BR>
     * @return long 最終更新時間が返されます.
     */
    public final long getLastTime()
    {
        return m_lastTime ;
    }
    
    /**
     * 対象のIDがIDロール内に含まれているかチェック.
     * <BR><BR>
     * 対象のIDがIDロール内に含まれているかチェックします.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、ロール内に存在します.<BR>
     *                 [false]が返された場合、ロール内に存在しません.
     */
    public final boolean isID( int id )
    {
        int i ;
        int len ;
        int[] roll = null ;
        
        len = m_max ;
        roll = m_roll ;
        
        for( i = 0 ; i < len ; i ++ ){
            if( id == roll[ i ] ){
                return true ;
            }
        }
        
        return false ;
    }
    
}
