/*
 * @(#)TcpMonServerThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.SettingException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;



/**
 * TCP/IPサーバ管理スレッド.
 * <BR><BR>
 * 対象のTCP/IPデータ処理を管理します.
 *
 * @version 1.00, 2004/10/05
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class TcpMonServerThread extends ExecutionThread
{
    
    /**
     * Acceptタイムアウト.
     */
    private static final int TIMEOUT_ACCEPT = 1500 ;
    
    /**
     * タイム存在時の追加値.
     */
    private static final long ADD_NEXT_TIME = 5 ;
    
    /**
     * 接続数 : デフォルト.
     */
    private static final int DEF_BACK_LOG = 50 ;
    
    
    /**
     * スレッドタイプ : Accept用.
     */
    private static final int TYPE_ACCEPT = 0x00000001 ;
    
    /**
     * スレッドタイプ : 管理用.
     */
    private static final int TYPE_MANAGER = 0x00000002 ;
    
    
    /**
     * 要素タイムアウト : タイムアウトなし.
     */
    private static final long NOT_TIMEOUT = 0L ;
    
    /**
     * 要素タイムアウト : デフォルト値.
     * 60sec -> 1分.
     */
    private static final long DEF_TIMEOUT = 60000L ;
    
    /**
     * 要素タイムアウト : 最小値.
     * 30sec.
     */
    public static final long MIN_TIMEOUT = 30000L ;
    
    /**
     * 要素タイムアウト : 最小値.
     * 3600sec -> 1時間.
     */
    public static final long MAX_TIMEOUT = 3600000L ;
    
    
    
    
    /**
     * サーバソケットオブジェクト.
     */
    private ServerSocket m_server = null ;
    
    /**
     * 要素利用タイムアウト値.
     */
    private long m_timeout = 0L ;
    
    /**
     * 接続数.
     */
    private int m_connLength = 0 ;
    
    /**
     * 送受信バッファ長.
     */
    private int m_bufLen = -1 ;
    
    /**
     * TCPサーバ要素管理.
     */
    private final ObjectArray m_table = new ObjectArray() ;
    
    /**
     * 新規要素格納オブジェクト.
     */
    private final ObjectArray m_newConnect = new ObjectArray() ;
    
    /**
     * Accept用.
     */
    private final LoopThread m_threadAC = new LoopThread() ;
    
    /**
     * 管理用.
     */
    private final LoopThread m_threadMan = new LoopThread() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    private final Synchronized m_syncAC = new Synchronized() ;
    private final Synchronized m_syncMan = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public TcpMonServerThread()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port )
        throws InputException,AccessException
    {
        this.open( port,TcpMonServerThread.DEF_BACK_LOG,TcpMonServerThread.DEF_TIMEOUT ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param timeout 要素タイムアウト値を設定します.<BR>
     *                設定可能な最小値は[30000L]です.<BR>
     *                設定可能な最大値は[3600000L]です.<BR>
     *                また、タイムアウト指定なしの場合[0L]を指定します.<BR>
     *                また、単位はミリ秒です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,long timeout )
        throws InputException,AccessException
    {
        this.open( port,TcpMonServerThread.DEF_BACK_LOG,timeout ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param backlog 接続最大数を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,int backlog )
        throws InputException,AccessException
    {
        this.open( port,backlog,TcpMonServerThread.DEF_TIMEOUT ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param backlog 接続最大数を設定します.
     * @param timeout 要素タイムアウト値を設定します.<BR>
     *                設定可能な最小値は[30000L]です.<BR>
     *                設定可能な最大値は[3600000L]です.<BR>
     *                また、タイムアウト指定なしの場合[0L]を指定します.<BR>
     *                また、単位はミリ秒です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,int backlog,long timeout )
        throws InputException,AccessException
    {
        
        if(
            timeout != TcpMonServerThread.NOT_TIMEOUT &&
            (
                timeout < TcpMonServerThread.MIN_TIMEOUT ||
                timeout > TcpMonServerThread.MAX_TIMEOUT
            ) ||
            port < 0 || port > 65535 || backlog <= 0
        )
        {
            throw new  InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync.create() ;
        m_syncAC.create() ;
        m_syncMan.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_server = new ServerSocket( port,backlog ) ;
                this.openTo( timeout,backlog ) ;
                
            }
            
        }catch( IOException io ){
            this.close() ;
            throw new AccessException( io ) ;
        }catch( AccessException ac ){
            this.close() ;
            throw ac ;
        }
        
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param addr バインド先のアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,InetAddress addr )
        throws InputException,AccessException
    {
        this.open( port,TcpMonServerThread.DEF_BACK_LOG,TcpMonServerThread.DEF_TIMEOUT,addr ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param timeout 要素タイムアウト値を設定します.<BR>
     *                設定可能な最小値は[30000L]です.<BR>
     *                設定可能な最大値は[3600000L]です.<BR>
     *                また、タイムアウト指定なしの場合[0L]を指定します.<BR>
     *                また、単位はミリ秒です.
     * @param addr バインド先のアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,long timeout,InetAddress addr )
        throws InputException,AccessException
    {
        this.open( port,TcpMonServerThread.DEF_BACK_LOG,timeout,addr ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param backlog 接続最大数を設定します.
     * @param addr バインド先のアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,int backlog,InetAddress addr )
        throws InputException,AccessException
    {
        this.open( port,backlog,TcpMonServerThread.DEF_TIMEOUT,addr ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * サーバオープンを行います.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param backlog 接続最大数を設定します.
     * @param timeout 要素タイムアウト値を設定します.<BR>
     *                設定可能な最小値は[30000L]です.<BR>
     *                設定可能な最大値は[3600000L]です.<BR>
     *                また、タイムアウト指定なしの場合[0L]を指定します.<BR>
     *                また、単位はミリ秒です.
     * @param addr バインド先のアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( int port,int backlog,long timeout,InetAddress addr )
        throws InputException,AccessException
    {
        
        if(
            timeout != TcpMonServerThread.NOT_TIMEOUT &&
            (
                timeout < TcpMonServerThread.MIN_TIMEOUT ||
                timeout > TcpMonServerThread.MAX_TIMEOUT
            ) ||
            port < 0 || port > 65535 || addr == null || backlog <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync.create() ;
        m_syncAC.create() ;
        m_syncMan.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_server = new ServerSocket( port,backlog,addr ) ;
                this.openTo( timeout,backlog ) ;
                
            }
        }catch( IOException io ){
            this.close() ;
            throw new AccessException( io ) ;
        }catch( AccessException ac ){
            this.close() ;
            throw ac ;
        }
        
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を実施します.
     */
    public final void close()
    {
        int i ;
        int len ;
        
        ObjectArray tbl = null ;
        
        m_sync.clear() ;
        m_syncAC.clear() ;
        m_syncMan.clear() ;
        
        try{
            m_server.close() ;
        }catch( Exception t ){
        }finally{
            m_server = null ;
        }
        
        m_threadAC.clear() ;
        m_threadMan.clear() ;
        
        try{
            
            tbl = m_table ;
            len = tbl.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                ( ( TcpProtocol )tbl.get( i ) ).disconnect() ;
            }
            
        }catch( Exception t ){
        }finally{
            tbl = null ;
        }
        
        m_newConnect.clear() ;
        m_table.clear() ;
        m_server = null ;
        m_timeout = 0L ;
        m_connLength = 0 ;
        
    }
    
    /**
     * 要素タイムアウト値を設定.
     * <BR><BR>
     * サーバコネクションによる各要素のタイムアウト値を設定します.
     * <BR>
     * @param timeout 要素タイムアウト値を設定します.<BR>
     *                設定可能な最小値は[30000L]です.<BR>
     *                設定可能な最大値は[3600000L]です.<BR>
     *                また、タイムアウト指定なしの場合[0L]を指定します.<BR>
     *                また、単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void setTimeout( long timeout )
        throws InputException
    {
        if(
            timeout != TcpMonServerThread.NOT_TIMEOUT &&
            (
                timeout < TcpMonServerThread.MIN_TIMEOUT ||
                timeout > TcpMonServerThread.MAX_TIMEOUT
            )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_timeout = timeout ;
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 送受信バッファ長を設定.
     * <BR><BR>
     * 送受信バッファ長を設定します.
     * <BR>
     * @param bufLen 送受信送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     */
    public final void setBuffer( int bufLen )
        throws InputException
    {
        if( bufLen <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_bufLen = bufLen ;
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * コネクション群を取得.
     * <BR><BR>
     * コネクション群を取得します.
     * <BR>
     * @param val コネクション群が返されます.
     * @return int 取得されたコネクション数が返されます.
     */
    public final int getAccept( ObjectArray val )
    {
        int i ;
        int len ;
        
        Object[] objs = null ;
        
        len = 0 ;
        
        if( val != null ){
            
            val.clear() ;
            
            try{
                
                synchronized( m_sync.get() ){
                    
                    len = m_newConnect.size() ;
                    objs = m_newConnect.getObjects() ;
                    m_newConnect.clear() ;
                    
                }
                
                for( i = 0 ; i < len ; i ++ ){
                    val.add( objs[ i ] ) ;
                    objs[ i ] = null ;
                }
                
            }catch( Exception t ){
                len = 0 ;
                val.clear() ;
            }finally{
                objs = null ;
            }
            
        }
        
        return len ;
    }
    
    /**
     * 要素タイムアウト値を取得.
     * <BR><BR>
     * サーバコネクションによる各要素のタイムアウト値が返されます.
     * <BR>
     * @return long タイムアウト値が返されます.
     */
    public final long getTimeout()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_timeout ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * サーバ同時接続数を取得.
     * <BR><BR>
     * サーバ同時接続数を取得します.
     * <BR>
     * @return int サーバ同時接続数が返されます.
     */
    public final int getMaxConnect()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_connLength ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 送受信バッファ長値を取得.
     * <BR><BR>
     * 設定されている送受信バッファ長値が返されます.
     * <BR>
     * @return int 対象の送受信バッファ長が返されます.
     */
    public final int getBuffer()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_bufLen ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * 対象のバインドアドレスを取得します.
     * <BR>
     * @param addr 対象のバインドアドレスが返されます.
     */
    public final void getBindAddress( ConnectAddress addr )
    {
        try{
            synchronized( m_sync.get() ){
                if( addr != null ){
                    addr.create(
                        m_server.getInetAddress(),
                        m_server.getLocalPort()
                    ) ;
                }
            }
        }catch( Exception t ){
            if( addr != null ){
                
                try{
                    addr.clear() ;
                    addr.create( NetDef.NOT_ADDR,NetDef.PORT_MIN ) ;
                }catch( Exception tt ){
                }
                
            }
        }
    }
    
    /**
     * サーバコネクション要素が存在するかチェック.
     * <BR><BR>
     * サーバコネクション要素が存在するかチェックします.
     * <BR>
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、新しい要素は存在します.<BR>
     *                 [false]が返された場合、新しい要素は存在しません.
     */
    public final boolean isAccept()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_newConnect.size() != 0 ){
                    ret = true ;
                }
                else{
                    ret = false ;
                }
                
            }
        }catch( Exception t ){
            this.close() ;
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 接続チェック.
     * <BR><BR>
     * 接続されているかチェックします.
     * <BR>
     * @return boolean 接続状況が返されます.<BR>
     *                 [true]が返された場合、接続されています.
     *                 [false]が返された場合、接続されていません.
     */
    public final boolean isConnect()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                // エラーチェック.
                m_server.getSoTimeout() ;
                
                // エラーが検知されない場合[true]を設定.
                ret = true ;
                
            }
        }catch( Exception t ){
            this.close() ;
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        if( obj == null || ( obj instanceof Integer ) == false ){
            throw new ExecutionException( "パラメータが不正です",ExecutionException.LEVEL_STOP ) ;
        }
        
        try{
            
            // スレッド待機.
            UtilCom.idleTime() ;
            
            // スレッド処理チェック.
            switch( ( ( Integer )obj ).intValue() ){
                
                // accept待ちスレッド.
                case TcpMonServerThread.TYPE_ACCEPT :
                    
                    this.execAccept() ;
                    break ;
                    
                // 通信管理スレッド.
                case TcpMonServerThread.TYPE_MANAGER :
                    
                    this.execManager() ;
                    break ;
                    
                default :
                    throw new ExecutionException( "パラメータが不正です",ExecutionException.LEVEL_STOP ) ;
            }
            
        }catch( ExecutionException ee ){
            throw ee ;
        }
    }
    
    
    
    /**
     * オープン共通.
     */
    private final void openTo( long timeout,int backlog )
        throws AccessException
    {
        try{
            
            m_timeout = timeout ;
            m_connLength = backlog ;
            m_threadAC.create( this,new Integer( TYPE_ACCEPT ) ) ;
            m_threadMan.create( this,new Integer( TYPE_MANAGER ) ) ;
            
            m_server.setSoTimeout( TcpMonServerThread.TIMEOUT_ACCEPT ) ;
            
            m_threadAC.startThread() ;
            m_threadMan.startThread() ;
            
        }catch( SocketException soc ){
            throw new AccessException( soc ) ;
        }catch( InputException in ){
            throw new AccessException( in ) ;
        }catch( SettingException st ){
            throw new AccessException( st ) ;
        }
        
    }
    
    /**
     * Accept待ち.
     */
    private final void execAccept()
        throws ExecutionException
    {
        long time ;
        Socket sc = null ;
        TcpProtocol net = null ;
        ObjectArray tbl = null ;
        
        try{
            
            synchronized( m_syncAC.get() ){
                
                // サーバコネクション待機.
                sc = m_server.accept() ;
                
                // サーバコネクション検知.
                if( sc != null ){
                    
                    // サーバコネクション要素を生成.
                    net = new TcpProtocol( sc,m_bufLen ) ;
                    
                    // 新要素を追加.
                    synchronized( m_sync.get() ){
                        
                        // 要素管理テーブルを取得.
                        tbl = m_table ;
                        
                        // 新規Accept要素に追加.
                        m_newConnect.add( net ) ;
                        
                        // 管理Accept要素に追加.
                        tbl.add( net ) ;
                        
                    }
                    
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( InterruptedIOException ii ){
        }catch( IOException io ){
        }catch( InputException in ){
        }finally{
            sc = null ;
            net = null ;
            tbl = null ;
        }
    }
    
    /**
     * 通信要素管理.
     */
    private final void execManager()
        throws ExecutionException
    {
        int i ;
        int len ;
        
        long timeout ;
        
        ObjectArray tbl = null ;
        TcpProtocol net = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                tbl = m_table ;
                len = tbl.size() ;
                
            }
            
            synchronized( m_syncMan.get() ){
                
                for( i = len-1 ; i >= 0 ; i -- ){
                    
                    try{
                        
                        synchronized( m_sync.get() ){
                            
                            net = ( TcpProtocol )tbl.get( i ) ;
                            timeout = m_timeout ;
                            
                        }
                        
                        // 対象TCP/IPオブジェクトが存在する場合.
                        if( net != null ){
                            
                            // 対象TCP/IPが既にクローズされている場合.
                            // 対象TCP/IPの利用時間がタイムアウトの場合.
                            if(
                                net.isConnect() == false ||
                                System.currentTimeMillis() >= ( net.getLastTime() + timeout )
                            )
                            {
                                
                                // 対象ネットワークをクローズ.
                                net.disconnect() ;
                                
                                // テーブルから削除.
                                synchronized( m_sync.get() ){
                                    tbl.remove( i ) ;
                                }
                                
                                // スレッド待機.
                                UtilCom.idleTime() ;
                                
                            }
                            
                            
                        }
                        
                    }catch( NullPointerException nul ){
                        throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
                    }catch( ExecutionException ee ){
                        throw ee ;
                    }catch( Exception tt ){
                    }finally{
                        net = null ;
                    }
                    
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( Exception t ){
        }finally{
            tbl = null ;
            net = null ;
        }
    }
    
}

