/*
 * @(#)MgcManagerImple.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.mgc ;

import java.io.Serializable;
import java.net.InetAddress;
import java.util.Arrays;

import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;


/**
 * マルチキャストグループ管理内部オブジェクト.
 * <BR><BR>
 * マルチキャストグループを管理する内部オブジェクトです.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcManagerImple implements Serializable,MgcManager,MgcWriteManager
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            MgcManagerImple.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * サーバ名によるグループ管理テーブル.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * シリアライズ先.
     */
    private String m_serializeName = null ;
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * オブジェクトを生成します.
     */
    public MgcManagerImple()
    {
        m_serializeName = null ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * オブジェクトを生成します.
     * <BR>
     * @param serializeName シリアライズ先を設定します.<BR>
     *                      [null]を設定した場合、シリアライズ先を指定しません.
     */
    public MgcManagerImple( String serializeName )
    {
        m_serializeName = serializeName ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 格納されている全ての条件を破棄します.
     */
    public final void clear()
    {
        int i ;
        int len ;
        
        String[] names = null ;
        CharTable tbl = null ;
        ObjectArray obj = null ;
        
        synchronized( this ){
            
            tbl = m_table ;
            names = tbl.getNames() ;
            
            if( names != null ){
                len = names.length ;
                for( i = 0 ; i < len ; i ++ ){
                    
                    if( ( obj = ( ObjectArray )tbl.get( names[ i ] ) ) != null ){
                        obj.clear() ;
                    }
                    
                    obj = null ;
                    names[ i ] = null ;
                    
                }
            }
            
            tbl.clear() ;
            names = null ;
            
        }
        
        System.gc() ;
        
    }
    
    /**
     * 管理情報を更新.
     * <BR><BR>
     * 対象の情報を更新します.<BR>
     * また、対象条件がテーブルに存在しない場合は追加となります.
     * <BR>
     * @param val 更新対象の情報を設定します.
     * @param boolean 新たに追加された場合の条件を返します.<BR>
     *                [true]が返された場合、新たに追加されました.<BR>
     *                [false]が返された場合、条件を更新したか処理に失敗しました.
     */
    public final boolean renew( MgcValue val )
    {
        boolean ret ;
        MgcServerValue sval = null ;
        
        ret = false ;
        
        if( val != null && val.isUse() == true ){
            
            try{
                
                sval = new MgcServerValue(
                    val.getServerName(),
                    val.getServerID(),
                    val.getProtocolType(),
                    val.getBindAddress(),
                    val.getBindPort(),
                    val.getConnectCount(),
                    val.getDefaultTime()
                ) ;
                
                ret = this.renew( sval ) ;
                
                if( sval != null ){
                    sval.clear() ;
                    sval = null ;
                }
                
            }catch( Exception e ){
                ret = false ;
            }finally{
                if( sval != null ){
                    sval.clear() ;
                    sval = null ;
                }
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 管理情報を更新.
     * <BR><BR>
     * 対象の情報を更新します.<BR>
     * また、対象条件がテーブルに存在しない場合は追加となります.
     * <BR>
     * @param val 更新対象の情報を設定します.
     * @param boolean 新たに追加された場合の条件を返します.<BR>
     *                [true]が返された場合、新たに追加されました.<BR>
     *                [false]が返された場合、条件を更新したか処理に失敗しました.
     */
    public final boolean renew( MgcServerValue val )
    {
        int i ;
        int len ;
        int objLen ;
        int id ;
        int port ;
        boolean flg ;
        boolean ret ;
        
        String name = null ;
        InetAddress addr = null ;
        CharTable tbl = null ;
        ObjectArray obj = null ;
        MgcValueImple mv = null ;
        Object[] objs = null ;
        
        if( val == null || val.isUse() == false ){
            return false ;
        }
        
        ret = false ;
        name = val.getServerName() ;
        id = val.getServerID() ;
        addr = val.getBindAddress() ;
        port = val.getBindPort() ;
        
        try{
            synchronized( this ){
                
                tbl = m_table ;
                
                // 対象サーバ名がテーブルに存在しない場合.
                if( ( obj = ( ObjectArray )tbl.get( name ) ) == null ){
                    
                    // サーバ名に対する条件をセット.
                    obj = new ObjectArray() ;
                    tbl.add( name,obj ) ;
                    
                }
                
                // 対象要素と同一の条件が存在する場合.
                len = obj.size() ;
                for( i = 0,flg = false ; i < len ; i ++ ){
                    
                    mv = ( MgcValueImple )obj.get( i ) ;
                    
                    // 同一条件が見つかった場合.
                    if(
                        port == mv.getBindPort() &&
                        addr.equals( mv.getBindAddress() ) == true
                    )
                    {
                        // 更新する.
                        mv.setDefaultTime( val.getTime() ) ;
                        mv.update() ;
                        flg = true ;
                        break ;
                        
                    }
                    
                }
                
                // 同一条件が見つからない場合.
                if( flg == false ){
                    
                    // 新しく追加.
                    mv = new MgcValueImple(
                        name,id,val.getProtocolType(),addr,port
                    ) ;
                    
                    // 標準時間をセット.
                    mv.setDefaultTime( val.getTime() ) ;
                    
                    obj.add( mv ) ;
                    ret = true ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            name = null ;
            addr = null ;
            tbl = null ;
            obj = null ;
            mv = null ;
            objs = null ;
        }
        
        return ret ;
    }
    
    /**
     * 管理情報から削除.
     * <BR><BR>
     * 管理情報から削除します.
     * <BR>
     * @param val 削除対象の要素を設定します.
     * @return boolean 削除処理が成功条件が返されます.<BR>
     *                 [true]が返された場合成功しました.<BR>
     *                 [false]が返された場合存在しないか、失敗しました.
     */
    public final boolean remove( MgcValue val )
    {
        boolean ret ;
        MgcServerValue sval = null ;
        
        ret = false ;
        
        if( val != null && val.isUse() == true ){
            
            try{
                
                sval = new MgcServerValue(
                    val.getServerName(),
                    val.getServerID(),
                    val.getProtocolType(),
                    val.getBindAddress(),
                    val.getBindPort(),
                    val.getConnectCount(),
                    val.getDefaultTime()
                ) ;
                
                ret = this.remove( sval ) ;
                
                if( sval != null ){
                    sval.clear() ;
                    sval = null ;
                }
                
            }catch( Exception e ){
                ret = false ;
            }finally{
                if( sval != null ){
                    sval.clear() ;
                    sval = null ;
                }
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 管理情報から削除.
     * <BR><BR>
     * 管理情報から削除します.
     * <BR>
     * @param val 削除対象の要素を設定します.
     * @return boolean 削除処理が成功条件が返されます.<BR>
     *                 [true]が返された場合成功しました.<BR>
     *                 [false]が返された場合存在しないか、失敗しました.
     */
    public final boolean remove( MgcServerValue val )
    {
        int i ;
        int len ;
        int objLen ;
        int id ;
        int port ;
        boolean flg ;
        boolean ret ;
        
        String name = null ;
        InetAddress addr = null ;
        CharTable tbl = null ;
        ObjectArray obj = null ;
        MgcValueImple mv = null ;
        Object[] objs = null ;
        
        if( val == null || val.isUse() == false ){
            return false ;
        }
        
        ret = false ;
        name = val.getServerName() ;
        id = val.getServerID() ;
        addr = val.getBindAddress() ;
        port = val.getBindPort() ;
        
        try{
            synchronized( this ){
                
                tbl = m_table ;
                
                // 対象サーバ名がテーブルに存在する場合.
                if( ( obj = ( ObjectArray )tbl.get( name ) ) != null ){
                    
                    // 対象要素と同一の条件が存在する場合.
                    len = obj.size() ;
                    
                    for( i = 0,flg = false ; i < len ; i ++ ){
                        
                        mv = ( MgcValueImple )obj.get( i ) ;
                        
                        // 同一条件が見つかった場合.
                        if(
                            port == mv.getBindPort() &&
                            addr.equals( mv.getBindAddress() ) == true
                        )
                        {
                            
                            // 削除する.
                            mv.destroy() ;
                            obj.remove( i ) ;
                            flg = true ;
                            ret = true ;
                            
                            break ;
                            
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = false ;
        }finally{
            name = null ;
            addr = null ;
            tbl = null ;
            obj = null ;
            mv = null ;
            objs = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象のサーバ名の情報をソート.
     * <BR><BR>
     * 対象のサーバ名の情報を接続数をキーとしてソートします.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     */
    public final void sort( String name )
    {
        int len ;
        ObjectArray obj = null ;
        
        if( name == null || name.length() <= 0 ){
            return ;
        }
        
        try{
            
            synchronized( this ){
                
                // 条件が存在する場合.
                if( ( obj = ( ObjectArray )m_table.get( name ) ) != null ){
                    
                    // 条件が１つ以上の場合.
                    if( ( len = obj.size() ) > 1 ){
                        
                        // 昇順でソートする.
                        Arrays.sort( obj.getObjects(),0,len ) ;
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
        }finally{
            obj = null ;
        }
    }
    
    /**
     * 対象サーバ名に対するサーバIDを取得.
     * <BR><BR>
     * 対象サーバ名に対するサーバIDを取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int サーバIDが返されます.<BR>
     *             [-1]が返された場合、取得に失敗しました.
     */
    public final int getServerID( String name )
    {
        int ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1 ;
        }
        
        ret = -1 ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = val.getServerID() ;
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対するプロトコルタイプを取得.
     * <BR><BR>
     * 対象サーバ名に対するプロトコルタイプを取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int プロトコルタイプが返されます.<BR>
     *             [-1]が返された場合、取得に失敗しました.
     */
    public final int getProtocolType( String name )
    {
        int ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1 ;
        }
        
        ret = -1 ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = val.getProtocolType() ;
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対するコネクションアドレスを取得.
     * <BR><BR>
     * 対象サーバ名に対するコネクションアドレスを取得します.
     * <BR>
     * @param out 格納対象のコネクションオブジェクトを設定します.
     * @param name 対象のサーバ名を設定します.
     * @return boolean 取得結果が返されます.<BR>
     *                 [true]が返された場合、情報の取得に成功しました.<BR>
     *                 [false]が返された場合、情報の取得に失敗しました.
     */
    public final boolean getConnectAddress( ConnectAddress out,String name )
    {
        boolean ret ;
        MgcValue val = null ;
        
        if( out == null || name == null || name.length() <= 0 ){
            return false ;
        }
        
        ret = false ;
        out.clear() ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    val.getBind( out ) ;
                    ret = true ;
                }
            }
        }catch( Exception e ){
            if( out != null ){
                out.clear() ;
            }
            ret = false ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対する更新時間を取得.
     * <BR><BR>
     * 対象サーバ名に対する更新時間を取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return long 更新時間返されます.<BR>
     *              [-1L]が返された場合、このオブジェクトは無効です.<BR>
     *              [Long.MIN_VALUE]が返された場合、この条件は設定されていません.
     */
    public final long getUpdateTime( String name )
    {
        long ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1L ;
        }
        
        ret = -1L ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = val.getUpdateTime() ;
                }
            }
        }catch( Exception e ){
            ret = -1L ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対するコネクション数を取得.
     * <BR><BR>
     * 対象サーバ名に対するコネクション数を取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int コネクション数が返されます.<BR>
     *             [-1]が返された場合、取得に失敗しました.
     */
    public final int getConnectCount( String name )
    {
        int ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1 ;
        }
        
        ret = -1 ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = val.getConnectCount() ;
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対する前回応答時間を取得.
     * <BR><BR>
     * 対象サーバ名に対する前回応答時間を取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int 前回の応答時間が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.<BR>
     *             [Integer.MIN_VALUE]が返された場合、この条件は設定されていません.
     */
    public final int getBeforeResponse( String name )
    {
        int ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1 ;
        }
        
        ret = -1 ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = val.getBeforeResponse() ;
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対する応答なしカウントを取得.
     * <BR><BR>
     * 対象サーバ名に対する応答なしカウントを取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int 応答なしカウントが返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    protected final int getNoResponseCount( String name )
    {
        int ret ;
        MgcValue val = null ;
        
        if( name == null || name.length() <= 0 ){
            return -1 ;
        }
        
        ret = -1 ;
        
        try{
            synchronized( this ){
                val = this.getMgcValue( name ) ;
                if( val != null ){
                    ret = ( ( MgcValueImple )val ).getNoResponseCount() ;
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            val = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名の条件を取得.
     * <BR><BR>
     * 対象サーバ名の条件を取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return MgcValue[] 対象の要素群が返されます.<BR>
     *                    [null]が返された場合、対象のサーバ名の
     *                    要素は存在しません.
     */
    public final MgcValue[] get( String name )
    {
        int len ;
        ObjectArray obj = null ;
        MgcValue[] ret = null ;
        
        if( name == null || name.length() <= 0 ){
            return null ;
        }
        
        try{
            
            synchronized( this ){
                
                if( ( obj = ( ObjectArray )m_table.get( name ) ) != null ){
                    
                    if( ( len = obj.size() ) > 0 ){
                        ret = new MgcValue[ len ] ;
                        System.arraycopy( obj.getObjects(),0,ret,0,len ) ;
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            obj = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象サーバ名に対する要素数を取得.
     * <BR><BR>
     * 対象サーバ名に対する要素数を取得します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @return int 要素数が返されます.
     */
    public final int getSize( String name )
    {
        int ret ;
        ObjectArray obj = null ;
        
        if( name == null || name.length() <= 0 ){
            return 0 ;
        }
        
        ret = 0 ;
        
        try{
            
            synchronized( this ){
                
                if( ( obj = ( ObjectArray )m_table.get( name ) ) != null ){
                    ret = obj.size() ;
                }
                
            }
            
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            obj = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 全ての条件を取得.
     * <BR><BR>
     * 全ての条件を取得します.
     * <BR>
     * @return MgcValue[] 対象の要素群が返されます.<BR>
     *                    [null]が返された場合要素は存在しません.
     */
    public final MgcValue[] getAll()
    {
        int i ;
        int len ;
        int objLen ;
        int allLen ;
        int pnt ;
        
        CharTable tbl = null ;
        ObjectArray obj = null ;
        String[] names = null ;
        MgcValue[] ret = null ;
        
        try{
            
            synchronized( this ){
                
                tbl = m_table ;
                names = tbl.getNames() ;
                
                if( names != null ){
                    
                    len = names.length ;
                    
                    for( i = 0,allLen = 0 ; i < len ; i ++ ){
                        
                        if( ( obj = ( ObjectArray )tbl.get( names[ i ] ) ) != null ){
                            allLen += obj.size() ;
                        }
                        
                    }
                    
                    if( allLen > 0 ){
                        
                        ret = new MgcValue[ allLen ] ;
                        
                        for( i = 0,pnt = 0 ; i < len ; i ++ ){
                        
                            if( ( obj = ( ObjectArray )tbl.get( names[ i ] ) ) != null ){
                                
                                if( ( objLen = obj.size() ) > 0 ){
                                    System.arraycopy( obj.getObjects(),0,ret,pnt,objLen ) ;
                                    pnt += objLen ;
                                }
                                
                            }
                            
                            names[ i ] = null ;
                        
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            tbl = null ;
            obj = null ;
            names = null ;
        }
        
        return ret ;
    }
    
    /**
     * 全ての要素数を取得.
     * <BR><BR>
     * 現在格納されている全ての要素数を取得します.
     * <BR>
     * @return int 全ての要素数が返されます.
     */
    public final int getAllSize()
    {
        int i ;
        int len ;
        int ret ;
        
        CharTable tbl = null ;
        ObjectArray obj = null ;
        String[] names = null ;
        
        ret = 0 ;
        
        try{
            
            synchronized( this ){
                
                tbl = m_table ;
                names = tbl.getNames() ;
                
                if( names != null ){
                    
                    len = names.length ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        if( ( obj = ( ObjectArray )tbl.get( names[ i ] ) ) != null ){
                            ret += obj.size() ;
                        }
                        
                        names[ i ] = null ;
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            tbl = null ;
            obj = null ;
            names = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のサーバ名一覧を取得.
     * <BR><BR>
     * 対象のサーバ名一覧を取得します.
     * <BR>
     * @return String[] 現在認識されているサーバ名一覧が返されます.<BR>
     *                  [null]が返された場合情報は存在しません.
     */
    public final String[] getServerNames()
    {
        String[] ret = null ;
        
        synchronized( this ){
            ret = m_table.getNames() ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のサーバ名数を取得.
     * <BR><BR>
     * 対象のサーバ名数を取得します.
     * <BR>
     * @return int 現在認識されているサーバ名数が返されます.
     */
    public final int getServerSize()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_table.size() ;
        }
        
        return ret ;
    }
    
    /**
     * シリアライズ先のファイル名を取得.
     * <BR><BR>
     * 設定されているシリアライズ先のファイル名を取得します.
     * <BR>
     * @return String シリアライズ先のファイル名が返されます.<BR>
     *                [null]が返された場合、情報は存在しません.
     */
    public final String getSerializeName()
    {
        String ret = null ;
        
        synchronized( this ){
            ret = m_serializeName ;
        }
        
        return ret ;
    }
    
    /**
     * 対象条件が有効であるかチェック.
     * <BR><BR>
     * 対象の条件が有効であるかチェックします.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param addr 対象のアドレス情報を設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public final boolean isUseMgc( String name,InetAddress addr,int port )
    {
        int i ;
        int len ;
        boolean ret ;
        
        MgcValue val = null ;
        MgcValue[] vals = null ;
        
        if( name == null || name.length() < 0 || addr == null || port <= 0 || port >= 65535 ){
            return false ;
        }
        
        ret = false ;
        
        try{
            synchronized( this ){
                
                vals = this.get( name ) ;
                
                if( vals != null ){
                    
                    len = vals.length ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        val = vals[ i ] ;
                        vals[ i ] = null ;
                        
                        if(
                            val != null && val.isUse() == true &&
                            val.getBindPort() == port &&
                            val.getBindAddress().equals( addr ) == true
                        )
                        {
                            ret = true ;
                            break ;
                        }
                        
                    }
                }
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            val = null ;
            vals = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * 対象サーバ名から、最適な条件を取得.
     */
    private final MgcValue getMgcValue( String name )
    {
        int i ;
        int cnt ;
        int len ;
        int rnd ;
        int pnt ;
        
        MgcValue[] tmp = null ;
        MgcValue ret = null ;
        
        try{
            
            // サーバ名に対する接続情報群を取得.
            tmp = this.get( name ) ;
            
            // 条件が存在する場合.
            if( tmp != null ){
                
                len = tmp.length ;
                
                // 情報が存在しない場合.
                if( len <= 0 ){
                    
                    // 情報が存在しない場合、
                    // 条件は取得できないから、
                    // 処理を抜ける.
                    return null ;
                    
                }
                // １つのサーバ名に対して２つ未満の
                // サーバが存在する場合.
                else if( len <= 2 ){
                    
                    // 先頭の条件を対象とする.
                    ret = tmp[ 0 ] ;
                    
                    // 取得した条件が不正な場合.
                    if( ret != null && ret.isUse() == false ){
                        ret = null ;
                    }
                    
                }
                // １つのサーバ名に対して２つ以上の
                // サーバが存在する場合.
                else{
                    
                    rnd = ( len / 4 ) + 1 ;
                    
                    // 規定範囲までループ.
                    for( cnt = 0 ;; cnt ++ ){
                        
                        // ランダムで選択.
                        pnt = UtilCom.random( rnd ) ;
                        ret = tmp[ pnt ] ;
                        
                        // 選択した条件が取得できた場合.
                        if( ret != null && ret.isUse() == true ){
                            break ;
                        }
                        // 選択条件の取得が、サーバ名数ループしても
                        // 取得出来ない場合.
                        else if( cnt >= len ){
                            ret = null ;
                            break ;
                        }
                        
                    }
                    
                }
                
                // 対象情報が存在しない場合.
                // 接続可能な条件を探す.
                if( ret == null ){
                    
                    // 先頭から取得処理を行う.
                    for( i = 0 ; i < len ; i ++ ){
                        
                        // 取得に成功した場合は、それを対象にして処理を抜ける
                        if( tmp[ i ] != null && tmp[ i ].isUse() == true ){
                            ret = tmp[ i ] ;
                            break ;
                        }
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
}

