/*
 * @(#)MgcServer.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.mgc ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.BaseMultiCast;
import org.maachang.commons.net.MultiCastV4;
import org.maachang.commons.thread.Synchronized;


/**
 * MGC-Serverオブジェクト.
 * <BR><BR>
 * このオブジェクトは、クライアント又は各サーバ通信の補助的存在を担います.<BR>
 * 役割として、同一カテゴリの存在を把握する機能をもっており、接続状況や
 * 切断情報を管理します.<BR>
 * そのため実際の電文交換を行う場合には、別途プロトコルを設置する必要があります.
 *  
 * @version 1.0.0 2005/08/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class MgcServer
{
    
    /**
     * マルチキャストオブジェクト.
     */
    private BaseMultiCast m_mcast = null ;
    
    /**
     * 自分のマシンステータス要素.
     */
    private MgcValueImple m_thisValue = null ;
    
    /**
     * 接続管理情報.
     */
    private final MgcManagerImple m_man = new MgcManagerImple() ;
    
    /**
     * Mgcサーバ受信スレッド.
     */
    private MgcSvRcvThread m_rcvThread = null ;
    
    /**
     * Mgcサーバ管理スレッド.
     */
    private MgcSvThread m_thread = null ;
    
    /**
     * 通信テーブル.
     */
    private final MgcTable m_connTable = new MgcTable() ;
    
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public MgcServer()
    {
        m_sync.clear() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,int port )
        throws InputException
    {
        this.open( mcast,id,name,type,null,port,null,-1 ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 主となる通信プロトコルのバインドアドレスを設定します.<BR>
     *             [null]を設定した場合、デフォルトのバインド先が選択されます.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,InetAddress addr,int port )
        throws InputException
    {
        this.open( mcast,id,name,type,addr,port,null,-1 ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,int port,int time )
        throws InputException
    {
        this.open( mcast,id,name,type,null,port,null,time ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @param table 通信に対する暗号表を設定します.<BR>
     *              [null]を設定した場合、デフォルトの暗号表を利用します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,int port,byte[] table,int time )
        throws InputException
    {
        this.open( mcast,id,name,type,null,port,table,time ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 主となる通信プロトコルのバインドアドレスを設定します.<BR>
     *             [null]を設定した場合、デフォルトのバインド先が選択されます.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,InetAddress addr,int port,int time )
        throws InputException
    {
        this.open( mcast,id,name,type,addr,port,null,time ) ;
    }
    
    /**
     * サーバオープン.
     * <BR><BR>
     * 対象のサーバをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param name 対象のサーバ名を設定します.
     * @param type 主となる通信プロトコルを設定します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 主となる通信プロトコルのバインドアドレスを設定します.<BR>
     *             [null]を設定した場合、デフォルトのバインド先が選択されます.
     * @param port 主となる通信プロトコルのバインドポートを設定します.
     * @param table 通信に対する暗号表を設定します.<BR>
     *              [null]を設定した場合、デフォルトの暗号表を利用します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,String name,int type,InetAddress addr,int port,byte[] table,int time )
        throws InputException
    {
        
        MgcValueImple val = null ;
        
        if(
            mcast == null || mcast.isOpen() == false ||
            ( table != null && table.length != MgcCommon.MAX_RAND_CODE )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        m_sync.create() ;
        
        try{
            
            val = new MgcValueImple( name,id,type,addr,port ) ;
            ( ( MultiCastV4 )mcast ).joinGroup( id ) ;
            
            synchronized( m_sync.get() ){
                
                m_mcast = mcast ;
                m_thisValue = val ;
                
                m_thread = new MgcSvThread( val,m_man,m_connTable,table,time ) ;
                m_rcvThread = new MgcSvRcvThread( val,mcast,m_connTable,table,time ) ;
                
            }
            
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
            throw new InputException( e ) ;
        }finally{
            val = null ;
        }
        
    }
    
    /**
     * サーバクローズ.
     * <BR><BR>
     * サーバをクローズします.
     */
    public final void close()
    {
        try{
            synchronized( m_sync.get() ){
                
                // 終了電文が送信できる場合は送信する.
                if( m_rcvThread.isUse() == true ){
                    try{
                        m_rcvThread.sendMgcServerEnd(
                            MgcCommon.SEND_MGC_SERVER_END_COUNT
                        ) ;
                    }catch( Exception ee ){
                    }
                }
                
                try{
                    m_rcvThread.destroy() ;
                }catch( Exception ee ){
                }
                
                try{
                    m_thread.destroy() ;
                }catch( Exception ee ){
                }
                
                try{
                    m_mcast.leaveGroup() ;
                }catch( Exception ee ){
                }
                
                if( m_thisValue != null ){
                    m_thisValue.destroy() ;
                }
                m_man.clear() ;
                m_connTable.clear() ;
                
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        m_mcast = null ;
        m_thisValue = null ;
        m_rcvThread = null ;
        m_thread = null ;
        
    }
    
    /**
     * 管理しているこのマシンの要素を取得.
     * <BR><BR>
     * 管理しているこのマシンの要素を取得します.
     * <BR>
     * @return MgcValue 管理しているこのマシンの要素が返されます.<BR>
     *                  [null]が返された場合、このオブジェクトはクローズされています.
     */
    public final MgcValue getThisValue()
    {
        MgcValue ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thisValue ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * マネージャオブジェクトを取得.
     * <BR><BR>
     * マネージャオブジェクトを取得します.
     * <BR>
     * @return MgcManager マネージャオブジェクトが返されます.
     *                    [null]が返された場合、このオブジェクトはクローズされています.
     */
    public final MgcManager getManager()
    {
        MgcManager ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_man ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されている暗号表を取得.
     * <BR><BR>
     * 設定されている暗号表を取得します.
     * <BR>
     * @return byte[] 設定されている暗号表が返されます.
     *                [null]が返された場合、デフォルト値が設定されているか
     *                オブジェクトがクローズされています.
     */
    public final byte[] getCodeTable()
    {
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getCodeTable() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 接続状況監視時間を取得.
     * <BR><BR>
     * 設定されている接続状況を監視する時間を取得します.
     * <BR>
     * @return int 接続状況監視時間が返されます.<BR>
     *             [-1]が返された場合、オブジェクトがクローズされています.
     */
    public final int getCheckConnectTime()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getCheckConnectTime() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * このオブジェクトがオープンされているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オープンされています.<BR>
     *                 [false]が返された場合、クローズされています.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                if(
                    m_thread.isUse() == true &&
                    m_rcvThread.isUse() == true
                )
                {
                    ret = true ;
                }
                else{
                    ret = false ;
                }
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

