/*
 * @(#)MgcSvThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.mgc ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;


/**
 * MGC-Serverスレッド.
 *  
 * @version 1.0.0 2005/07/31
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcSvThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MgcSvRcvThread.class ) ;
    
    /**
     * 再送時間 : デフォルト値.
     * 150sec.
     */
    protected static final int DEF_RESEND = 150000 ;
    
    /**
     * 再送時間 : 最小値.
     * 30sec.
     */
    protected static final int MIN_RESEND = 30000 ;
    
    /**
     * 要素破棄時間 : 最小値.
     * 90sec.
     */
    private static final int MIN_DELETE_TIME = MIN_RESEND ;
    
    /**
     * 応答なしチェックタイミング.
     */
    private static final int DEF_TIMING = 10000 ;
    
    /**
     * 応答なしカウント規定値.
     */
    private static final int MAX_NORESPONSE = 3 ;
    
    
    
    /**
     * このマシンを示す要素.
     */
    private MgcValueImple m_thisValue = null ;
    
    /**
     * コネクション管理テーブル.
     */
    private MgcManagerImple m_man = null ;
    
    /**
     * サーバテーブル.
     */
    private MgcTable m_svTable = null ;
    
    /**
     * 応答なしの時の破棄時間.
     */
    private int m_deleteTime = 0 ;
    
    /**
     * 暗号表(256).
     */
    private byte[] m_codeTable = null ;
    
    
    
    /**
     * 格納用オブジェクト.
     */
    private final MgcServerValue m_value = new MgcServerValue() ;
    
    /**
     * 前回処理時間.
     */
    private long m_befTime = 0L ;
    
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private MgcSvThread(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の条件を設定してオブジェクトを生成します.
     * <BR>
     * @param thisValue このマシンを示す要素を設定します.
     * @param man 各接続状況を管理するオブジェクトを設定します.
     * @param sv 対象のMgcサーバテーブルを設定します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @exception InputException 入力例外.
     */
    public MgcSvThread( MgcValueImple thisValue,MgcManagerImple man,MgcTable sv,byte[] table )
        throws InputException
    {
        this( thisValue,man,sv,table,DEF_RESEND ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の条件を設定してオブジェクトを生成します.
     * <BR>
     * @param thisValue このマシンを示す要素を設定します.
     * @param man 各接続状況を管理するオブジェクトを設定します.
     * @param sv 対象のMgcサーバテーブルを設定します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @param delTime 応答チェック時間を設定します.<BR>
     *                [MgcSvRcvThread]の[resend]引数と同様の値を設定してください.<BR>
     *                [30000]以下の場合は設定しても意味を持ちません.
     * @exception InputException 入力例外.
     */
    public MgcSvThread( MgcValueImple thisValue,MgcManagerImple man,MgcTable sv,byte[] table,int delTime )
        throws InputException
    {
        
        if(
            thisValue == null || thisValue.isUse() == false ||
            man == null || sv == null
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        delTime = ( delTime <= MIN_DELETE_TIME ) ? MIN_DELETE_TIME : delTime ;
        table = ( table == null || table.length != MgcCommon.MAX_RAND_CODE ) ?
            null : table ;
        
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                
                // 基本条件を設定.
                m_thisValue = thisValue ;
                m_man = man ;
                m_svTable = sv ;
                m_codeTable = table ;
                m_deleteTime = delTime ;
                
                m_befTime = System.currentTimeMillis() ;
                
                // スレッドを生成.
                m_thread.create( this ) ;
                m_thread.startThread() ;
                
            }
        }catch( Exception e ){
            this.destroy() ;
            throw new InputException( e ) ;
        }
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクト情報を破棄します.
     */
    public final void destroy()
    {
        try{
            synchronized( m_sync.get() ){
                
                m_thread.clear() ;
                
                m_thisValue = null ;
                m_man = null ;
                m_svTable = null ;
                m_codeTable = null ;
                m_deleteTime = 0 ;
                
                m_befTime = 0L ;
                m_value.clear() ;
                
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        
        m_thisValue = null ;
        m_man = null ;
        m_svTable = null ;
        m_codeTable = null ;
        m_deleteTime = 0 ;
        
        m_befTime = 0L ;
        m_value.clear() ;
        
    }
    
    /**
     * 設定されている暗号表を取得.
     * <BR><BR>
     * 設定されている暗号表を取得します.
     * <BR>
     * @return byte[] 設定されている暗号表が返されます.
     *                [null]が返された場合、デフォルト値が設定されているか
     *                オブジェクトが破棄されています.
     */
    public final byte[] getCodeTable()
    {
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_codeTable ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 接続状況監視時間を取得.
     * <BR><BR>
     * 設定されている接続状況を監視する時間を取得します.
     * <BR>
     * @return int 接続状況監視時間が返されます.<BR>
     *             [-1]が返された場合、オブジェクトが破棄されています.
     */
    public final int getCheckConnectTime()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_deleteTime ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクト利用可能チェック.
     * <BR><BR>
     * オブジェクト情報が利用可能かチェックします.
     * <BR>
     * @return boolean オブジェクトが利用可能かチェックします.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        int deleteTime ;
        long befTime ;
        
        MgcValueImple thisValue = null ;
        MgcManagerImple man = null ;
        MgcTable svTable = null ;
        MgcServerValue value = null ;
        
        MgcValue[] vals = null ;
        MgcValueImple val = null ;
        byte[] table = null ;
        byte[] rcv = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                thisValue = m_thisValue ;
                man = m_man ;
                svTable = m_svTable ;
                value = m_value ;
                table = m_codeTable ;
                deleteTime = m_deleteTime ;
                
                befTime = m_befTime + DEF_TIMING ;
                
            }
            
            ///////////////////////////////////////
            // 受信された情報が存在するかチェック.
            ///////////////////////////////////////
            if( ( len = svTable.size() ) > 0 ){
                
                // 受信データ数ループ.
                for( i = 0 ; i < len ; i ++ ){
                    
                    // 受信情報を取得.
                    if( ( rcv = svTable.get() ) == null ){
                        continue ;
                    }
                    
                    try{
                        
                        // 受信情報を解析.
                        MgcCommon.getMgcServer( value,rcv,table,thisValue.getServerID() ) ;
                        
                        //<><><><><><><><><><><><><><><><>
                        // 受信タイプが「終了通知」の場合.
                        //<><><><><><><><><><><><><><><><>
                        if( MgcCommon.isMgcServerEnd( rcv ) == true ){
                            
                            // 管理オブジェクトから削除.
                            if( man.remove( value ) == true ){
                                
                                LOG.info(
                                    "対象のMgc要素(" +
                                    " name:" + value.getServerName() +
                                    " id:" + value.getServerID() +
                                    " addr:" + value.getBindAddress() +
                                    " port:" + value.getBindPort() +
                                    ")が終了通知による削除されました"
                                ) ;
                                
                            }
                            
                        }
                        //<><><><><><><><><><><><><><><>
                        // 受信タイプが通知情報の場合.
                        //<><><><><><><><><><><><><><><>
                        else{
                            
                            // 管理オブジェクトに登録.
                            if( man.renew( value ) == true ){
                                
                                LOG.info(
                                    "対象のMgc要素(" +
                                    " name:" + value.getServerName() +
                                    " id:" + value.getServerID() +
                                    " addr:" + value.getBindAddress() +
                                    " port:" + value.getBindPort() +
                                    ")が検知されました"
                                ) ;
                                
                            }
                            
                            // 管理オブジェクトをソート.
                            man.sort( value.getServerName() ) ;
                            
                        }
                        
                        // Mgc標準時間を計算.
                        thisValue.setDefaultTime( value.getTime() ) ;
                        thisValue.update() ;
                        
                    }catch( Exception ee ){
                        LOG.error( "エラーが発生しました",ee ) ;
                    }
                    
                    rcv = null ;
                    
                }
                
            }
            ////////////////////////////////////////
            // 管理テーブルをチェックする条件の場合.
            ////////////////////////////////////////
            else if( befTime <= System.currentTimeMillis() ){
                
                // 現在管理されている全ての要素を取得.
                if( ( vals = man.getAll() ) != null ){
                    
                    len = vals.length ;
                    
                    // 規定時間を越していないかチェック
                    for( i = 0 ; i < len ; i ++ ){
                        
                        val = ( MgcValueImple )vals[ i ] ;
                        
                        // 一定期間応答が無い場合.
                        if( val.getUpdateTime() + deleteTime <= System.currentTimeMillis() ){
                            
                            // 応答なしカウントが規定値を越す場合.
                            if( val.getNoResponseCount() >= MAX_NORESPONSE ){
                                
                                // 削除処理.
                                synchronized( man ){
                                    if(
                                        man.isUseMgc(
                                            val.getServerName(),
                                            val.getBindAddress(),
                                            val.getBindPort()
                                        ) == true
                                    )
                                    {
                                        
                                        LOG.warn(
                                            "対象のMgc要素(" +
                                            " name:" + val.getServerName() +
                                            " id:" + val.getServerID() +
                                            " addr:" + val.getBindAddress() +
                                            " port:" + val.getBindPort() +
                                            ")が応答がないため削除されました"
                                        ) ;
                                        
                                        // 管理オブジェクトから削除.
                                        man.remove( val ) ;
                                        
                                    }
                                }
                                
                            }
                            // 応答なしカウントが規定値以下の場合.
                            else{
                                
                                val.addNoResponseCount() ;
                                
                            }
                            
                        }
                        
                        val = null ;
                        vals[ i ] = null ;
                        
                    }
                    
                }
                
                // 前回処理時間を更新.
                synchronized( m_sync.get() ){
                    m_befTime = System.currentTimeMillis() ;
                }
                
            }
            /////////////////////////////
            // 処理条件が存在しない場合.
            /////////////////////////////
            else{
                
                // 一定期間待機.
                UtilCom.idleTime() ;
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( Exception e ){
            e.printStackTrace() ;
        }finally{
            thisValue = null ;
            man = null ;
            svTable = null ;
            value = null ;
            vals = null ;
            val = null ;
            rcv = null ;
        }
        
    }
    
}

