/*
 * @(#)BinMemoryResource.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;


/**
 * メモリー領域を利用したバイナリオブジェクト.
 * <BR><BR>
 * メモリー領域を利用したバイナリオブジェクトを生成します.
 *
 * @version     1.00, 2005/04/01
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BinMemoryResource implements BinResource
{
    
    /**
     * バイナリ単位.
     */
    private static final int BINARY_LENGTH = 512 ;
    
    /**
     * バイナリ情報.
     */
    private byte[] m_binary = null ;
    
    /**
     * 有効書き込みデータ位置.
     */
    private int m_usePoint = 0 ;
    
    /**
     * 全体データ長.
     */
    private int m_allLength = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public BinMemoryResource()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 生成数を設定してバイナリ情報を生成します.
     * <BR>
     * @param size 生成対象のバイナリ長を設定します.
     * @exception InputException 入力例外.
     */
    public BinMemoryResource( int size )
        throws InputException
    {
        this.create( size ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * バイナリ情報を設定して生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     */
    public BinMemoryResource( byte[] binary )
        throws InputException
    {
        this.create( binary ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 生成数を設定してバイナリ情報を生成します.
     * <BR>
     * @param size 生成対象のバイナリ長を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int size )
        throws InputException
    {
        if( size <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                this.newSize( size ) ;
            }
        }catch( Exception e ){
            this.clear() ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * バイナリ情報を設定して生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( byte[] binary )
        throws InputException
    {
        if( binary == null || binary.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                m_binary = binary ;
                m_allLength = m_usePoint = binary.length ;
            }
        }catch( Exception e ){
            this.clear() ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * バイナリ情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_binary = null ;
        m_usePoint = 0 ;
        m_allLength = -1 ;
    }
    
    /**
     * リセット処理.
     * <BR><BR>
     * 有効データ長をリセットします.
     */
    public final void reset()
    {
        try{
            synchronized( m_sync.get() ){
                m_usePoint = 0 ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象条件に情報を設定します.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param b 設定対象のバイト情報を設定します.
     */
    public final void set( int no,int b )
    {
        try{
            synchronized( m_sync.get() ){
                
                this.newSize( no+1 ) ;
                m_binary[ no ] = ( byte )( b & 0x000000ff ) ;
                m_usePoint = ( m_usePoint <= no+1 ) ? no+1 : m_usePoint ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
        }
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,byte[] bin ) throws ArrayIndexOutOfBoundsException
    {
        if( bin == null || no < 0 ){
            if( bin == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( bin.length <= 0 ){
            return 0 ;
        }
        
        return this.setBinary( no,bin,0,bin.length ) ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @param off 設定対象のオフセット値を設定します.
     * @param len 設定対象のバイナリ長を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,byte[] bin,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        int ret ;
        
        if( bin == null || off < 0 || len <= 0 || no < 0 ){
            if( bin == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = bin.length ) <= 0 ){
            return 0 ;
        }
        else if( off >= inputLen ){
            throw new ArrayIndexOutOfBoundsException(
                "設定条件(off:" + off +
                ")は設定バイナリ長(" + inputLen +
                ")を越しています"
            ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                this.newSize( inputLen + no ) ;
                System.arraycopy( bin,off,m_binary,no,inputLen ) ;
                m_usePoint = ( m_usePoint <= ( no + inputLen ) ) ? ( no + inputLen ) : m_usePoint ;
                
                ret = inputLen ;
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリオブジェクトを設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,BinResource bin ) throws ArrayIndexOutOfBoundsException
    {
        if( bin == null || bin.isUse() == false || no < 0 ){
            if( bin == null || bin.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( bin.size() <= 0 ){
            return 0 ;
        }
        
        return this.setBinary( no,bin,0,bin.size() ) ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリオブジェクトを設定します.
     * @param off 設定対象のオフセット値を設定します.
     * @param len 設定対象のバイナリ長を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,BinResource bin,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int i ;
        int lenI ;
        int inputLen ;
        int etc ;
        int pnt ;
        int bpnt ;
        int wlen ;
        int ret ;
        
        byte[] binary = null ;
        byte[] buf = null ;
        
        if( bin == null || bin.isUse() == false || off < 0 || len <= 0 || no < 0 ){
            if( bin == null || bin.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = bin.size() ) <= 0 ){
            return 0 ;
        }
        else if( off >= inputLen ){
            throw new ArrayIndexOutOfBoundsException(
                "設定条件(off:" + off +
                ")は設定バイナリ長(" + inputLen +
                ")を越しています"
            ) ;
        }
        
        
        try{
            
            synchronized( m_sync.get() ){
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                this.newSize( inputLen + no ) ;
                binary = m_binary ;
                
                lenI = inputLen / ResourceDef.BUFFER_LENGTH ;
                etc = inputLen % ResourceDef.BUFFER_LENGTH ;
                
                buf = new byte[ ResourceDef.BUFFER_LENGTH ] ;
                for( i = 0,pnt = no,bpnt = off,ret = 0 ; i < lenI ; i ++ ){
                    
                    wlen = bin.getBinary( buf,bpnt ) ;
                    System.arraycopy( buf,0,binary,pnt,wlen ) ;
                    
                    pnt += wlen ;
                    bpnt += wlen ;
                    ret += wlen ;
                    
                }
                
                if( etc != 0 ){
                    wlen = bin.getBinary( buf,bpnt,0,etc ) ;
                    System.arraycopy( buf,0,binary,pnt,wlen ) ;
                    ret += wlen ;
                }
                
                m_usePoint = ( m_usePoint <= ( no + inputLen ) ) ? ( no + inputLen ) : m_usePoint ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }finally{
            binary = null ;
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象条件の情報を取得します.
     * <BR>
     * @param no 取得対象項番を設定します.
     * @return int 対象のバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int get( int no ) throws ArrayIndexOutOfBoundsException
    {
        int ret ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_usePoint <= no ){
                    throw new ArrayIndexOutOfBoundsException(
                        "指定位置(" + no +
                        ")は有効データ長(" + m_usePoint +
                        ")を越しています"
                    ) ;
                }
                
                ret = ( int )( m_binary[ no ] & 0x000000ff ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @return byte[] 設定されているバイナリ情報が返されます.
     */
    public final byte[] getBinary()
    {
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                if( m_usePoint <= 0 ){
                    return null ;
                }
                ret = new byte[ m_usePoint ] ;
                System.arraycopy( m_binary,0,ret,0,m_usePoint ) ;
            }
            
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @return byte[] 取得されたバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final byte[] getBinary( int no ) throws ArrayIndexOutOfBoundsException
    {
        int len ;
        byte[] ret = null ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_usePoint <= no ){
                    throw new ArrayIndexOutOfBoundsException(
                        "指定位置(" + no +
                        ")は有効データ長(" + m_usePoint +
                        ")を越しています"
                    ) ;
                }
                
                len = m_usePoint ;
                ret = new byte[ len - no ] ;
                System.arraycopy( m_binary,no,ret,0,len-no ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return byte[] 取得されたバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final byte[] getBinary( int no,int len ) throws ArrayIndexOutOfBoundsException
    {
        byte[] ret = null ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( len <= 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_usePoint <= no ){
                    throw new ArrayIndexOutOfBoundsException(
                        "指定位置(" + no +
                        ")は有効データ長(" + m_usePoint +
                        ")を越しています"
                    ) ;
                }
                
                len = ( m_usePoint < no + len ) ? m_usePoint - no : len ;
                
                ret = new byte[ len ] ;
                System.arraycopy( m_binary,no,ret,0,len ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @return int 取得されたバイナリ長が返されます.
     */
    public final int getBinary( byte[] out )
    {
        int inputLen ;
        
        if( out == null ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,0,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( byte[] out,int no ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        
        if( out == null ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,no,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @param off 取得対象のバイナリオフセット値を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( byte[] out,int no,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        int ret ;
        
        if( out == null || off < 0 || len <= 0 || no < 0 ){
            if( out == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                inputLen = ( m_usePoint < no + inputLen ) ? m_usePoint - no : inputLen ;
                System.arraycopy( m_binary,no,out,off,inputLen ) ;
                
                ret = inputLen ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @return int 取得されたバイナリ長が返されます.
     */
    public final int getBinary( BinResource out )
    {
        int inputLen ;
        
        if( out == null || out.isUse() == false ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,0,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( BinResource out,int no ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        
        if( out == null || out.isUse() == false ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,no,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @param off 取得対象のバイナリオフセット値を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( BinResource out,int no,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        int ret ;
        
        if( out == null || out.isUse() == false || off < 0 || len <= 0 || no < 0 ){
            if( out == null || out.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                inputLen = ( m_usePoint < no + inputLen ) ? m_usePoint - no : inputLen ;
                //inputLen = ( m_usePoint < no + len ) ? m_usePoint - no : len ;
                ret = out.setBinary( off,this,no,inputLen ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の有効バイナリ長を取得.
     * <BR><BR>
     * 現在の有効なバイナリ長を取得します.
     * <BR>
     * @return int 現在の有効なバイナリ長が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_usePoint ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のバイナリ長を取得.
     * <BR><BR>
     * 現在のバイナリ長を取得します.
     * <BR>
     * @return int 現在のバイナリ長が返されます.
     */
    public final int getAllSize()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_binary.length ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトタイプを取得.
     * <BR><BR>
     * オブジェクトタイプを取得します.
     * <BR>
     * @return int オブジェクトタイプが返されます.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]が返された場合、
     *             [org.maachang.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]が返された場合、
     *             [org.maachang.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]が返された場合、
     *             [org.maachang.commons.resource.BinCacheResource]オブジェクトです.
     */
    public final int getType()
    {
        return BinResource.BIN_RESOURCE_TYPE_MEMORY ;
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR><BR>
     * オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_binary != null ) ? true : false ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    /**
     * バイナリ長を増やす.
     */
    private final void newSize( int size ) throws InputException
    {
        int len ;
        byte[] tmp = null ;
        
        if( size <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( m_binary != null ){
                
                if( m_allLength < size ){
                    
                    size = ( size / BINARY_LENGTH ) + ( size % BINARY_LENGTH ) ;
                    size *= BINARY_LENGTH ;
                    
                    tmp = m_binary ;
                    m_binary = new byte[ size ] ;
                    System.arraycopy( tmp,0,m_binary,0,m_allLength ) ;
                    
                    m_allLength = size ;
                    
                }
                
            }
            else{
                
                size = ( size / BINARY_LENGTH ) + ( size % BINARY_LENGTH ) ;
                size *= BINARY_LENGTH ;
                
                m_binary = new byte[ size ] ;
                m_allLength = size ;
                
            }
            
        }catch( Exception e ){
            throw new InputException( e ) ;
        }finally{
            tmp = null ;
        }
        
    }
    
}
