/*
 * @(#)BinResourceInputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource ;

import java.io.IOException;
import java.io.InputStream;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.array.IntArray;


/**
 * バイナリリソースInputStream.
 * <BR><BR>
 * [org.maachang.commons.resource.BinResource]をInputStreamで利用可能にしたオブジェクトです.
 *  
 * @version 1.0.0 2005/04/02
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class BinResourceInputStream extends InputStream
{
    
    /**
     * バイナリリソース.
     */
    private BinResource m_resource = null ;
    
    /**
     * 読み込み位置.
     */
    private int m_seek = 0 ;
    
    /**
     * 読み込みデータ長.
     */
    private int m_length = -1 ;
    
    /**
     * マークポイント.
     */
    private final IntArray m_mark = new IntArray() ;
    
    /**
     * クローズモード.
     */
    private boolean m_closeMode = false ;
    
    
    /**
     * コンストラクタ.
     */
    private BinResourceInputStream()
    {
        super() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param resource 読み込み対象のバイナリリソースオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceInputStream( BinResource resource )
        throws InputException
    {
        this( false,resource ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param closeMode クローズ処理時に、渡されるリソースをクローズするか設定します.<BR>
     *                  [true]の場合、クローズ時に破棄されます.<BR>
     *                  [false]の場合、クローズ時に継続されます.
     * @param resource 読み込み対象のバイナリリソースオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceInputStream( boolean closeMode,BinResource resource )
        throws InputException
    {
        this( closeMode,resource,0 ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param closeMode クローズ処理時に、渡されるリソースをクローズするか設定します.<BR>
     *                  [true]の場合、クローズ時に破棄されます.<BR>
     *                  [false]の場合、クローズ時に継続されます.
     * @param resource 読み込み対象のバイナリリソースオブジェクトを設定します.
     * @param offset 読み込み開始位置を設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceInputStream( boolean closeMode,BinResource resource,int offset )
        throws InputException
    {
        super() ;
        
        if( resource == null || resource.isUse() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.close() ;
        
        m_resource = resource ;
        m_closeMode = closeMode ;
        m_seek = ( offset <= 0 ) ? 0 : offset ;
        m_length = resource.size() ;
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param resource 読み込み対象のバイナリリソースオブジェクトを設定します.
     * @param offset 読み込み開始位置を設定します.
     * @param length 読み込みデータ長を設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceInputStream( BinResource resource,int offset,int length )
        throws InputException
    {
        this( false,resource,offset,length ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param closeMode クローズ処理時に、渡されるリソースをクローズするか設定します.<BR>
     *                  [true]の場合、クローズ時に破棄されます.<BR>
     *                  [false]の場合、クローズ時に継続されます.
     * @param resource 読み込み対象のバイナリリソースオブジェクトを設定します.
     * @param offset 読み込み開始位置を設定します.
     * @param length 読み込みデータ長を設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceInputStream( boolean closeMode,BinResource resource,int offset,int length )
        throws InputException
    {
        super() ;
        
        int resourceLen ;
        
        if( resource == null || resource.isUse() == false || offset < 0 || length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.close() ;
        
        resourceLen = resource.size() ;
        
        m_resource = resource ;
        m_closeMode = closeMode ;
        m_seek = offset ;
        m_length = ( resourceLen < offset + length ) ? resourceLen : offset + length ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     */
    public final void close()
    {
        if( m_closeMode == true ){
            if( m_resource != null ){
                m_resource.clear() ;
            }
        }
        m_resource = null ;
        m_seek = 0 ;
        m_length = -1 ;
        m_mark.clear() ;
        m_closeMode = false ;
    }
    
    /**
     * １バイトのデータを読み込みます.
     * <BR><BR>
     * １バイトのデータを読み込みます.
     * <BR>
     * @return int １バイトのデータが返されます.
     * @exception IOException IO例外.
     */
    public final int read() throws IOException
    {
        int ret ;
        
        try{
            if( m_length <= m_seek ){
                ret = -1 ;
            }
            else{
                ret = ( int )( m_resource.get( m_seek ) & 0x000000ff ) ;
                m_seek ++ ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary ) throws IOException
    {
        return this.read( binary,0,binary.length ) ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @param offset 読み込み格納先の位置を設定します.
     * @param length 読み込み情報長を設定します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary,int offset,int length )
        throws IOException
    {
        int ret ;
        
        if(
            (
                offset | length | ( offset + length ) |
                ( binary.length - ( offset + length ) )
            ) < 0
        )
        {
            throw new IndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if ( length == 0 ){
            return 0 ;
        }
        
        try{
            if( m_length <= m_seek + length ){
                ret = ( m_length <= m_seek ) ?
                    -1 :
                    m_resource.getBinary( binary,m_seek,offset,m_length - m_seek ) ;
            }
            else{
                ret = m_resource.getBinary( binary,m_seek,offset,length ) ;
            }
            
            m_seek += ( ret <= 0 ) ? 0 : ret ;
            
        }catch( Exception e ){
            throw new IOException( e.getMessage() ) ;
        }
        
        return ret ;
    }
    
    /**
     * スキップ処理.
     * <BR><BR>
     * スキップ処理を行います.
     * <BR>
     * @param n スキップ長を設定します.
     * @return long 実際にスキップされたデータ長が返されます.
     */
    public final long skip( long n ) throws IOException
    {
        int seek ;
        int length ;
        long ret ;
        
        ret = 0L ;
        
        try{
                
            if( n > 0L ){
                
                seek = m_seek ;
                length = m_length ;
                
                if( length <= ( n + seek ) ){
                    
                    ret = length - seek ;
                    m_seek = length ;
                    
                }
                else{
                    
                    ret = n ;
                    m_seek += n ;
                    
                }
            }
            
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
        
    }
    
    /**
     * 読み込みが可能なデータ長を取得.
     * <BR><BR>
     * 読み込みが可能なデータ長を取得します.
     * <BR>
     * @return int 読み込みが可能なデータ長が返されます.
     * @exception IOException IO例外.
     */
    public final int available() throws IOException
    {
        int ret ;
        
        try{
            ret = m_length - m_seek ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * マーク処理.
     * <BR><BR>
     * マーク処理を行います.<BR>
     * またマークした位置に戻す場合は
     * [BinResourceInputStream.reset()]を利用します.
     * <BR>
     * @param offset 現在の位置からのオフセット値を設定します.
     */
    public final void mark( int offset )
    {
        int seek ;
        
        try{
            
            seek = m_seek + offset ;
            
            if( seek >= 0 && seek < m_length ){
                m_mark.add( seek ) ;
            }
            
        }catch( Exception t ){
        }
    }
    
    /**
     * マーク処理に対するリセット処理.
     * <BR><BR>
     * リセット処理を行います.
     * <BR>
     * @exception IOException IO例外.
     */
    public final void reset() throws IOException
    {
        try{
            
            if( m_mark.size() == 0 ){
                throw new IOException(
                    "マークした位置情報は存在しません"
                ) ;
            }
            
            m_seek = m_mark.remove( 0 ) ;
            
        }catch( IOException io ){
            throw io ;
        }catch( Exception t ){
        }
    }
    
    /**
     * マーク処理サポートチェック.
     * <BR><BR>
     * マーク処理がサポートされているかチェックします.
     * <BR>
     * @return boolean マーク処理がサポートされているか返されます.<BR>
     *                 [true]が返された場合サポートされています.
     *                 [false]が返された場合サポートされていません.
     */
    public final boolean markSupported()
    {
        return true ;
    }
    
    /**
     * バイナリリソースを取得.
     * <BR><BR>
     * 設定されているバイナリリソースを取得します.
     * <BR>
     * @return BinResource 対象のバイナリリソースが返されます.
     */
    public final BinResource getBinResource()
    {
        return m_resource ;
    }
    
}

