/*
 * @(#)ConvertResourceParam.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource ;

import java.math.BigDecimal;

import org.maachang.commons.exception.InputException;


/**
 * プリミティブ型の情報を、リソース情報にコンバートする為の処理群.
 * <BR><BR>
 * プリミティブ型の情報を、リソース情報にコンバートする為の処理群です.
 *
 * @version 1.00,2005/08/11
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ConvertResourceParam
{
    /**
     * 数値キャラクタ終了コード.
     */
    private static final char ENDCHAR = 0x0000 ;
    
    /**
     * boolean : true.
     */
    private static final int BOOLEAN_TRUE_CD = 0x00000001 ;
    
    /**
     * boolean : false.
     */
    private static final int BOOLEAN_FALSE_CD = 0x00000000 ;
    
    
    
    /**
     * データ長 : boolean.
     */
    public static final int LENGTH_BOOLEAN = 1 ;
    
    /**
     * データ長 : byte.
     */
    public static final int LENGTH_BYTE = 1 ;
    
    /**
     * データ長 : char.
     */
    public static final int LENGTH_CHAR = 2 ;
    
    /**
     * データ長 : short.
     */
    public static final int LENGTH_SHORT = 2 ;
    
    /**
     * データ長 : int.
     */
    public static final int LENGTH_INT = 4 ;
    
    /**
     * データ長 : long.
     */
    public static final int LENGTH_LONG = 8 ;
    
    
    
    /**
     * コンストラクタ.
     */
    private ConvertResourceParam()
    {
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * boolean情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertBoolean( BinResource out,int offset,boolean value )
    {
        
        out.set(
            offset,
            ( byte )(
                ( value == true ) ? BOOLEAN_TRUE_CD : BOOLEAN_FALSE_CD
            )
        ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * boolean情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertBoolean( BufferedBinResource out,int offset,boolean value )
    {
        
        out.set(
            offset,
            ( byte )(
                ( value == true ) ? BOOLEAN_TRUE_CD : BOOLEAN_FALSE_CD
            )
        ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をboolean変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return boolean 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final boolean convertBoolean( int off,BinResource binary )
        throws InputException
    {
        boolean ret ;
        
        try{
            
            ret = ( ( binary.get( off ) & 0x000000ff ) == BOOLEAN_TRUE_CD ) ?
                true : false ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をboolean変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return boolean 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final boolean convertBoolean( int off,BufferedBinResource binary )
        throws InputException
    {
        boolean ret ;
        
        try{
            
            ret = ( ( binary.get( off ) & 0x000000ff ) == BOOLEAN_TRUE_CD ) ?
                true : false ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * byte情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertByte( BinResource out,int offset,byte value )
    {
        out.set( offset,value ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * byte情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertByte( BufferedBinResource out,int offset,byte value )
    {
        out.set( offset,value ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をbyte変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return byte 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte convertByte( int off,BinResource binary )
        throws InputException
    {
        byte ret ;
        
        try{
            
            ret = ( byte )( binary.get( off ) & 0x000000ff ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をbyte変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return byte 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte convertByte( int off,BufferedBinResource binary )
        throws InputException
    {
        byte ret ;
        
        try{
            
            ret = ( byte )( binary.get( off ) & 0x000000ff ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * char情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertChar( BinResource out,int offset,char value )
    {
        byte[] bin = null ;
        
        bin = new byte[ 2 ] ;
        
        bin[ 0 ] = ( byte )(   value & 0x000000ff ) ;
        bin[ 1 ] = ( byte )( ( value & 0x0000ff00 ) >> 8 ) ;
        
        out.setBinary( offset,bin ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * char情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertChar( BufferedBinResource out,int offset,char value )
    {
        out.set( offset,( byte )(   value & 0x000000ff ) ) ;
        out.set( offset+1,( byte )( ( value & 0x0000ff00 ) >> 8 ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をchar変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return char 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final char convertChar( int off,BinResource binary )
        throws InputException
    {
        char ret ;
        byte[] bin = null ;
        
        try{
            
            bin = binary.getBinary( off,2 ) ;
            
            ret = ( char )(
                (   bin[ 0 ] & 0x000000ff ) |
                ( ( bin[ 1 ] & 0x000000ff ) << 8 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をchar変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return char 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final char convertChar( int off,BufferedBinResource binary )
        throws InputException
    {
        char ret ;
        
        try{
            
            ret = ( char )(
                (   binary.get( off ) & 0x000000ff ) |
                ( ( binary.get( off+1 ) & 0x000000ff ) << 8 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * short情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertShort( BinResource out,int offset,short value )
    {
        byte[] bin = null ;
        
        bin = new byte[ 2 ] ;
        
        bin[ 0 ] = ( byte )(   value & 0x000000ff ) ;
        bin[ 1 ] = ( byte )( ( value & 0x0000ff00 ) >> 8 ) ;
        
        out.setBinary( offset,bin ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * short情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertShort( BufferedBinResource out,int offset,short value )
    {
        out.set( offset,( byte )(   value & 0x000000ff ) ) ;
        out.set( offset+1,( byte )( ( value & 0x0000ff00 ) >> 8 ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をshort変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return short 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final short convertShort( int off,BinResource binary )
        throws InputException
    {
        short ret ;
        byte[] bin = null ;
        
        try{
            
            bin = binary.getBinary( off,2 ) ;
            
            ret = ( short )(
                (   bin[ 0 ] & 0x000000ff ) |
                ( ( bin[ 1 ] & 0x000000ff ) << 8 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をshort変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return short 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final short convertShort( int off,BufferedBinResource binary )
        throws InputException
    {
        short ret ;
        
        try{
            
            ret = ( short )(
                (   binary.get( off ) & 0x000000ff ) |
                ( ( binary.get( off+1 ) & 0x000000ff ) << 8 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * int情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertInt( BinResource out,int offset,int value )
    {
        byte[] bin = null ;
        
        bin = new byte[ 4 ] ;
        
        bin[ 0 ] = ( byte )(   value & 0x000000ff ) ;
        bin[ 1 ] = ( byte )( ( value & 0x0000ff00 ) >> 8 ) ;
        bin[ 2 ] = ( byte )( ( value & 0x00ff0000 ) >> 16 ) ;
        bin[ 3 ] = ( byte )( ( value & 0xff000000 ) >> 24 ) ;
        
        out.setBinary( offset,bin ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * int情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertInt( BufferedBinResource out,int offset,int value )
    {
        out.set( offset,( byte )(   value & 0x000000ff ) ) ;
        out.set( offset+1,( byte )( ( value & 0x0000ff00 ) >> 8 ) ) ;
        out.set( offset+2,( byte )( ( value & 0x00ff0000 ) >> 16 ) ) ;
        out.set( offset+3,( byte )( ( value & 0xff000000 ) >> 24 ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をint変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return int 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final int convertInt( int off,BinResource binary )
        throws InputException
    {
        int ret ;
        byte[] bin = null ;
        
        try{
            
            bin = binary.getBinary( off,4 ) ;
            
            ret = ( int )(
                (   bin[ 0 ] & 0x000000ff ) |
                ( ( bin[ 1 ] & 0x000000ff ) << 8 ) |
                ( ( bin[ 2 ] & 0x000000ff ) << 16 ) |
                ( ( bin[ 3 ] & 0x000000ff ) << 24 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をint変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return int 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final int convertInt( int off,BufferedBinResource binary )
        throws InputException
    {
        int ret ;
        
        try{
            
            ret = ( short )(
                (   binary.get( off ) & 0x000000ff ) |
                ( ( binary.get( off+1 ) & 0x000000ff ) << 8 ) |
                ( ( binary.get( off+2 ) & 0x000000ff ) << 16 ) |
                ( ( binary.get( off+3 ) & 0x000000ff ) << 24 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * long情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertLong( BinResource out,int offset,long value )
    {
        byte[] bin = null ;
        
        bin = new byte[ 8 ] ;
        
        bin[ 0 ] = ( byte )(   value & 0x00000000000000ffL ) ;
        bin[ 1 ] = ( byte )( ( value & 0x000000000000ff00L ) >> 8 ) ;
        bin[ 2 ] = ( byte )( ( value & 0x0000000000ff0000L ) >> 16 ) ;
        bin[ 3 ] = ( byte )( ( value & 0x00000000ff000000L ) >> 24 ) ;
        bin[ 4 ] = ( byte )( ( value & 0x000000ff00000000L ) >> 32 ) ;
        bin[ 5 ] = ( byte )( ( value & 0x0000ff0000000000L ) >> 40 ) ;
        bin[ 6 ] = ( byte )( ( value & 0x00ff000000000000L ) >> 48 ) ;
        bin[ 7 ] = ( byte )( ( value & 0xff00000000000000L ) >> 56 ) ;
        
        out.setBinary( offset,bin ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * long情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     */
    public static final void convertLong( BufferedBinResource out,int offset,long value )
    {
        out.set( offset,( byte )(   value & 0x00000000000000ffL ) ) ;
        out.set( offset+1,( byte )( ( value & 0x000000000000ff00L ) >> 8 ) ) ;
        out.set( offset+2,( byte )( ( value & 0x0000000000ff0000L ) >> 16 ) ) ;
        out.set( offset+3,( byte )( ( value & 0x00000000ff000000L ) >> 24 ) ) ;
        out.set( offset+4,( byte )( ( value & 0x000000ff00000000L ) >> 32 ) ) ;
        out.set( offset+5,( byte )( ( value & 0x0000ff0000000000L ) >> 40 ) ) ;
        out.set( offset+6,( byte )( ( value & 0x00ff000000000000L ) >> 48 ) ) ;
        out.set( offset+7,( byte )( ( value & 0xff00000000000000L ) >> 56 ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をlong変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return long 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final long convertLong( int off,BinResource binary )
        throws InputException
    {
        long ret ;
        byte[] bin = null ;
        
        try{
            
            bin = binary.getBinary( off,8 ) ;
            
            ret = ( long )(
                (   bin[ 0 ] & 0x00000000000000ffL ) |
                ( ( bin[ 1 ] & 0x00000000000000ffL ) << 8 ) |
                ( ( bin[ 2 ] & 0x00000000000000ffL ) << 16 ) |
                ( ( bin[ 3 ] & 0x00000000000000ffL ) << 24 ) |
                ( ( bin[ 4 ] & 0x00000000000000ffL ) << 32 ) |
                ( ( bin[ 5 ] & 0x00000000000000ffL ) << 40 ) |
                ( ( bin[ 6 ] & 0x00000000000000ffL ) << 48 ) |
                ( ( bin[ 7 ] & 0x00000000000000ffL ) << 56 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をlong変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return long 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final long convertLong( int off,BufferedBinResource binary )
        throws InputException
    {
        long ret ;
        
        try{
            
            ret = ( long )(
                (   binary.get( off ) & 0x00000000000000ffL ) |
                ( ( binary.get( off+1 ) & 0x00000000000000ffL ) << 8 ) |
                ( ( binary.get( off+2 ) & 0x00000000000000ffL ) << 16 ) |
                ( ( binary.get( off+3 ) & 0x00000000000000ffL ) << 24 ) |
                ( ( binary.get( off+4 ) & 0x00000000000000ffL ) << 32 ) |
                ( ( binary.get( off+5 ) & 0x00000000000000ffL ) << 40 ) |
                ( ( binary.get( off+6 ) & 0x00000000000000ffL ) << 48 ) |
                ( ( binary.get( off+7 ) & 0x00000000000000ffL ) << 56 )
            ) ;
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * float情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertFloat( BinResource out,int offset,float value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,String.valueOf( value ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * float情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertFloat( BufferedBinResource out,int offset,float value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,String.valueOf( value ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をfloat変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return float 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final float convertFloat( int off,int length,BinResource binary )
        throws InputException
    {
        return Float.parseFloat( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をfloat変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return float 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final float convertFloat( int off,int length,BufferedBinResource binary )
        throws InputException
    {
        return Float.parseFloat( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * double情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDouble( BinResource out,int offset,double value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,String.valueOf( value ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * double情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDouble( BufferedBinResource out,int offset,double value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,String.valueOf( value ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をdouble変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return double 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final double convertDouble( int off,int length,BinResource binary )
        throws InputException
    {
        return Double.parseDouble( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をdouble変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return double 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final double convertDouble( int off,int length,BufferedBinResource binary )
        throws InputException
    {
        return Double.parseDouble( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * Decimal情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDecimal( BinResource out,int offset,BigDecimal value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,value.toString() ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * Decimal情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDecimal( BufferedBinResource out,int offset,BigDecimal value )
    {
        return ConvertResourceParam.convertFloatStringByBinary( out,offset,value.toString() ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をBigDecimal変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return BigDecimal 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final BigDecimal convertDecimal( int off,int length,BinResource binary )
        throws InputException
    {
        return new BigDecimal( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をBigDecimal変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return BigDecimal 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final BigDecimal convertDecimal( int off,int length,BufferedBinResource binary )
        throws InputException
    {
        return new BigDecimal( ConvertResourceParam.convertFloatBinaryByString( off,binary ) ) ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * String情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換されたバイナリ長が返されます.
     */
    public static final int convertString( BinResource out,int offset,String value,String charset )
    {
        
        int ret ;
        
        try{
            ret = Resource.convertStringByBinResource( out,value,offset,value.length(),charset ) ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * String情報をバイナリ変換します.
     * <BR>
     * @param out 設定対象のバイナリ情報を設定します.
     * @param offset 設定対象のオフセット位置を設定します.
     * @param value 変換対象の処理を設定します.
     * @return int 変換されたバイナリ長が返されます.
     */
    public static final int convertString( BufferedBinResource out,int offset,String value,String charset )
    {
        
        int ret ;
        
        try{
            ret = Resource.convertStringByBinResource( out.getResource(),value,offset,value.length(),charset ) ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をString変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return String 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final String convertString( int off,int length,BinResource binary,String charset )
        throws InputException
    {
        String ret = null ;
        
        try{
            ret = Resource.convertBinResourceByString( binary,off,length,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンバート処理.
     * <BR><BR>
     * 対象のバイナリ情報をString変換します.
     * <BR>
     * @param off 変換対象バイナリのオフセット値を設定します.
     * @param length 変換対象バイナリの変換データ長を設定します.
     * @param binary 変換対象のバイナリ情報を設定します.
     * @return String 変換された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final String convertString( int off,int length,BufferedBinResource binary,String charset )
        throws InputException
    {
        String ret = null ;
        
        try{
            ret = Resource.convertBinResourceByString( binary.getResource(),off,length,charset ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
        
        return ret ;
        
    }
    
    
    
    
    /**
     * 小数点情報をバイナリに変換.
     */
    private static final int convertFloatStringByBinary( BinResource out,int off,String data )
    {
        int i,j ;
        int len ;
        byte dat ;
        int ret ;
        
        byte[] bin = null ;
        
        len = data.length() ;
        ret = ( len + 1 ) / 2 ;
        
        bin = new byte[ ret + 2 ] ;
        
        bin[ 0 ] = ( byte )( ret & 0x000000ff ) ;
        bin[ 1 ] = ( byte )( ( ret & 0x0000ff00 ) >> 8 ) ;
        
        for( i = 0,j = 2 ; i < len ; i ++ ){
            
            dat = ConvertResourceParam.cnvCharByNumBin( data.charAt( i ) ) ;
            
            if( ( i & 0x00000001 ) == 0 ){
                bin[ j ] = ( byte )( dat & 0x0000000f ) ;
            }else{
                bin[ j ] |= ( byte )( ( dat & 0x0000000f ) << 4 ) ;
                j ++ ;
            }
            
        }
        
        out.setBinary( off,bin ) ;
        
        return ret ;
        
    }
    
    /**
     * 小数点情報をバイナリに変換.
     */
    private static final int convertFloatStringByBinary( BufferedBinResource out,int off,String data )
    {
        int i,j ;
        int len ;
        int cd ;
        byte dat ;
        int ret ;
        
        len = data.length() ;
        ret = ( len + 1 ) / 2 ;
        
        out.set( off,( byte )( ret & 0x000000ff ) ) ;
        out.set( off+1,( byte )( ( ret & 0x0000ff00 ) >> 8 ) ) ;
        
        for( i = 0,j = off+2,cd = -1 ; i < len ; i ++ ){
            
            dat = ConvertResourceParam.cnvCharByNumBin( data.charAt( i ) ) ;
            
            if( ( i & 0x00000001 ) == 0 ){
                cd = dat & 0x0000000f  ;
            }else{
                cd |= ( ( dat & 0x0000000f ) << 4 ) ;
                out.set( j,cd ) ;
                cd = -1 ;
                j ++ ;
            }
            
        }
        
        if( cd != -1 ){
            out.set( off+2+len-1,cd ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 小数点バイナリ情報を文字列に変換.
     */
    public static final String convertFloatBinaryByString( int off,BinResource binary )
        throws InputException
    {
        
        int i ;
        int len ;
        
        char dat ;
        byte[] bin = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        if( binary == null || binary.isUse() == false || binary.size() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        try{
            
            bin = binary.getBinary( off,2 ) ;
            len = ( int )(
                ( bin[ 0 ] & 0x000000ff ) |
                ( ( bin[ 1 ] & 0x000000ff ) << 8 )
            ) ;
            
            bin = binary.getBinary( off+2,len ) ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                dat = ConvertResourceParam.cnvNumBinByChar( ( byte )( bin[ i ] & 0x0000000f ) ) ;
                if( dat == ConvertResourceParam.ENDCHAR ){
                    break ;
                }
                
                buf.append( dat ) ;
                
                dat = ConvertResourceParam.cnvNumBinByChar( ( byte )( ( bin[ i ] & 0x000000f0 ) >> 4 ) ) ;
                if( dat == ConvertResourceParam.ENDCHAR ){
                    break ;
                }
                
                buf.append( dat ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception e ){
            throw new InputException( e ) ;
        }finally{
            buf = null ;
            bin = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 小数点バイナリ情報を文字列に変換.
     */
    public static final String convertFloatBinaryByString( int off,BufferedBinResource binary )
        throws InputException
    {
        
        int i,j ;
        int len ;
        
        char dat ;
        StringBuffer buf = null ;
        String ret = null ;
        
        if( binary == null || binary.size() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        try{
            
            len = ( int )(
                ( binary.get( off ) & 0x000000ff ) |
                ( ( binary.get( off+1 ) & 0x000000ff ) << 8 )
            ) ;
            
            for( i = 0,j = off+2 ; i < len ; i ++,j ++ ){
                
                dat = ConvertResourceParam.cnvNumBinByChar( ( byte )( binary.get( j ) & 0x0000000f ) ) ;
                if( dat == ConvertResourceParam.ENDCHAR ){
                    break ;
                }
                
                buf.append( dat ) ;
                
                dat = ConvertResourceParam.cnvNumBinByChar( ( byte )( ( binary.get( j ) & 0x000000f0 ) >> 4 ) ) ;
                if( dat == ConvertResourceParam.ENDCHAR ){
                    break ;
                }
                
                buf.append( dat ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception e ){
            throw new InputException( e ) ;
        }finally{
            buf = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 数値変換 : 文字コードから数値バイナリ変換コードに変換.
     */
    private static final byte cnvCharByNumBin( char cd )
    {
        
        byte ret ;
        
        switch( cd ){
            
            case '0' : ret = 0x01 ; break ;
            case '1' : ret = 0x02 ; break ;
            case '2' : ret = 0x03 ; break ;
            case '3' : ret = 0x04 ; break ;
            case '4' : ret = 0x05 ; break ;
            case '5' : ret = 0x06 ; break ;
            case '6' : ret = 0x07 ; break ;
            case '7' : ret = 0x08 ; break ;
            case '8' : ret = 0x09 ; break ;
            case '9' : ret = 0x0a ; break ;
            case '+' : ret = 0x0b ; break ;
            case '-' : ret = 0x0c ; break ;
            case '.' : ret = 0x0d ; break ;
            
            default : ret = 0x0f ; break ;
        }
        
        return ret ;
        
    }
    
    /**
     * 数値変換 : 数値バイナリ変換コードから、文字コードに変換.
     */
    private static final char cnvNumBinByChar( byte cd )
    {
        
        char ret ;
        
        switch( cd ){
            
            case 0x00 : ret = ConvertResourceParam.ENDCHAR ; break ;
            case 0x01 : ret = '0' ; break ;
            case 0x02 : ret = '1' ; break ;
            case 0x03 : ret = '2' ; break ;
            case 0x04 : ret = '3' ; break ;
            case 0x05 : ret = '4' ; break ;
            case 0x06 : ret = '5' ; break ;
            case 0x07 : ret = '6' ; break ;
            case 0x08 : ret = '7' ; break ;
            case 0x09 : ret = '8' ; break ;
            case 0x0a : ret = '9' ; break ;
            case 0x0b : ret = '+' ; break ;
            case 0x0c : ret = '-' ; break ;
            case 0x0d : ret = '.' ; break ;
            
            default : ret = '?' ; break ;
            
        }
        
        return ret ;
        
    }
    
}

