/*
 * @(#)SectorDataFile.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource.cache;

import java.io.RandomAccessFile;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.io.IOCom;


/**
 * セクターデータファイル領域.
 *  
 * @version 2006/08/02
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class SectorDataFile {
    
    /**
     * ファイルポインタ.
     */
    private RandomAccessFile filePointer = null ;
    
    /**
     * セクタサイズ.
     */
    private int maxSector = -1 ;
    
    /**
     * ファイル名.
     */
    private String fileName = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private SectorDataFile() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名と、オープンセクタサイズを設定してファイルオープンします.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @param size オープンセクタサイズを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public SectorDataFile( String name,int size )
        throws InputException,AccessException {
        
        long len ;
        
        if (
            name == null || name.length() <= 0 ||
            size <= 0 || size > SectorIDManage.MAX_SECTOR
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            len = ( long )size * SectorIDManage.SECTOR_LENGTH ;
            
            if ( IOCom.isFileExists( name ) == false ) {
                filePointer = new RandomAccessFile( name,"rw" ) ;
                filePointer.setLength( len ) ;
            }
            else {
                filePointer = new RandomAccessFile( name,"rw" ) ;
                if ( len != filePointer.length() ) {
                    filePointer.setLength( len ) ;
                }
            }
            
            maxSector = size ;
            fileName = name ;
            
        } catch ( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.destroy() ;
        } catch ( Exception t ) {
            
        }
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        if( filePointer != null ){
            
            try{
                
                filePointer.close() ;
                
            } catch ( Exception e ) {
                
            }
            
        }
        
        filePointer = null ;
        maxSector = -1 ;
        
    }
    
    /**
     * セクタ項番と読み込みバイト数を指定して情報取得.
     * <BR><BR>
     * セクタ項番と読み込みバイト数を指定して情報を取得します.
     * <BR>
     * @param out 取得情報を格納するバイナリ情報を設定します.
     * @param sectorNo セクタ位置を設定します.
     * @param sectorOffset セクタ位置からの読み込み開始位置を設定します.
     * @param offset 指定バイナリの開始値を設定します.
     * @param length 対象の読み込み数を設定します
     * @return int 取得されたバイナリ領域が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public int read( byte[] out,int sectorNo,int sectorOffset,int offset,int length )
        throws InputException,AccessException {
        
        long sp ;
        int ret ;
        
        if (
            out == null || out.length <= 0 ||
            sectorNo < 0 || sectorNo > maxSector ||
            sectorOffset >= ( int )SectorIDManage.SECTOR_LENGTH ||
            length <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( filePointer != null ) {
            
            sectorOffset = ( sectorOffset <= 0 ) ? 0 : sectorOffset ;
            length = ( length + offset >= out.length ) ?
                out.length - offset : length ;
            
            sp = ( long )( sectorNo * SectorIDManage.SECTOR_LENGTH ) +
                ( long )sectorOffset ;
            
            try {
                
                filePointer.seek( sp ) ;
                ret = filePointer.read( out,offset,length ) ;
                
            } catch ( Exception e ) {
                throw new AccessException( e ) ;
            }
            
        }
        else {
            
            ret = -1 ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * セクタ項番と書き込みバイト数を指定して情報書き込み.
     * <BR><BR>
     * セクタ項番と書き込みバイト数を指定して情報を書き込みます.
     * <BR>
     * @param in 書き込み対象のバイナリ情報を設定します.
     * @param sectorNo セクタ位置を設定します.
     * @param sectorOffset セクタ位置からの書き込み開始位置を設定します.
     * @param offset 指定バイナリの開始値を設定します.
     * @param length 書き込みバイト数を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void write( byte[] in,int sectorNo,int sectorOffset,int offset,int length )
        throws InputException,AccessException {
        
        long sp ;
        
        if (
            in == null || in.length <= 0 ||
            sectorNo < 0 || sectorNo > maxSector ||
            sectorOffset >= ( int )SectorIDManage.SECTOR_LENGTH ||
            length <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( filePointer != null ) {
            
            sectorOffset = ( sectorOffset <= 0 ) ? 0 : sectorOffset ;
            length = ( length + offset >= in.length ) ?
                in.length - offset : length ;
            
            sp = ( long )( sectorNo * SectorIDManage.SECTOR_LENGTH ) +
                ( long )sectorOffset ;
            
            try {
                
                filePointer.seek( sp ) ;
                filePointer.write( in,offset,length ) ;
                
            } catch ( Exception e ) {
                throw new AccessException( e ) ;
            }
            
        }
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * オープンされているファイル名を取得します.
     * <BR>
     * @return String ファイル名が返されます.<BR>
     *                [null]が返された場合、ファイルはオープンされていません.
     */
    public String getFileName() {
        
        return fileName ;
        
    }
    
    
    /**
     * オブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されえます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public boolean isUseObject() {
       
        if ( filePointer != null ) {
            return true ;
        }
        else {
            return false ;
        }
        
    }
    
}

