/*
 * @(#)SectorIDManage.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource.cache ;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.NumberTable;
import org.maachang.commons.util.array.IntArray;


/**
 * セクターID管理オブジェクト.
 * <BR><BR>
 * 各セクターのID管理を行うオブジェクトです.
 *
 * @version 2006/07/28
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class SectorIDManage implements Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            SectorIDManage.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * １セクタ情報.
     */
    public static final int SECTOR_LENGTH = CacheDef.SECTOR_LENGTH ;
    
    /**
     * 最低セクター数.
     */
    public static final int MIN_SECTOR = CacheDef.MIN_SECTOR ;
    
    /**
     * 最大セクター数.
     */
    public static final int MAX_SECTOR = CacheDef.MAX_SECTOR ;
    
    
    
    /**
     * セクターON/OFF管理用.
     */
    private UseSectorFlags useSectorFlags = null ;
    
    /**
     * 利用IDに対するセクタ位置管理用.
     */
    private NumberTable useIDBySector = null ;
    
    /**
     * コンストラクタ.
     */
    private SectorIDManage() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * セクター数を設定して、オブジェクトを生成します.
     * <BR>
     * @param size 対象のセクター数を設定します.<BR>
     *             設定可能な最小値は[CacheDef.MIN_SECTOR]です.<BR>
     *             設定可能な最大値は[CacheDef.MAX_SECTOR]です.
     */
    public SectorIDManage( int size ) {
        
        if ( size <= MIN_SECTOR ) {
            size = MIN_SECTOR ;
        }
        else if ( size >= MAX_SECTOR ) {
            size = MAX_SECTOR ;
        }
        
        try {
            
            useSectorFlags = new UseSectorFlags( size ) ;
            size = useSectorFlags.getMaxSize() ;
            useIDBySector = new NumberTable() ;
            
        } catch ( Exception e ) {
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.destroy() ;
        } catch ( Exception t ) {
            
        }
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        if ( useSectorFlags != null ) {
            useSectorFlags.clear() ;
        }
        
        if ( useIDBySector != null ) {
            useIDBySector.clear() ;
        }
        
        useSectorFlags = null ;
        useIDBySector = null ;
        
    }
    
    /**
     * 空きセクタを取得して、利用IDで予約.
     * <BR><BR>
     * 空きセクタを取得して、利用IDで予約します.
     * <BR>
     * @param useID 予約対象の利用IDを設定します.
     * @return int 予約されたセクタ位置が返されます.
     * @exception InputException 入力例外.
     * @exception NotUseSectorException 有効なセクタが存在しない場合の例外.
     */
    public int searchByReservationOneSector( int useID )
        throws InputException,NotUseSectorException {
        
        int ret = -1 ;
        
        IntArray ary = null ;
        
        if ( useID < 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( useSectorFlags != null ) {
            
            if ( ( ret = useSectorFlags.useSearchSector() ) == -1 ) {
                
                throw new NotUseSectorException( "セクタ空き領域は存在しません" ) ;
                
            }
            
            if ( ( ary = ( IntArray )useIDBySector.get( useID ) ) == null ) {
                
                ary = new IntArray() ;
                useIDBySector.add( useID,ary ) ;
                
            }
            
            useSectorFlags.put( true,ret ) ;
            ary.add( ret ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 空きセクタを取得して、利用IDで複数セクタを予約.
     * <BR><BR>
     * 空きセクタを取得して、利用IDで複数セクタを予約します.
     * <BR>
     * @param useID 予約対象の利用IDを設定します.
     * @param reservationSize 予約したいセクタ数を設定します.
     * @return int 予約されたセクタ数が返されます.<BR>
     *             [0]が返された場合、空きセクタは存在しません.
     * @exception InputException 入力例外.
     */
    public int searchByReservationSector( int useID,int reservationSize )
        throws InputException {
        
        int i ;
        int index ;
        
        int code ;
        int ret = 0 ;
        
        IntArray ary = null ;
        
        if ( useID < 0 || reservationSize <= 0 || reservationSize > MAX_SECTOR ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( useSectorFlags != null ) {
            
            for( i = 0,ret = 0,index = -1 ; i < reservationSize ; i ++ ){
                
                if ( ( code = useSectorFlags.useSearchSector( index ) ) == -1 ) {
                    
                    break ;
                    
                }
                
                if( i == 0 ){
                    if ( ( ary = ( IntArray )useIDBySector.get( useID ) ) == null ) {
                        
                        ary = new IntArray() ;
                        useIDBySector.add( useID,ary ) ;
                        
                    }
                }
                
                useSectorFlags.put( true,code ) ;
                ary.add( code ) ;
                ret ++ ;
                index = code ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * セクタ位置に対して利用IDで予約解除.
     * <BR><BR>
     * 指定したセクタ位置に対して、利用IDで予約解除します.
     * <BR>
     * @param useID 解除対象の利用IDを設定します.
     * @param no 解除対象の項番を設定します.
     * @exception InputException 入力例外.
     */
    public void releaseOneSector( int useID,int no )
        throws InputException {
        
        int sectorNo ;
        
        IntArray ary = null ;
        
        if ( useID < 0 || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( useSectorFlags != null ) {
            
            try {
                
                if( ( ary = ( IntArray )useIDBySector.get( useID ) ) != null ){
                    
                    sectorNo = ary.remove( no ) ;
                    
                    if( ary.size() <= 0 ){
                        useIDBySector.remove( useID ) ;
                    }
                    if( sectorNo < Integer.MAX_VALUE ){
                        useSectorFlags.put( false,sectorNo ) ;
                    }
                    
                }
                
            } catch ( Exception e ) {
            }
            
        }
        
    }
    
    /**
     * 利用IDに対する、予約セクタを全て解除.
     * <BR><BR>
     * 利用IDに対する、予約されているセクタを全て解除します.
     * <BR>
     * @param useID 解除対象の利用IDを設定します.
     * @exception InputException 入力例外.
     */
    public void releaseUseIDByAllSector( int useID )
        throws InputException
    {
        int i ;
        int len ;
        int sectorNo ;
        
        IntArray ary = null ;
        
        if ( useID < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( useSectorFlags != null ) {
            
            if ( ( ary = ( IntArray )useIDBySector.get( useID ) ) != null ) {
                
                len = ary.size() ;
                for ( i = 0 ; i < len ; i ++ ) {
                    
                    sectorNo = ary.get( i ) ;
                    useSectorFlags.put( false,sectorNo ) ;
                    
                }
                
                useIDBySector.remove( useID ) ;
                
            }
            
        }
        
    }
    
    /**
     * 利用IDに対する、指定項番の予約セクタ位置を取得します.
     * <BR><BR>
     * 利用IDに対する、指定項番の予約されたセクタ位置を取得します.<BR>
     * また、この情報は予約された順番に格納されています.
     * <BR>
     * @param useID 対象の利用IDを設定します.
     * @param no 取得項番を設定します。
     * @return int 対象の予約セクタが返されます.<BR>
     *             [-1]が返された場合、情報の取得に失敗しました.
     */
    public int getReservationOneSector( int useID,int no ) {
        
        int ret = -1 ;
        
        IntArray ary = null ;
        
        if ( useID < 0 || no < 0 ) {
            return -1 ;
        }
        
        if ( useSectorFlags != null ) {
            
            try {
                
                if ( ( ary = ( IntArray )useIDBySector.get( useID ) ) != null ) {
                    
                    ret = ary.get( no ) ;
                    
                }
                
            } catch ( Exception e ) {
                
                ret = -1 ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対する、予約セクタ利用領域群を取得.
     * <BR><BR>
     * 利用IDに対する、予約されているセクタ利用領域群を取得します.
     * <BR>
     * @param useID 対象の利用IDを設定します.
     * @return int[] 利用IDに対する予約セクタ利用領域群が返されます.<BR>
     *               [null]が返された場合、利用IDに対する予約セクタは存在しません.
     */
    public int[] getReservationAllUse( int useID ) {
        
        int i ;
        int len ;
        
        IntArray ary = null ;
        int[] ret = null ;
        
        if ( useID < 0 ) {
            return null ;
        }
        
        if ( useSectorFlags != null ) {
            
            if (
                ( ary = ( IntArray )useIDBySector.get( useID ) ) != null &&
                ( len = ary.size() ) > 0
            ) {
                
                ret = new int[ len ] ;
                
                for ( i = 0 ; i < len ; i ++ ) {
                    
                    ret[ i ] = ary.get( i ) ;
                    
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対して、予約されているセクタ数を取得.
     * <BR><BR>
     * 利用IDに対して、予約されているセクタ数を取得します.
     * <BR>
     * @param useID 対象の利用IDを設定します.
     * @return int 利用IDに対する予約セクタ数が返されます.<BR>
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getReservationSectorSize( int useID ) {
        
        int ret = -1 ;
        
        IntArray ary = null ;
        
        if ( useID < 0 ){
            return -1 ;
        }
        
        if ( useSectorFlags != null ) {
            
            try {
                
                if( ( ary = ( IntArray )useIDBySector.get( useID ) ) != null ){
                    
                    ret = ary.size() ;
                    
                }
                
            } catch ( Exception e ) {
                
                ret = -1 ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 生成されている最大セクタ数を取得.
     * <BR><BR>
     * 生成されている最大セクタ数を取得します.
     * <BR>
     * @return int 生成されている最大セクタ数が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public int getMaxSector() {
        
        int ret ;
        
        try {
            ret = useSectorFlags.getMaxSize() ;
        } catch ( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在利用されているセクタ数を取得.
     * <BR><BR>
     * 現在利用されているセクタ数を取得します.
     * <BR>
     * @return int 現在利用されているセクタ数が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public int getUseSector() {
        
        int ret ;
        
        try {
            ret = useSectorFlags.getUseSize() ;
        } catch ( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * このオブジェクトが有効であるか取得.
     * <BR><BR>
     * このオブジェクトが有効であるか取得します.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、このオブジェクトは有効です.<BR>
     *                 [false]が返された場合、このオブジェクトは無効です.
     */
    public boolean isUseObject() {
        
        return ( useSectorFlags == null ) ? false : true ;
        
    }
    
}

