/*
 * @(#)SerializeUtil.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.serialize ;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import org.maachang.commons.conv.CodeBase32;
import org.maachang.commons.conv.SHA1;
import org.maachang.commons.conv.ScIO;
import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.io.IOCom;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.zip.GZIPBinary;


/**
 * シリアライズユーティリティ.
 *
 * @version 1.00, 2005/10/02
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class SerializeUtil
{
    
    /**
     * 暗号シリアライズヘッダ.
     */
    private static final byte[] C32SERI_HEADER = {
        0x73,0x33,0x32,0x53
    } ;
    
    /**
     * 対象名に対するserialVersionUIDを計算.
     * <BR><BR>
     * 対象名に対するserialVersionUIDを計算します.
     * <BR>
     * @param name 対象のクラス名を設定します.
     * @return long 対象のserialVersionUIDが返されます.
     */
    public static final long serialVersionUID( String name ) {
        
        int i ;
        int len ;
        long ret = 0 ;
        
        len = name.length() ;
        for( i = 0 ; i < len ; i ++ ) {
            
            ret = ( ( 31L * ret ) +
                ( long )( name.charAt( i ) & 0x0000ffff ) ) &
                    0x3fffffffffffffffL ;
            
        }
        
        return ret ;
    }
    
    /**
     * 対象ファイルにオブジェクトをシリアライズ.
     * <BR><BR>
     * 対象ファイルに対してオブジェクトをシリアライズ.
     * <BR>
     * @param name シリアライズ先のファイル名を設定します.
     * @param obj シリアライズ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void putSerialize( String name,Serializable obj )
        throws InputException,AccessException
    {
        ObjectOutputStream o = null ;
        
        if( name == null || name.length() <= 0 || obj == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            o = new ObjectOutputStream(
                new FileOutputStream( name )
            ) ;
            o.writeObject( obj ) ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            try{
                o.close() ;
            }catch( Exception ee ){
            }
            o = null ;
        }
    }
    
    /**
     * 対象ファイルから、オブジェクトを復元.
     * <BR><BR>
     * 対象のファイル名からオブジェクトを復元します.
     * <BR>
     * @param name 復元対象のファイル名を設定します.
     * @return Serializable 復元されたオブジェクトが返されます.<BR>
     *                      復元に失敗した場合[null]が返されます.
     * @exception AccessException アクセス例外.
     */
    public static final Serializable getSerialize( String name )
        throws AccessException
    {
        ObjectInputStream in = null;
        Serializable ret = null ;
        
        if( name == null || name.length() <= 0 ){
            return ret ;
        }
        
        try{
            in = new ObjectInputStream(
                new FileInputStream( name )
            ) ;
            ret = ( Serializable )in.readObject() ;
            SerializeUtil.execInitSerialize( ret ) ;
            
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        
        return ret ;
    }
    
    /**
     * 暗号化シリアライズ.
     * <BR><BR>
     * 暗号化シリアライズ処理を行います.
     * <BR>
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name シリアライズ先のファイル名を設定します.
     * @param obj シリアライズ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void putCb32Serialize( boolean mode,String cb32,String name,Serializable obj )
        throws InputException,AccessException
    {
        int len ;
        int pnt ;
        int stepCd ;
        int rnd ;
        
        byte[] objBin = null ;
        byte[] defKey = null ;
        byte[] key = null ;
        byte[] seri = null ;
        
        CodeBase32 cb32Obj = null ;
        
        if( name == null || name.length() <= 0 || obj == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            objBin = SerializeUtil.convertObjectByBinary( obj ) ;
            defKey = SerializeUtil.getDefaultKey( mode,name ) ;
            key = SerializeUtil.getCb32Key( cb32 ) ;
            
            // GZIPで圧縮.
            objBin = GZIPBinary.getInstance().convertBinaryByGZIP( objBin ) ;
            
            // CB32で暗号処理.
            cb32Obj = new CodeBase32( defKey ) ;
            stepCd = cb32Obj.encryption( key,objBin,0,objBin.length ) ;
            cb32Obj = null ;
            defKey = null ;
            key = null ;
            
            rnd = UtilCom.random( 65535 ) ;
            
            // さらにScIOで暗号処理.
            ScIO.input( objBin,rnd,0,objBin.length ) ;
            
            // ヘッダ+ステップコードを設定.
            len = objBin.length ;
            pnt = C32SERI_HEADER.length ;
            seri = new byte[ len + 3 + pnt ] ;
            System.arraycopy( C32SERI_HEADER,0,seri,0,pnt ) ;
            seri[ pnt ] = ( byte )stepCd ;
            pnt += 1 ;
            seri[ pnt ] = ( byte )( rnd & 0x000000ff ) ;
            seri[ pnt+1 ] = ( byte )( ( rnd & 0x0000ff00 ) >> 8 ) ;
            pnt += 2 ;
            System.arraycopy( objBin,0,seri,pnt,len ) ;
            objBin = null ;
            
            // 対象条件をファイル出力.
            IOCom.setFile( name,seri ) ;
            seri = null ;
            
        }catch( AccessException ae ){
            throw ae ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            objBin = null ;
            defKey = null ;
            key = null ;
            seri = null ;
            cb32Obj = null ;
        }
    }
    
    /**
     * 暗号化シリアライズオブジェクトを取得.
     * <BR><BR>
     * 暗号化シリアライズオブジェクトを取得します.
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name シリアライズ先のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Serializable getCb32Serialize( boolean mode,String cb32,String name )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        int pnt ;
        int stepCd ;
        int rnd ;
        
        byte[] bin = null ;
        byte[] seri = null ;
        byte[] defKey = null ;
        byte[] key = null ;
        
        CodeBase32 cb32Obj = null ;
        Serializable ret = null ;
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            bin = IOCom.getFile( name ) ;
            len = C32SERI_HEADER.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( bin[ i ] != C32SERI_HEADER[ i ] ){
                    throw new AccessException( "ファイルシンボルが不正です" ) ;
                }
            }
            
            defKey = SerializeUtil.getDefaultKey( mode,name ) ;
            key = SerializeUtil.getCb32Key( cb32 ) ;
            
            stepCd = ( int )( bin[ len ] & 0x000000ff ) ;
            pnt = len+1 ;
            rnd = ( int )(
                ( bin[ pnt ] & 0x000000ff ) |
                ( ( bin[ pnt + 1 ] & 0x000000ff ) << 8 )
            ) ;
            pnt += 2 ;
            
            len = bin.length - pnt ;
            seri = new byte[ len ] ;
            System.arraycopy( bin,pnt,seri,0,len ) ;
            
            // ScIOによる解析.
            ScIO.output( seri,rnd,0,len ) ;
            
            // CB32による解析.
            cb32Obj = new CodeBase32( defKey ) ;
            cb32Obj.analysis( key,stepCd,seri,0,len ) ;
            cb32Obj = null ;
            defKey = null ;
            key = null ;
            
            // GZIPによる解凍.
            seri = GZIPBinary.getInstance().convertGZIPByBinary( seri ) ;
            
            // バイナリをシリアライズ用オブジェクトに変換.
            ret = SerializeUtil.convertBinaryByObject( seri ) ;
            seri = null ;
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            bin = null ;
            seri = null ;
            defKey = null ;
            key = null ;
            cb32Obj = null ;
        }
        
        return ret ;
    }
    
    /**
     * シリアライズオブジェクトをコピー.
     * <BR><BR>
     * 対象のシリアライズオブジェクトをコピーします.
     * <BR>
     * @param obj コピー元のオブジェクトを設定します.
     * @return Serializable コピーされたオブジェクト情報が返されます.<BR>
     *                      [null]が返されてた場合、オブジェクトのコピーに失敗しました.
     * @exception IOException IO例外.
     * @exception ClassNotFoundException クラス非存在例外.
     */
    public static final Serializable copyObject( Serializable obj )
        throws IOException,ClassNotFoundException
    {
        byte[] tmp = null ;
        ByteArrayOutputStream bin = null ;
        ObjectInputStream in = null ;
        ObjectOutputStream out = null ;
        
        Object cp = null ;
        Serializable ret = null ;
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            out = new ObjectOutputStream( bin ) ;
            
            out.writeObject( obj ) ;
            out.close() ;
            out = null ;
            
            tmp = bin.toByteArray() ;
            
            in = new ObjectInputStream( new ByteArrayInputStream( tmp ) ) ;
            cp = in.readObject() ;
            
            in.close() ;
            in = null ;
            
            if( cp != null ){
                ret = ( Serializable )cp ;
                SerializeUtil.execInitSerialize( ret ) ;
            }
            else{
                ret = null ;
            }
            
        }catch( ClassNotFoundException cf ){
            ret = null ;
        }catch( IOException ie ){
            ret = null ;
        }finally{
            
            if( out != null ){
                try{
                    out.close() ;
                }catch( Exception e ){
                }
            }
            
            if( in != null ){
                try{
                    in.close() ;
                }catch( Exception e ){
                }
            }
            
            tmp = null ;
            bin = null ;
            in = null ;
            out = null ;
            cp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象シリアライズオブジェクトをバイナリに変換.
     * <BR><BR>
     * 対象シリアライズオブジェクトをバイナリに変換します.
     * <BR>
     * @param obj 対象のシリアライズオブジェクトを設定します.
     * @return byte[] 変換されたバイナリオブジェクトが返されます.<BR>
     *                [null]が返された場合バイナリ変換に失敗しました.
     */
    public static final byte[] convertObjectByBinary( Serializable obj )
    {
        ByteArrayOutputStream bin = null ;
        ObjectOutputStream out = null ;
        
        byte[] ret = null ;
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            out = new ObjectOutputStream( bin ) ;
            
            out.writeObject( obj ) ;
            out.close() ;
            out = null ;
            
            ret = bin.toByteArray() ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            
            if( out != null ){
                try{
                    out.close() ;
                }catch( Exception e ){
                }
            }
            
            bin = null ;
            out = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象バイナリをシリアライズオブジェクトに変換.
     * <BR><BR>
     * 対象バイナリをシリアライズオブジェクトに変換します.
     * <BR>
     * @param bin 対象のバイナリオブジェクトを設定します.
     * @return Serializable 変換されたオブジェクト情報が返されます.<BR>
     *                      [null]が返された場合オブジェクト変換に失敗しました.
     */
    public static final Serializable convertBinaryByObject( byte[] bin )
    {
        ObjectInputStream in = null ;
        Object cp = null ;
        Serializable ret = null ;
        
        try{
            
            in = new ObjectInputStream( new ByteArrayInputStream( bin ) ) ;
            cp = in.readObject() ;
            
            in.close() ;
            in = null ;
            
            if( cp != null ){
                ret = ( Serializable )cp ;
                SerializeUtil.execInitSerialize( ret ) ;
            }
            else{
                ret = null ;
            }
            
        }catch( ClassNotFoundException cf ){
            ret = null ;
        }catch( IOException ie ){
            ret = null ;
        }finally{
            
            if( in != null ){
                try{
                    in.close() ;
                }catch( Exception e ){
                }
            }
            
            in = null ;
            cp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ロードオブジェクトが[InitSerialize]属性の場合の処理.
     */
    private static final void execInitSerialize( Serializable o )
    {
        if( o == null ){
            return ;
        }
        
        if( o instanceof InitSerialize ){
            ( ( InitSerialize )o ).initSerializable() ;
        }
    }
    
    /**
     * デフォルトキーを生成.
     */
    private static final byte[] getDefaultKey( boolean mode,String name )
    {
        StringBuffer buf = null ;
        byte[] ret = null ;
        
        try{
            
            buf = new StringBuffer() ;
            
            buf.append( SerializeUtil.class.getName() ) ;
            buf.append( "@@" ) ;
            
            if( mode == true ){
                buf.append( name ) ;
                buf.append( "@" ) ;
            }
            else{
                buf.append( SerializeUtil.class.getName() ) ;
                buf.append( "$" ) ;
            }
            
            buf.append( BaseDef.LOGIN_USERNAME ) ;
            buf.append( "@" ) ;
            buf.append( BaseDef.JAVA_VERSION ) ;
            buf.append( "&" ) ;
            buf.append( BaseDef.JAVA_VM_NAME ) ;
            buf.append( "&" ) ;
            buf.append( BaseDef.OS_NAME ) ;
            buf.append( ":" ) ;
            buf.append( BaseDef.THIS_CHARSET ) ;
            
            ret = CodeBase32.convertStringByCode32Key( SHA1.convert( buf.toString().getBytes() ) ) ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 暗号キーを生成.
     */
    private static final byte[] getCb32Key( String cb32 )
    {
        byte[] ret = null ;
        
        try{
            if( cb32 == null ){
                ret = CodeBase32.convertStringByCode32Key( SerializeUtil.class.getName() ) ;
            }
            else{
                ret = CodeBase32.convertStringByCode32Key( SHA1.convert( cb32.getBytes() ) ) ;
            }
        }catch( Exception e ){
            if( cb32 != null ){
                try{
                    ret = CodeBase32.convertStringByCode32Key( SerializeUtil.class.getName() ) ;
                }catch( Exception ee ){
                    ret = null ;
                }
            }
        }
        
        return ret ;
    }
    
}

