/*
 * @(#)BaseThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;

/**
 * 基本スレッド処理.
 * <BR><BR>
 * 基本スレッド処理を実施します.<BR>
 * また、このオブジェクトはパッケージ内用です.
 *
 * @version     1.00, 2003/10/26
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
class BaseThread
{
    
    /**
     * スレッド破棄待機時間.
     */
    protected static final int DESTROY_WAIT = 500 ;
    
    
    
    /**
     * スレッド処理管理.
     */
    private volatile CoreThread m_thread = null ;
    
    /**
     * スレッド実施処理部.
     */
    private ProcessThread m_process = null ;
    
    /**
     * デーモンスレッド生成フラグ.
     */
    private boolean m_daemon = false ;
    
    /**
     * 生成元スレッド名.
     */
    private String m_srcName = null ;
    
    /**
     * 設定引数用オブジェクト.
     */
    private Object m_param = null ;
    
    /**
     * 同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public BaseThread()
    {}
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッド準備処理.
     * <BR><BR>
     * スレッドを実施するための準備処理を設定します.
     * このスレッドはデーモンスレッドとして生成されます.
     * <BR>
     * @param process スレッド処理定義が行われたオブジェクトを設定します.
     * @param param そのスレッドに対して渡される引数情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実行例外.
     */
    public final void create( ProcessThread process,Object param )
        throws InputException,ExecutionException
    {
        try{
            
            this.create( false,process,param,true ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
    }
    
    /**
     * スレッド準備処理.
     * <BR><BR>
     * スレッドを実施するための準備処理を設定します.
     * このスレッドはデーモンスレッドとして生成されます.
     * <BR>
     * @param noLog 非ログ出力フラグを設定します.
     * @param process スレッド処理定義が行われたオブジェクトを設定します.
     * @param param そのスレッドに対して渡される引数情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実行例外.
     */
    public final void create( boolean noLog,ProcessThread process,Object param )
        throws InputException,ExecutionException
    {
        try{
            
            this.create( noLog,process,param,true ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
    }
    
    /**
     * スレッド準備処理.
     * <BR><BR>
     * スレッドを実施するための準備処理を設定します.
     * <BR>
     * @param process スレッド処理定義が行われたオブジェクトを設定します.
     * @param param そのスレッドに対して渡される引数情報を設定します.
     * @param daemon デーモンスレッド設定を指定します.<BR>
     *               [true]を指定した場合、対象スレッドはデーモンスレッド
     *               として実施されます.<BR>
     *               [false]を指定した場合、対象スレッドはユーザスレッド
     *               として実施されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実行例外.
     */
    public final void create( ProcessThread process,Object param,boolean daemon )
        throws InputException,ExecutionException
    {
        try{
            
            this.create( false,process,param,daemon ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
    }
    
    /**
     * スレッド準備処理.
     * <BR><BR>
     * スレッドを実施するための準備処理を設定します.
     * <BR>
     * @param noLog 非ログ出力フラグを設定します.
     * @param process スレッド処理定義が行われたオブジェクトを設定します.
     * @param param そのスレッドに対して渡される引数情報を設定します.
     * @param daemon デーモンスレッド設定を指定します.<BR>
     *               [true]を指定した場合、対象スレッドはデーモンスレッド
     *               として実施されます.<BR>
     *               [false]を指定した場合、対象スレッドはユーザスレッド
     *               として実施されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実行例外.
     */
    public final void create( boolean noLog,ProcessThread process,Object param,boolean daemon )
        throws InputException,ExecutionException
    {
        CoreThread core = null ;
        
        if( process == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            core = new CoreThread( noLog,process,param,daemon ) ;
            
            m_thread = core ;
            m_srcName = Thread.currentThread().getName() ;
            
            m_process = process ;
            m_param = param ;
            m_daemon = daemon ;
            
        }catch( BaseException be ){
            
            this.clear() ;
            throw new ExecutionException( be ) ;
            
        }catch( Exception t ){
            
            this.clear() ;
            throw new ExecutionException( t ) ;
            
        }finally{
            core = null ;
        }
        
    }
    
    /**
     * スレッド情報クリア.
     * <BR><BR>
     * スレッド情報をクリアします.
     */
    public final void clear()
    {
        
        try{
            m_sync.clear() ;
        }catch( Exception t ){
        }
        try{
            this.exitThread() ;
        }catch( Exception t ){
        }
        
        m_thread = null ;
        m_srcName = null ;
        m_process = null ;
        m_param = null ;
        m_daemon = false ;
        
    }
    
    /**
     * スレッド実行処理.
     * <BR><BR>
     * スレッド処理を実行します.
     */
    public final void startThread()
    {
        int state ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                // スレッド実行.
                m_thread.startup() ;
                
            }
            
            // スレッド実行待ち.
            /*for( ;; ){
                
                state = this.getState() ;
                
                if(
                    state == ExecutionThread.STATE_NOT ||
                    state == ExecutionThread.STATE_EXECUTION
                )
                {
                    if( state == ExecutionThread.STATE_NOT ){
                        this.clear() ;
                    }
                    break ;
                }
                
                UtilCom.cpuCreate() ;
                
            }*/
        
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッド終了処理.
     * <BR><BR>
     * スレッド終了処理を実行します.<BR>
     * また、この処理は、[BaseThread#startThread()]で生成／動作した、
     * 同一のスレッドで実施した場合有効となりません.
     */
    public final void exitThread()
    {
        
        String name = null ;
        try{
            
            if( m_thread == null ){
                
                try{
                    m_sync.clear() ;
                }catch( Exception t ){
                }
                
                m_thread = null ;
                m_srcName = null ;
                m_process = null ;
                m_param = null ;
                m_daemon = false ;
                
                return ;
            }
            
            synchronized( m_sync.get() ){
                
                name = m_thread.getThreadName() ;
                
                // 対象スレッドがThreadManagerに存在しない、
                // 又は終了処理対象のスレッドが動作対象スレッドで
                // ない場合.
                if(
                    name == null ||
                    name.equals( Thread.currentThread().getName() ) == false
                )
                {
                    
                    m_thread.cansell() ;
                    
                }
                
            }
        
        }catch( Exception t ){
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * スレッドを破棄.
     * <BR><BR>
     * 現在実行中のスレッドを破棄します.
     */
    protected final void destroyThread()
    {
        try{
            synchronized( m_sync.get() ){
                m_thread.stop() ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 対象実施オブジェクト名を設定.
     * <BR><BR>
     * 対象実施オブジェクト名を設定します.
     * <BR>
     * @param name 対象実施オブジェクト名を設定します.
     */
    protected final void setObjectName( String name )
    {
        try{
            synchronized( m_sync.get() ){
                m_thread.setObjectName( name ) ;
            }
        }catch( Exception t ){
            m_thread.setObjectName( null ) ;
        }
    }
    
    /**
     * スレッドプライオリティを設定.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを設定します.
     * <BR>
     * @param newPriority 新しいスレッドプライオリティを設定します.
     * @exception InputException 入力例外.
     */
    public final void setPriority( int newPriority )
        throws InputException
    {
        if(
            newPriority < Thread.MIN_PRIORITY ||
            newPriority > Thread.MAX_PRIORITY
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_thread.setPriority( newPriority ) ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 対象のスレッド名の取得.
     * <BR><BR>
     * 対象のスレッド名を取得します.
     * <BR>
     * @return String 対象のスレッド名が返されます.<BR>
     *                スレッドが実施されてない場合[null]が返されます.
     */
    public final String getName()
    {
        String ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadName() ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド生成元のスレッド名を取得.
     * <BR><BR>
     * スレッド生成元のスレッド名を取得します.
     * <BR>
     * @return String 生成元スレッド名が返されます.
     */
    private final String getSrcName()
    {
        String ret = null ;
        
        if( m_thread == null ){
            return null ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_srcName ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getThreadState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadState() ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドプライオリティを取得.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを取得します.
     * <BR>
     * @return int 設定されているスレッドプライオリティが返されます.<BR>
     *             スレッドが存在しない場合[-1]が返されます.
     */
    public final int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getPriority() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド生成チェック.
     * <BR><BR>
     * スレッドが生成されているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、生成されています.
     *                 [false]が返された場合、生成されていません.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( m_process == null ) ? false : true ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行チェック.
     * <BR><BR>
     * スレッド情報が実行されたかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、スレッド情報は動作しています.<BR>
     *                 [false]が返された場合、スレッド情報は動作していません.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 生成スレッドモード取得.
     * <BR><BR>
     * 生成されたスレッドのモードを取得します.
     * <BR>
     * @return boolean 生成されたスレッドモードが返されます.<BR>
     *                 [true]が返された場合、生成されたスレッドは
     *                 デーモンスレッドです.<BR>
     *                 [false]が返された場合、生成されたスレッドは
     *                 ユーザスレッドか、生成されていません.
     */
    public final boolean isMode()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_daemon ;
            }
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
}

