/*
 * @(#)CoreThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.UtilCom;


/**
 * コアスレッド実装.
 * <BR><BR>
 * スレッドでのコア処理を実施します.<BR>
 * また、このオブジェクトはパッケージ内用です.
 *
 * @version     1.00, 2003/10/26
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
class CoreThread extends Thread
{
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( CoreThread.class ) ;
    
    
    
    /**
     * スレッド実施インターフェイス.
     */
    private ProcessThread m_process = null ;
    
    /**
     * スレッドセット引数.
     */
    private Object m_in = null ;
    
    /**
     * 対象スレッド名.
     */
    private String m_name = null ;
    
    /**
     * 実施スレッドオブジェクト名.
     */
    private volatile String m_objectName = null ;
    
    /**
     * スレッド実施有無確認フラグ.
     */
    private boolean m_isThread = false ;
    
    /**
     * スレッド終了管理フラグ.
     */
    private boolean m_isExitThread = true ;
    
    /**
     * スレッド実行処理例外.
     */
    private ExecutionException m_err = null ;
    
    /**
     * スレッド開始フラグ.
     */
    private volatile boolean m_startup = false ;
    
    /**
     * スレッドカウントフラグ.
     */
    private volatile boolean m_tcntFlg = false ;
    
    /**
     * スレッドログ非出力フラグ.
     */
    private volatile boolean m_noLogFlg = false ;
    
    /**
     * スレッドステータス.
     */
    private volatile int m_state = ExecutionThread.STATE_NOT ;
    
    
    
    /**
     * 同期処理用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * コンストラクタです.
     */
    private CoreThread()
    {
        
    }
    
    /**
     * スレッド初期化.
     * <BR><BR>
     * スレッド処理での初期化処理を実施します.<BR>
     * また、このコンストラクタでは、スレッドモードは
     * 「ユーザスレッド」となります.
     * <BR>
     * @param process スレッド実行を行う処理を指定します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public CoreThread( ProcessThread process,Object in )
        throws InputException,AccessException
    {
        this( false,process,in ) ;
    }
    
    /**
     * スレッド初期化.
     * <BR><BR>
     * スレッド処理での初期化処理を実施します.<BR>
     * また、このコンストラクタでは、スレッドモードは
     * 「ユーザスレッド」となります.
     * <BR>
     * @param noLog 非ログ出力フラグを設定します.
     * @param process スレッド実行を行う処理を指定します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public CoreThread( boolean noLog,ProcessThread process,Object in )
        throws InputException,AccessException
    {
        //super( CoreThread.createThreadName() ) ;
        
        String name = null ;
        try{
            
            if( process == null ){
                
                throw new InputException( "引数が不正です" ) ;
            }
            
            m_sync.create() ;
            name = this.getName() ;
            this.checkAccess() ;
            this.setDaemon( false ) ;
            
            synchronized( m_sync.get() ){
                
                m_state = ExecutionThread.STATE_CREATE ;
                m_process = process ;
                m_in = in ;
                m_startup = false ;
                m_name = name ;
                m_noLogFlg = noLog ;
                
            }
            
            ThreadManager.getInstance().initMethod( this ) ;
            
            synchronized( m_sync.get() ){
                m_state = ExecutionThread.STATE_STARTUP ;
            }
            
            this.start() ;
            
        }catch( SecurityException se ){
            
            this.clear() ;
            name = null ;
            throw new AccessException( se ) ;
            
        }catch( InputException ie ){
            
            this.clear() ;
            name = null ;
            throw ie ;
            
        }catch( Exception t ){
            
            this.clear() ;
            name = null ;
            throw new AccessException( t ) ;
            
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * スレッド初期化.
     * <BR><BR>
     * スレッド処理での初期化処理を実施します.
     * <BR>
     * @param process スレッド実行を行う処理を指定します.
     * @param in スレッド処理での引数情報を指定します.
     * @param daemon デーモンスレッド設定を指定します.<BR>
     *               [true]を指定した場合、対象スレッドはデーモンスレッド
     *               として実施されます.<BR>
     *               [false]を指定した場合、対象スレッドはユーザスレッド
     *               として実施されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public CoreThread( ProcessThread process,Object in,boolean daemon )
        throws InputException,AccessException
    {
        this( false,process,in,daemon ) ;
    }
    
    /**
     * スレッド初期化.
     * <BR><BR>
     * スレッド処理での初期化処理を実施します.
     * <BR>
     * @param noLog 非ログ出力フラグを設定します.
     * @param process スレッド実行を行う処理を指定します.
     * @param in スレッド処理での引数情報を指定します.
     * @param daemon デーモンスレッド設定を指定します.<BR>
     *               [true]を指定した場合、対象スレッドはデーモンスレッド
     *               として実施されます.<BR>
     *               [false]を指定した場合、対象スレッドはユーザスレッド
     *               として実施されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public CoreThread( boolean noLog,ProcessThread process,Object in,boolean daemon )
        throws InputException,AccessException
    {
        //super( CoreThread.createThreadName() ) ;
        String name = null ;
        
        try{
            
            if( process == null ){
                
                throw new InputException( "引数が不正です" ) ;
            }
            
            m_sync.create() ;
            name = this.getName() ;
            this.checkAccess() ;
            this.setDaemon( daemon ) ;
            
            synchronized( m_sync.get() ){
                
                m_state = ExecutionThread.STATE_CREATE ;
                m_process = process ;
                m_in = in ;
                m_startup = false ;
                m_name = name ;
                m_noLogFlg = noLog ;
                
            }
            
            ThreadManager.getInstance().initMethod( this ) ;
            
            synchronized( m_sync.get() ){
                m_state = ExecutionThread.STATE_STARTUP ;
            }
            
            this.start() ;
            
        }catch( SecurityException se ){
            
            this.clear() ;
            name = null ;
            throw new AccessException( se ) ;
            
        }catch( InputException ie ){
            
            this.clear() ;
            name = null ;
            throw ie ;
            
        }catch( Exception t ){
            
            this.clear() ;
            name = null ;
            throw new AccessException( t ) ;
            
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 処理クリア.
     * <BR><BR>
     * 対象のスレッド処理をクリアします.
     * この処理は、本質的に安全な処理ではなく、スレッドを強制的にストップ
     * させることにより、強制終了させます.
     * そのため、基本的なスレッド破棄に対しては、[CoreThread#cansell()]を実施する
     * ことを進めます.
     */
    public final void clear()
    {
        
        try{
            this.cansell() ;
        }catch( Exception t ){
        }
        
        m_sync.clear() ;
        
        m_process = null ;
        m_in = null ;
        m_isThread = false ;
        m_isExitThread = true ;
        m_err = null ;
        m_name = null ;
        m_objectName = null ;
        m_state = ExecutionThread.STATE_NOT ;
        
        ThreadManager.getInstance().clear( this ) ;
        
    }
    
    /**
     * スレッド開始処理.
     * <BR><BR>
     * スレッド処理を開始します.
     */
    public final void startup()
    {
        try{
            synchronized( m_sync.get() ){
                m_startup = true ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * スレッド破棄.
     * <BR><BR>
     * スレッド情報を破棄します.
     */
    public final void cansell()
    {
        StringBuffer buf = null ;
        
        try{
            
            this.outEnd() ;
            m_objectName = null ;
            
        }catch( Exception t ){
        }finally{
            buf = null ;
        }
        
        ThreadManager.getInstance().exitThread( this ) ;
    }
    
    /**
     * 対象実施オブジェクト名を設定.
     * <BR><BR>
     * 対象実施オブジェクト名を設定します.
     * <BR>
     * @param name 対象実施オブジェクト名を設定します.
     */
    protected final void setObjectName( String name )
    {
        try{
            synchronized( m_sync.get() ){
                m_objectName = name ;
            }
        }catch( Exception t ){
            m_objectName = null ;
        }
    }
    
    /**
     * スレッド例外情報取得.
     * <BR><BR>
     * スレッド処理実施での例外処理を取得します.
     * ただし、スレッドが[this#cancell()]メソッドで忠実に
     * スレッド破棄されている場合、以下の条件となります.<BR>
     * [ExecutionException#getLevel() == ExecutionException.LEVEL_STOP ].
     * <BR>
     * @return err スレッド例外処理が返されます.<BR>
     *             例外処理が存在しない場合、[null]が返されます.
     */
    public final ExecutionException getThreadException()
    {
        ExecutionException ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_err ;
                m_err = null ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド名の取得.
     * <BR><BR>
     * 対象のスレッド名を取得します.
     * <BR>
     * @return String 対象のスレッド名が返されます.
     */
    protected final String getThreadName()
    {
        String ret = null ;
        try{
            synchronized( m_sync.get() ){
                ret = m_name ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getThreadState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_state ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行有無取得.
     * <BR><BR>
     * スレッド処理が実施されているかどうかの実行有無を取得します.
     * <BR>
     * @return boolean スレッド実行有無が返されます.<BR>
     *                 [true]が返された場合、スレッドが実施されています.<BR>
     *                 [false]が返された場合、スレッドは実施されていません.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_isThread ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド終了有無取得.
     * <BR><BR>
     * スレッド処理開始後のスレッド処理終了確認を行う場合の情報を取得します.
     * <BR>
     * @return boolean スレッド実行終了確認を行います.<BR>
     *                 [true]が返された場合、スレッド処理は終了しました.
     *                 [false]が返された場合、スレッド処理は実施されていないか、
     *                 終了していません.
     */
    public final boolean isExitThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_isExitThread ;
            }
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド例外発生チェック.
     * <BR><BR>
     * スレッド実施で、例外が発生したかチェックします.
     * <BR>
     * @return boolean 例外が発生したかチェックします.<BR>
     *                 [true]が返された場合、例外が発生しています.
     *                 [false]が返された場合、例外が発生していません.
     */
    public final boolean isException()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_err != null ) ? true : false ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド処理実施.
     * <BR><BR>
     * スレッド処理実施を行います.
     */
    public final void run()
    {
        boolean checkMethod ;
        
        StringBuffer buf = null ;
        ProcessThread process = null ;
        Object input = null ;
        
        checkMethod = false ;
        
        this.outStart() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_process == null ){
                    throw new ExecutionException( "実施対象の処理が存在しません" ) ;
                }
                
                // スレッドID登録.
                ThreadManager.getInstance().initMethod() ;
                
                m_state = ExecutionThread.STATE_START ;
                m_isThread = false ;
                m_isExitThread = false ;
                
            }
            
            // スレッドID登録チェックON.
            checkMethod = true ;
            
            // スタートアップ処理待ち.
            for( ;; ){
                
                synchronized( m_sync.get() ){
                    
                    m_state = ExecutionThread.STATE_WAIT_EXECUITON ;
                    
                    if( m_startup == true ){
                        break ;
                    }else if( m_isExitThread == true ){
                        return ;
                    }
                    
                }
                
                UtilCom.idleTime() ;
                
            }
            
            synchronized( m_sync.get() ){
                
                m_state = ExecutionThread.STATE_EXECUTION ;
                process = ( ProcessThread )m_process ;
                input = ( Object )m_in ;
                m_err = null ;
                m_isThread = true ;
                
                // スレッド起動.
                ThreadManager.getInstance().startMethod( this ) ;
                
            }
            
            // スレッドID登録チェックOFF.
            checkMethod = false ;
            
            // スレッド実施処理開始.
            process.processThread( input ) ;
            
        }catch( OutOfMemoryError mem ) {
            LOG.error( "## OutOfMemoryError",mem) ;
        }catch( ThreadDeath death ){
            
            LOG.debug( "## ThreadExit-threadDeth",death ) ;
            //LOG.debug( "## ThreadExit-threadDeth" ) ;
            
            /////////////////////////////
            // スレッド破棄時の処理設定.
            /////////////////////////////
            try{
                process.destroyThread( m_in ) ;
            }catch( NullPointerException nue ){
            }catch( Exception t ){
                CoreThread.LOG.error( t.getMessage(),t ) ;
            }
            
            this.outDethEnd() ;
            throw death ;
            
        }catch( ExecutionException ex ){
            
            //LOG.debug( "## ThreadExit-ExecutionException",ex ) ;
            //LOG.debug( "## ThreadExit-ExecutionException" ) ;
            
            try{
                synchronized( m_sync.get() ){
                    m_err = ex ;
                }
            }catch( Exception tt ){
                m_err = null ;
            }
            
        }catch( Exception t ){
            
            LOG.debug( "## ThreadExit-Exception",t ) ;
            //LOG.debug( "## ThreadExit-Exception" ) ;
            
            try{
                synchronized( m_sync.get() ){
                    if( t instanceof NullPointerException ){
                        m_err = new ExecutionException(
                            t,ExecutionException.LEVEL_NORMAL ) ;
                    }else{
                        m_err = new ExecutionException(
                            t,ExecutionException.LEVEL_DETECTION_BUG ) ;
                    }
                }
            }catch( Exception tt ){
            }
            
        }finally{
            
            //////////////////////////////
            // スレッドマネージャから破棄.
            //////////////////////////////
            
            this.outEnd() ;
            
            m_objectName = null ;
            
            // スレッドID登録から、スレッド登録間での間にエラーが
            // 発生した場合の処理.
            if( checkMethod == true ){
                ThreadManager.getInstance().initMethodByError() ;
            }
            
            ThreadManager.getInstance().clear( this ) ;
            
            try{
                
                synchronized( m_sync.get() ){
                    
                    if(
                        m_err != null && process != null &&
                        m_err.getLevel() != ExecutionException.LEVEL_STOP
                    )
                    {
                        process.toException( ( ExecutionException )m_err ) ;
                    }
                    
                    m_isThread = false ;
                    m_isExitThread = true ;
                    m_startup = false ;
                    
                    process = null ;
                    input = null ;
                    buf = null ;
                    
                    m_process = null ;
                    m_in = null ;
                    
                }
                
            }catch( Exception tt ){
                
                m_isThread = false ;
                m_isExitThread = true ;
                m_startup = false ;
                
                process = null ;
                input = null ;
                buf = null ;
                
                m_process = null ;
                m_in = null ;
                
            }
            
            
        }
        
        
    }
    
    /**
     * 開始スレッド情報を出力.
     */
    private synchronized final void outStart()
    {
        StringBuffer buf = null ;
        
        if( m_noLogFlg == false && m_tcntFlg == false ){
            
            ThreadManager.getInstance().addThreadCount() ;
            m_tcntFlg = true ;
            
            // スレッド起動条件を登録.
            if( m_objectName != null ){
                
                // スレッド開始ログを出力.
                buf = new StringBuffer() ;
                buf.append( " ++ 起動(" ) ;
                buf.append( ThreadManager.getInstance().getThreadCount() ) ;
                buf.append( "):スレッド名( " ) ;
                buf.append( m_name ) ;
                buf.append( " ) オブジェクト名( " ) ;
                buf.append( m_objectName ) ;
                buf.append( " )" ) ;
                
                //System.out.println( buf.toString() ) ;
                CoreThread.LOG.info( buf.toString() ) ;
                
                buf = null ;
            }
        }
        
    }
    
    /**
     * 終了スレッド情報を出力.
     */
    private synchronized final void outEnd()
    {
        StringBuffer buf = null ;
        
        if( m_noLogFlg == false && m_tcntFlg == true ){
            
            ThreadManager.getInstance().removeThreadCount() ;
            m_tcntFlg = false ;
            
            // スレッド終了条件を登録.
            if( m_objectName != null ){
                
                // スレッド終了ログを出力.
                buf = new StringBuffer() ;
                buf.append( " -- 停止(" ) ;
                buf.append( ThreadManager.getInstance().getThreadCount() ) ;
                buf.append( "):スレッド名( " ) ;
                buf.append( m_name ) ;
                buf.append( " ) オブジェクト名( " ) ;
                buf.append( m_objectName ) ;
                buf.append( " )" ) ;
                
                //System.out.println( buf.toString() ) ;
                CoreThread.LOG.info( buf.toString() ) ;
                
                buf = null ;
            }
        }
    }
    
    /**
     * Thread#stop()によるスレッド情報を出力.
     */
    private synchronized final void outDethEnd()
    {
        StringBuffer buf = null ;
        
        if( m_noLogFlg == false && m_tcntFlg == true ){
            
            ThreadManager.getInstance().removeThreadCount() ;
            m_tcntFlg = false ;
            
            // スレッド終了条件を登録.
            if( m_objectName != null ){
                
                // スレッド終了ログを出力.
                buf = new StringBuffer() ;
                buf.append( " ## 破棄(" ) ;
                buf.append( ThreadManager.getInstance().getThreadCount() ) ;
                buf.append( "):スレッド名( " ) ;
                buf.append( m_name ) ;
                buf.append( " ) オブジェクト名( " ) ;
                buf.append( m_objectName ) ;
                buf.append( " )" ) ;
                
                //System.out.println( buf.toString() ) ;
                CoreThread.LOG.info( buf.toString() ) ;
                
                buf = null ;
            }
        }
    }
    
    
//  /**
//   * ０埋めデータ値.
//   */
//  private static final String ZERO_DAT = "0000000" ;
//  
//  /**
//   * スレッド頭文字.
//   */
//  private static final String THREAD_HEADER_NAME = "JRcCommons@<" ;
//  
//  /**
//   * スレッド終了文字.
//   */
//  private static final String THREAD_END_NAME = ">" ;
//  
//  /**
//   * スレッド名ID最小値.
//   */
//  private static final int MIN_THREAD_NAME_ID = 1 ;
//  
//  /**
//   * スレッド名ID最大値.
//   */
//  private static final int MAX_THREAD_NAME_ID = 9999999 ;
//  
//  /**
//   * スレッド名ID管理.
//   */
//  private static final SequenceID SEQ_ID = new SequenceID(
//      MIN_THREAD_NAME_ID,MAX_THREAD_NAME_ID
//  ) ;
//  
//  /**
//   * スレッド名をシーケンスIDにより生成.
//   */
//  private static final String createThreadName()
//  {
//      String ret = null ;
//      String idName = null ;
//      StringBuffer buf = null ;
//      Synchronized sync = null ;
//      
//      sync = ThreadManager.getInstance().getSync() ;
//      
//      try{
//          
//          for( ;; ){
//              
//              buf = null ;
//              buf = new StringBuffer() ;
//              
//              buf.append( THREAD_HEADER_NAME ) ;
//              
//              synchronized( sync.get() ){
//                  
//                  idName = String.valueOf( SEQ_ID.getID() ) ;
//                  buf.append( ZERO_DAT.substring( idName.length() ) ) ;
//                  buf.append( idName ) ;
//                  buf.append( THREAD_END_NAME ) ;
//                  ret = buf.toString() ;
//                  
//                  if( ThreadManager.getInstance().isName( ret ) == false ){
//                      break ;
//                  }
//                  
//              }
//              
//              ret = null ;
//          }
//          
//      }catch( Exception t ){
//          ret = null ;
//      }finally{
//          idName = null ;
//          buf = null ;
//          sync = null ;
//      }
//      
//      return ret ;
//      
//  }
    
}

