/*
 * @(#)ThreadManMonitor.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.CharTable;


/**
 * スレッド管理情報監視処理.
 * <BR><BR>
 * スレッド管理テーブルを監視するための処理です.
 *
 * @version     1.00, 2004/01/31
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
class ThreadMonitor extends Thread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ThreadMonitor.class ) ;
    
    /**
     * CPUタイムクロック割り当て.
     */
    private static final int CPU_CREATE = 15 ;
    
    
    /**
     * スレッド破棄後の待ち時間 : デフォルト値.
     * 5秒.
     */
    private static final long DEFAULT_END_TIME = 500L ;
    
    /**
     * スレッド監視対象テーブル.
     */
    private CharTable m_table = null ;
    
    /**
     * スレッド破棄後の待ち時間.
     */
    private long m_endTime = 0L ;
    
    /**
     * スレッド最終アクセス応答なしの待ち時間.
     */
    private long m_lastAccessTime = 0L ;
    
    /**
     * スレッド終了フラグ.
     */
    private boolean m_exitFlg = false ;
    
    
    /**
     * スレッド管理テーブル同期オブジェクト.
     */
    private Synchronized m_sync = null ;
    
    
    /**
     * コンストラクタ.
     */
    private ThreadMonitor()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 監視処理用の情報をセットします.
     * <BR>
     * @param table 監視対象のテーブル情報を設定します.
     * @param sync 監視テーブル用同期オブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public ThreadMonitor( CharTable table,Synchronized sync )
        throws InputException,AccessException
    {
        this( null,table,sync,ThreadMonitor.DEFAULT_END_TIME,0L ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 監視処理用の情報をセットします.
     * <BR>
     * @param name 生成スレッド名を設定します.
     * @param table 監視対象のテーブル情報を設定します.
     * @param sync 監視テーブル用同期オブジェクトを設定します.
     * @param end スレッド破棄後の待ち時間を設定します.<BR>
     *            また、この設定値はミリ秒単位で設定します.
     * @param access 最終アクセス応答なしの待ち時間を設定します.
     *               また、この設定値はミリ秒単位で設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    protected ThreadMonitor(
        String name,CharTable table,Synchronized sync,long end,long access
    )
        throws InputException,AccessException
    {
        if( table == null || sync == null || end <= 0 || access < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_table = table ;
        m_sync = sync ;
        m_endTime = end ;
        m_lastAccessTime = access ;
        
        try{
            
            this.checkAccess() ;
            this.setDaemon( true ) ;
            this.start() ;
            
        }catch( Exception t ){
            
            m_table = null ;
            m_sync = null ;
            m_endTime = 0L ;
            m_lastAccessTime = 0L ;
            
            throw new AccessException( t ) ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        try{
            //this.interrupt() ;
            synchronized( m_sync.get() ){
                m_exitFlg = true ;
            }
        }catch( Exception t ){
            m_exitFlg = true ;
        }
        
        m_sync = null ;
        m_table = null ;
        m_endTime = 0L ;
        m_lastAccessTime = 0L ;
        
    }
    
    /**
     * スレッド処理.
     */
    public final void run()
    {
        int i ;
        int len ;
        
        long end,access ;
        long time ;
        long destroyTime ;
        long lastTime ;
        
        boolean flg ;
        
        CharTable table = null ;
        
        ThreadState pause = null ;
        String[] keys = null ;
        String key = null ;
        
        
        
        end = m_endTime ;
        access = m_lastAccessTime ;
        
        table = m_table ;
        
        LOG.info(
            "*** スレッドモニター( " +
            this.getName() +
            " )は開始しました ***"
        ) ;
        
        try{
            
            for( ;; ){
                
                try{
                    
                    ThreadMonitor.endThread() ;
                    
                    synchronized( m_sync.get() ){
                        if( m_exitFlg == true ){
                            return ;
                        }
                        keys = m_table.getNames() ;
                    }
                    
                    if( keys == null ){
                        continue ;
                    }
                    
                    len = keys.length ;
                    
                    for( i = len-1 ; i >= 0 ; i -- ){
                        
                        key = keys[ i ] ;
                        keys[ i ] = null ;
                        pause = null ;
                        
                        time = System.currentTimeMillis() ;
                        
                        synchronized( m_sync.get() ){
                            
                            // このスレッド処理終了を検知.
                            if( m_table == null || m_exitFlg == true ){
                                return ;
                            }
                            
                            try{
                                pause = ( ThreadState )table.get( key ) ;
                            }catch( Exception t ){
                                continue ;
                            }
                            
                        }
                        
                        // スレッドが破棄されている場合.
                        if(
                            pause != null &&
                            pause.isEndThread() == true &&
                            ( destroyTime = pause.getDestroyTime() ) != 0L
                        )
                        {
                            
                            // スレッド破棄待ち時間条件に一致した場合.
                            if( destroyTime + end <= time ){
                                
                                // スレッド管理テーブルから破棄.
                                try{
                                    ThreadManager.removeID( pause.getID() ) ;
                                    ThreadManager.removeID( pause.getBeforeID() ) ;
                                }catch( Exception e ){
                                }
                                
                                table.remove( key ) ;
                                
                            }
                            
                        }
                        
                        // スレッド最終アクセス応答なし時間が有効な場合.
                        if( access != 0L ){
                            
                            // スレッド破棄待ち時間条件に一致した場合.
                            if( pause.getLastAccessTime() + access <= time ){
                                
                                // スレッド管理要素をクリア.
                                try{
                                    pause.clear() ;
                                }catch( Exception e ){
                                }
                                
                            }
                            
                        }
                        
                        ThreadMonitor.endThread() ;
                        
                    }
                
                }catch( NullPointerException nul ){
                    throw nul ;
                }
            }
            
        }catch( OutOfMemoryError mem ) {
            LOG.error( "OutOfMemoryError",mem ) ;
        }catch( NullPointerException nue ){
            LOG.info(
                "スレッドモニター( " +
                this.getName() +
                " )は停止しました"
            ) ;
            
        }catch( InterruptedException ie ){
            
            LOG.info(
                "割り込みによりスレッドモニター( " +
                this.getName() +
                " )は停止しました"
            ) ;
            
        }catch( ThreadDeath death ){
            
            LOG.info(
                "スレッド破棄によりスレッドモニター( " +
                this.getName() +
                " )は停止しました"
            ) ;
            
            throw death ;
            
        }catch( Exception t ){
            
            LOG.info(
                "例外( " + t.getClass().getName() +
                " )によりスレッドモニター( " +
                this.getName() +
                " )は停止しました",
                t
            ) ;
            
            
        }finally{
            
            try{
                len = keys.length ;
                for( i = 0 ; i < len ; i ++ ){
                    keys[ i ] = null ;
                }
            }catch( Exception tt ){
            }
            
            table = null ;
            
            pause = null ;
            keys = null ;
            key = null ;
            
        }
        
    }
    
    /**
     * スレッド破棄チェック.
     */
    private static final void endThread()
        throws InterruptedException
    {
        Thread.sleep( ThreadMonitor.CPU_CREATE ) ;
    }
    
    
}

