/*
 * @(#)ThreadManPause.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.CharTableSync;


/**
 * スレッドステートオブジェクト.
 * <BR><BR>
 * スレッドの要素情報です.
 *
 * @version     1.00, 2004/01/31
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ThreadState
{
    
    /**
     * スレッド終了フラグ.
     */
    private volatile boolean m_end_flg = false ;
    
    /**
     * 対象スレッド名.
     */
    private String m_name = null ;
    
    /**
     * ID.
     */
    private volatile int m_id = ThreadManager.NOSET_ID ;
    
    /**
     * 以前のIDを管理.
     */
    private volatile int m_before_id = ThreadManager.NOT_ID ;
    
    /**
     * スレッド生成時時間.
     */
    private volatile long m_create_time = System.currentTimeMillis() ;
    
    /**
     * スレッド破棄時間.
     */
    private volatile long m_destroy_time = 0L ;
    
    /**
     * 最終アクセス時間.
     */
    private volatile long m_last_access = System.currentTimeMillis() ;
    
    /**
     * データ格納パラメータ.
     */
    private CharTableSync m_table = null ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    
    /**
     * コンストラクタ.
     */
    private ThreadState()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のスレッド名をThreadオブジェクトで設定します.
     * <BR>
     * @param thread 対象のスレッドオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    protected ThreadState( Thread thread )
        throws InputException
    {
        if( thread == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_sync.create() ;
        this.m_name = thread.getName() ;
        this.m_table = new CharTableSync() ;
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
/*
System.out.println( " > clear: >> name[" +
    m_name + "] endFlg[" +
    m_end_flg + "] destTime[" +
    m_destroy_time + "] lastAccess[" +
    m_last_access + "]:::" ) ;
*/
        try{
            synchronized( m_sync.get() ){
                m_table.clear() ;
            }
        }catch( Exception t ){
        }
        
        m_sync.clear() ;
        
        if( this.m_id != ThreadManager.NOT_ID ){
            
            this.m_before_id = this.m_id ;
            this.m_id = ThreadManager.NOT_ID ;
            
        }
        
        this.m_name = null ;
        this.m_end_flg = true ;
        this.m_destroy_time = System.currentTimeMillis() ;
        this.m_last_access = System.currentTimeMillis() ;
        this.m_table = null ;
        
    }
    
    /**
     * スレッド終了フラグの設定.
     * <BR><BR>
     * スレッド終了フラグを設定します.
     */
    public final void endThread()
    {
/*
System.out.println( "  > endThread: >> name[" +
    m_name + "] endFlg[" +
    m_end_flg + "] destTime[" +
    m_destroy_time + "] lastAccess[" +
    m_last_access + "]:::" ) ;
*/
        try{
            
            try{
                synchronized( m_sync.get() ){
                    m_table.clear() ;
                }
            }catch( Exception tt ){
            }
            
            if( this.m_id != ThreadManager.NOT_ID ){
                
                this.m_before_id = this.m_id ;
                this.m_id = ThreadManager.NOT_ID ;
                
            }
            
            this.m_name = null ;
            this.m_end_flg = true ;
            this.m_destroy_time = System.currentTimeMillis() ;
            this.m_last_access = System.currentTimeMillis() ;
            this.m_table = null ;
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッドID情報の設定.
     * <BR><BR>
     * スレッドID情報を設定します.
     * <BR>
     * @param id 対象のIDを設定します.
     */
    public final void setID( int id )
    {
        if( id < 0 ){
            return ;
        }
        
        try{
            synchronized( m_sync.get() ){
                this.m_id = id ;
            }
        }catch( Exception t ){
            this.m_id = ThreadManager.NOT_ID ;
        }
        
    }
    
    /**
     * 最終アクセス時間のセット.
     * <BR><BR>
     * 最終アクセス時間をセットします.
     */
    public final void setLastAccessTime()
    {
        this.m_last_access = System.currentTimeMillis() ;
    }
    
    /**
     * パラメータの設定.
     * <BR><BR>
     * パラメータを設定します.
     * <BR>
     * @param name 対象パラメータ名を設定します.
     * @param value 対象パラメータ値を設定します.
     * @exception InputException 入力例外.
     */
    public final void setParameter( String name,Object value )
        throws InputException
    {
        
        if( name == null || value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_table.add( name,value ) ;
            }
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 対象のスレッド名を取得.
     * <BR><BR>
     * 対象のスレッド名を取得します.
     * <BR>
     * @return String 対象のスレッド名が返されます.
     */
    public final String getName()
    {
        String ret = null ;
        
        ret = this.m_name ;
        
        return ret ;
    }
    
    /**
     * スレッドID情報の取得.
     * <BR><BR>
     * スレッドID情報を取得します.
     * <BR>
     * @return int 対象のIDが返されます.
     */
    public final int getID()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.m_id ;
            }
        }catch( Exception t ){
            ret = ThreadManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 状態変化前のバックアップスレッドID情報の取得.
     * <BR><BR>
     * 状態変化前のバックアップスレッドID情報を取得します.
     * <BR>
     * @return int 対象のIDが返されます.
     */
    public final int getBeforeID()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.m_before_id ;
            }
        }catch( Exception t ){
            ret = ThreadManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    
    /**
     * スレッド生成時間の取得.
     * <BR><BR>
     * スレッド生成時間を取得します.
     * <BR>
     * @return long スレッド生成時間が返されます.<BR>
     *              また、返される時間単位はミリ秒です.
     */
    public final long getCreateTime()
    {
        long ret ;
        
        ret = this.m_create_time ;
        
        return ret ;
    }
    
    /**
     * スレッド削除時間の取得.
     * <BR><BR>
     * スレッド削除時間を取得します.
     * <BR>
     * @return long スレッド削除時間が返されます.<BR>
     *              また、返される時間単位はミリ秒です.
     */
    public final long getDestroyTime()
    {
        long ret ;
        
        ret = this.m_destroy_time ;
        
        return ret ;
    }
    
    /**
     * 最終アクセス時間の取得.
     * <BR><BR>
     * 最終アクセス時間を取得します.
     * <BR>
     * @return long 最終アクセス時間が返されます.<BR>
     *              また、返される時間単位はミリ秒です.
     */
    public final long getLastAccessTime()
    {
        long ret ;
        
        ret = this.m_last_access ;
        
        return ret ;
    }
    
    /**
     * パラメータの取得.
     * <BR><BR>
     * パラメータを取得します.
     * <BR>
     * @param name 対象パラメータ名を設定します.
     * @return Object 対象パラメータ値が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final Object getParameter( String name )
    {
        
        Object ret = null ;
        
        if( name == null || name.length() <= 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table.get( name ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータ名群の取得.
     * <BR><BR>
     * パラメータ名群を取得します.
     * <BR>
     * @return String[] 対象のパラメータ群が返されます.
     */
    public final String[] getParameterNames()
    {
        
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table.getNames() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * スレッド終了フラグの取得.
     * <BR><BR>
     * スレッド終了フラグを取得します.
     * <BR>
     * @return boolean スレッド終了フラグが返されます.<BR>
     *                 [true]が返された場合、スレッド終了を意味します.<BR>
     *                 [false]が返された場合、スレッド動作を意味します.
     */
    public final boolean isEndThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.m_end_flg ;
            }
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
}

