/*
 * @(#)TimerLock.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.ExceptionUtil;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.StackTrace;
import org.maachang.commons.util.UtilCom;


/**
 * タイマーロック処理.
 * <BR><BR>
 * タイマーロック処理をサポートします.<BR>
 * このオブジェクトを利用して同期処理を実施することで、
 * デッドロックの検知が可能です.<BR>
 * また、利用方法として、以下のように行います.
 * <BR>
 * <PRE>
 * public class Test{
 *   
 *   private final Synchronized m_sync = new Synchronized() ;
 *   
 *   public final void startTest()
 *   {
 *      // Lock開始.
 *      TimerLock.get().lock( m_sync ) ;
 *      
 *      .....
 *      ...
 *      .
 *      
 *      // Lock終了.
 *      TimerLock.get().unlock( m_sync ) ;
 *      
 *   }
 * }
 * </PRE>
 *  
 * @version 1.0.0 2005/03/29
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class TimerLock
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( TimerLock.class ) ;
    
    /**
     * シングルトン.
     */
    private static final TimerLock SNGL = new TimerLock() ;
    
    /**
     * ログ取得フラグ.
     */
    private volatile boolean m_logMode = false ;
    
    /**
     * コンストラクタ.
     */
    private TimerLock(){}
    
    /**
     * オブジェクト情報を取得.
     * <BR><BR>
     * オブジェクト情報を取得します.
     * <BR>
     * @return TimerLock オブジェクト情報が返されます.
     */
    public static final TimerLock get()
    {
        return SNGL ;
    }
    
    /**
     * ロック処理.
     * <BR><BR>
     * ロック処理を実施します.
     * <BR>
     * @param sync ロック対象の同期オブジェクトを設定します.
     * @exception DeadLockException デッドロック例外.
     * @exception ExecutionException 実行例外.
     */
    public final void lock( Synchronized sync )
        throws DeadLockException,ExecutionException
    {
        long timeout ;
        boolean mode ;
        
        String tmp = null ;
        StringBuffer buf = null ;
        StackTrace trace = null ;
        
        if( sync != null ){
            
            mode = this.getLogOutput() ;
            
            if( sync.get() == null ){
                throw new ExecutionException(
                    "同期オブジェクトは無効です",
                    ExecutionException.LEVEL_STOP
                ) ;
            }
            else if( sync.increment() == false ){
                
                timeout = System.currentTimeMillis() + sync.getTimeout() ;
                
                for( ;; ){
                    
                    if( sync.get() == null ){
                        throw new ExecutionException(
                            "同期オブジェクトが無効になりました",
                            ExecutionException.LEVEL_STOP
                        ) ;
                    }
                    else if( timeout <= System.currentTimeMillis() ){
                        
                        buf = new StringBuffer() ;
                        
                        buf.append( "ロック開始処理がスレッド(" ) ;
                        buf.append( sync.getLockByThreadName() ) ;
                        buf.append( ")のロック中によりタイムアウト(" ) ;
                        buf.append( sync.getTimeout() ) ;
                        buf.append( ")しました" ) ;
                        
                        tmp = buf.toString() ;
                        buf = null ;
                        
                        LOG.error( tmp ) ;
                        throw new DeadLockException( tmp ) ;
                        
                    }
                    else if( sync.increment() == true ){
                        break ;
                    }
                    
                    UtilCom.idleTime() ;
                    
                }
            }
            
            if( mode == true ){
                
                try{
                    
                    trace = ExceptionUtil.getPauseStackTrace( new Throwable() ) ;
                    
                    buf = new StringBuffer() ;
                    buf.append( " ++-++ @lock / name:" ) ;
                    buf.append( Thread.currentThread().getName() ) ;
                    buf.append( " object:" ) ;
                    buf.append( trace.getObject( 1 ) ) ;
                    buf.append( " method:" ) ;
                    buf.append( trace.getMethod( 1 ) ) ;
                    buf.append( " line:" ) ;
                    buf.append( trace.getLine( 1 ) ) ;
                    
                    tmp = buf.toString() ;
                    buf = null ;
                    
                    LOG.debug( tmp ) ;
                    
                }catch( Exception e ){
                }finally{
                    
                    trace.clear() ;
                    trace = null ;
                    buf = null ;
                    tmp = null ;
                    
                }
                
            }
            
        }
        
    }
    
    /**
     * アンロック処理.
     * <BR><BR>
     * アンロック処理を実施します.
     * <BR>
     * @param sync アンロック対象の同期オブジェクトを設定します.
     * @exception ExecutionException 実行例外.
     */
    public final void unlock( Synchronized sync )
        throws ExecutionException
    {
        boolean mode ;
        
        String tmp = null ;
        StringBuffer buf = null ;
        StackTrace trace = null ;
        
        if( sync != null ){
            
            mode = this.getLogOutput() ;
            
            if( sync.get() == null ){
                throw new ExecutionException(
                    "同期オブジェクトが無効です",
                    ExecutionException.LEVEL_STOP
                ) ;
            }
            else if( sync.decrement() == true ){
                
                if( mode == true ){
                    
                    try{
                        
                        trace = ExceptionUtil.getPauseStackTrace( new Throwable() ) ;
                        
                        buf = new StringBuffer() ;
                        buf.append( " --+-- @unlock / name:" ) ;
                        buf.append( Thread.currentThread().getName() ) ;
                        buf.append( " object:" ) ;
                        buf.append( trace.getObject( 1 ) ) ;
                        buf.append( " method:" ) ;
                        buf.append( trace.getMethod( 1 ) ) ;
                        buf.append( " line:" ) ;
                        buf.append( trace.getLine( 1 ) ) ;
                        
                        tmp = buf.toString() ;
                        
                        LOG.debug( tmp ) ;
                        
                    }catch( Exception e ){
                    }finally{
                        trace.clear() ;
                        trace = null ;
                        buf = null ;
                        tmp = null ;
                    }
                    
                }
                
            }
            
        }
        
    }
    
    /**
     * ログ出力フラグを設定.
     * <BR><BR>
     * ログ出力フラグを設定します.<BR>
     * この情報はロック/アンロック時にログ出力するか否かを
     * 設定します.
     * <BR>
     * @param mode ログ出力フラグを設定します.<BR>
     *             [true]を設定した場合、ログ情報が出力されます.<BR>
     *             [false]を設定した場合、ログ情報が出力されません.
     */
    public synchronized final void setLogOutput( boolean mode )
    {
        m_logMode = mode ;
    }
    
    /**
     * ログ出力フラグを取得.
     * <BR><BR>
     * ログ出力フラグを取得します.<BR>
     * この情報はロック/アンロック時にログ出力するか否かの
     * 情報です.
     * <BR>
     * @return boolean ログ出力フラグが返されます.<BR>
     *                 [true]が返された場合、ログ情報が出力されます.<BR>
     *                 [false]が返された場合、ログ情報が出力されません.
     */
    public synchronized final boolean getLogOutput()
    {
        return m_logMode ;
    }
}

