/*
 * @(#)TimerThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.SettingException;
import org.maachang.commons.util.UtilCom;


/**
 * スレッド実施によるタイマー処理。<BR>
 * スレッド実施によるタイマー処理をサポートします。
 *
 * @version 1.0.0 2003/11/02
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class TimerThread extends ProcessThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( TimerThread.class ) ;
    
    
    /**
     * スレッド動作フラグ名.
     */
    private static final String RUN_NAME = "run" ;
    
    /**
     * スレッド終了フラグ名.
     */
    private static final String END_NAME = "end" ;
    
    /**
     * スレッド終了フラグ名.
     */
    private static final String STOP_NAME = "stop" ;
    
    /**
     * スレッド動作タイマー値名.
     */
    private static final String TIMER_NAME = "timer" ;
    
    
    
    /**
     * 実行オブジェクト.
     */
    private ExecutionThread m_exec = null ;
    
    /**
     * スレッド実施処理.
     */
    private final BaseThread m_thread = new BaseThread() ;
    
    /**
     * スレッドステータス.
     */
    private ThreadState m_state = null ;
    
    /**
     * スレッド実施時間.
     */
    private volatile long m_nowTime = 0L ;
    
    /**
     * スレッド同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public TimerThread()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec スレッド実行対象オブジェクト.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public TimerThread( ExecutionThread exec,long time )
        throws InputException
    {
        try{
            this.create( exec,time ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec スレッド実行対象オブジェクト.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public TimerThread( ExecutionThread exec,Object param,long time )
        throws InputException
    {
        try{
            this.create( exec,param,time ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec スレッド実行対象オブジェクト.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public TimerThread( boolean mode,ExecutionThread exec,Object param,long time )
        throws InputException
    {
        try{
            this.create( mode,exec,param,time ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec スレッド実行対象オブジェクト.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void create( ExecutionThread exec,long time )
        throws InputException
    {
        try{
            this.create( exec,null,time ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec スレッド実行対象オブジェクト.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void create( ExecutionThread exec,Object param,long time )
        throws InputException
    {
        this.create( true,exec,param,time ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec スレッド実行対象オブジェクト.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @param time スレッド実行タイム値を設定します.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void create( boolean mode,ExecutionThread exec,Object param,long time )
        throws InputException
    {
        
        if( exec == null || time <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            m_thread.create( ( mode == true ) ? false : true,this,param,true ) ;
            
            if( mode == true ){
                m_thread.setObjectName( exec.getClass().getName() ) ;
            }
            else{
                m_thread.setObjectName( null ) ;
            }
            
            m_state = this.initState( time ) ;
            m_exec = exec ;
            
        }catch( Exception t ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        
        try{
            m_thread.clear() ;
        }catch( Exception t ){
        }
        
        try{
            m_state.clear() ;
        }catch( Exception tt ){
        }
        
        m_exec = null ;
        m_state = null ;
        m_nowTime = 0L ;
        
    }
    
    /**
     * スレッド開始処理を実施.
     * <BR><BR>
     * スレッド開始処理を実施します.
     * <BR>
     * @exception SettingException 設定例外.
     */
    public final void startThread()
        throws SettingException
    {
        int state ;
        boolean flg ;
        
        if( this.isExecution() == false ){
            
            throw new SettingException(
                "設定情報が不足しているため、スレッドを実施できません"
            ) ;
            
        }
        
        // スレッド実行.
        m_thread.startThread() ;
        
        // スレッド実行待ち.
        /*for( ;; ){
            
            state = m_thread.getState() ;
            
            if(
                state == ExecutionThread.STATE_NOT ||
                state == ExecutionThread.STATE_EXECUTION
            )
            {
                if( state == ExecutionThread.STATE_NOT ){
                    this.clear() ;
                    throw new SettingException(
                        "スレッド実行に失敗しました"
                    ) ;
                }
                break ;
            }
            
            UtilCom.cpuCreate() ;
            
        }*/
        
    }
    
    /**
     * スレッド終了処理を実施.
     * <BR><BR>
     * スレッド終了処理を実施します.
     */
    public final void exitThread()
    {
        
        try{
            
            synchronized( m_sync.get() ){
                
                this.setFlg( TimerThread.END_NAME,true ) ;
                
            }
            
        }catch( Exception t ){
        }finally{
            try{
                m_thread.exitThread() ;
            }catch( Exception t1 ){
            }
        }
        
    }
    
    
    /**
     * ループストップ処理.
     * <BR><BR>
     * ループストップ処理のON／OFFを行います.
     */
    public final void stopLoop()
    {
        boolean flg ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                flg = ( this.getFlg( TimerThread.STOP_NAME ) == true ) ?
                    false : true ;
                
                this.setFlg( TimerThread.END_NAME,flg ) ;
                
            }
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッド破棄を実施.
     * <BR><BR>
     * スレッド破棄を実施します.
     */
    public final void destroyThread()
    {
        
        try{
            
            synchronized( m_sync.get() ){
                this.setFlg( TimerThread.END_NAME,true ) ;
            }
            
            UtilCom.idleSleep( BaseThread.DESTROY_WAIT ) ;
            
        }catch( Exception t ){
        }finally{
            
            try{
                m_thread.destroyThread() ;
            }catch( Exception t1 ){
            }
            
        }
        
    }
    
    /**
     * スレッドプライオリティを設定.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを設定します.
     * <BR>
     * @param newPriority 新しいスレッドプライオリティを設定します.
     * @exception InputException 入力例外.
     */
    public final void setPriority( int newPriority )
        throws InputException
    {
        try{
            synchronized( m_sync.get() ){
                m_thread.setPriority( newPriority ) ;
            }
        }catch( InputException in ){
            throw in ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 設定タイムアウト値の取得.
     * <BR><BR>
     * 設定されているタイムアウト値を取得します.
     * <BR>
     * @return long 設定されているタイムアウト値を取得します.<BR>
     *              [-1L]が返された場合、スレッドは有効ではありません.
     */
    public final long getTime()
    {
        long ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = (
                    ( long[] )m_state.getParameter(
                        TimerThread.TIMER_NAME
                    )
                )[ 0 ] ;
                
            }
            
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadState() ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施時間の取得.
     * <BR><BR>
     * スレッドが実施された時間を取得します.
     * <BR>
     * @return long スレッドが実施された時間が返されます.
     */
    public final long getThreadTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_nowTime ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドプライオリティを取得.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを取得します.
     * <BR>
     * @return int 設定されているスレッドプライオリティが返されます.<BR>
     *             スレッドが存在しない場合[-1]が返されます.
     */
    public final int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getPriority() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行用オブジェクトを取得.
     * <BR><BR>
     * スレッド実行用オブジェクトを取得します.
     * <BR>
     * @return ExecutionThread スレッド実行用オブジェクトが返されます.
     */
    public final ExecutionThread getExecutionThread()
    {
        ExecutionThread ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_exec ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド開始チェック.
     * <BR><BR>
     * スレッドが開始されているかチェックします.
     * <BR>
     * @return boolean スレッドチェック.<BR>
     * スレッドが開始している場合は、戻り値は[true]で返されます。<BR>
     * スレッドが開始されていない場合、戻り値は[false]で返されます。
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = this.getFlg( TimerThread.RUN_NAME ) ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ループストップチェック.
     * <BR><BR>
     * ループストップチェックを取得します.
     * <BR>
     * @return boolean [true]が返された場合、ループ処理はストップしています.
     *                 [false]が返された場合、ループ処理はストップしていません.
     */
    public final boolean isStop()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = this.getFlg( TimerThread.STOP_NAME ) ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ループエンドチェック.
     * <BR><BR>
     * ループエンドチェックを取得します.
     * <BR>
     * @return boolean [true]が返された場合、ループ処理は終了待ち状態です.
     *                 [false]が返された場合、ループ処理は終了待ち状態ではありません.
     */
    public final boolean isEnd()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = this.getFlg( TimerThread.END_NAME ) ;
            }
            
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行.
     * <BR><BR>
     * スレッド実行を行う処理部を実装します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception ExecutionException 実行例外.
     * @exception ThreadDeath スレッド破棄例外.
     */
    protected void processThread( Object in )
        throws ExecutionException,ThreadDeath
    {
        int mode ;
        long waitTimer ;
        long nowTimer ;
        
        mode = 0 ;
        
        try{
            
            if( ( waitTimer = this.getTime() ) == -1 ){
                return ;
            }
            
            synchronized( m_sync.get() ){
                
                this.setFlg( TimerThread.RUN_NAME,true ) ;
                this.setFlg( TimerThread.END_NAME,false ) ;
                this.setFlg( TimerThread.STOP_NAME,false ) ;
                m_nowTime = System.currentTimeMillis() ;
                
            }
            
            m_exec.init( in ) ;
            
            nowTimer = System.currentTimeMillis() + waitTimer ;
            
            for( ;; ){
                
                //Thread.yield() ;
                ThreadManager.isExit() ;
                
                if( this.isEnd() == true ){
                    
                    break ;
                    
                }
                
                if( this.isStop() == true ){
                    
                    if( mode == 0 || mode == 2 ){
                        
                        mode = 1 ;
                        m_exec.stop( in ) ;
                        
                        synchronized( m_sync.get() ){
                            m_nowTime = System.currentTimeMillis() ;
                        }
                        
                    }
                    
                    nowTimer = System.currentTimeMillis() + waitTimer ;
                    
                    UtilCom.idleTime() ;
                    continue ;
                }
                
                if( System.currentTimeMillis() >= nowTimer ){
                    
                    nowTimer = System.currentTimeMillis() + waitTimer ;
                    m_exec.execution( in ) ;
                    mode = 2 ;
                    
                    ThreadManager.isExit() ;
                    
                    synchronized( m_sync.get() ){
                        m_nowTime = System.currentTimeMillis() ;
                    }
                    
                    continue ;
                }
                
                UtilCom.idleTime() ;
                //Thread.yield() ;
                
            }
            
        }catch( ThreadDeath death ){
            throw death ;
        }catch( OutOfMemoryError mem ) {
        }catch( NullPointerException nul ){
        }catch( ExecutionException ex1 ){
            
            throw ex1 ;
            
        }catch( Exception t ){
        }finally{
            
            try{
                
                if( mode == 2 ){
                    m_exec.exit( in ) ;
                }
                
            }catch( Exception tt ){
            }
            
            try{
                
                synchronized( m_sync.get() ){
                    
                    this.setFlg( TimerThread.RUN_NAME,false ) ;
                    this.setFlg( TimerThread.END_NAME,true ) ;
                    this.setFlg( TimerThread.STOP_NAME,false ) ;
                    m_nowTime = 0L ;
                    
                }
                
            }catch( Exception tt ){
                m_nowTime = 0L ;
            }
        }
        
    }
    
    /**
     * スレッド破棄時の処理.
     * <BR><BR>
     * スレッド破棄時の処理を生成します.
     * <BR>
     * @param in スレッド処理で渡される引数情報です.
     * @exception Exception 処理エラーを設定します.
     */
    protected void destroyThread( Object in )
        throws Exception
    {
        m_exec.destroy( in ) ;
    }
    
    /**
     * スレッド例外発生処理実施.
     * <BR><BR>
     * スレッド処理での例外が発生した場合、実施されます.
     * <BR>
     * @param err 実行例外が返されます.
     */
    protected void toException( ExecutionException err )
    {
        // 特になし..
    }
    
    /**
     * 情報設定準備チェック.
     */
    private final boolean isExecution()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( m_exec == null ) ? false : true ;
            }
            
        }catch( Exception t ){
            //t.printStackTrace() ;
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ステータス初期化処理.
     */
    private final ThreadState initState( long time )
        throws AccessException
    {
        
        boolean[] flg = null ;
        long[] tm = null ;
        ThreadState ret = null ;
        
        try{
            
            // スレッド対象のパラメータ値を取得.
            ret = ThreadManager.
                    getInstance().
                        getState( m_thread.getName() ) ;
            if( ret == null ){
                LOG.error(
                    "スレッド名:" + m_thread.getName() +
                    "に対するスレッドは存在しません"
                ) ;
            }
            
            // 各パラメータを初期処理.
            flg = new boolean[ 1 ] ;
            flg[ 0 ] = false ;
            ret.setParameter( TimerThread.RUN_NAME,flg ) ;
            
            flg = new boolean[ 1 ] ;
            flg[ 0 ] = true ;
            ret.setParameter( TimerThread.END_NAME,flg ) ;
            
            flg = new boolean[ 1 ] ;
            flg[ 0 ] = false ;
            ret.setParameter( TimerThread.STOP_NAME,flg ) ;
            
            tm = new long[ 1 ] ;
            tm[ 0 ] = time ;
            ret.setParameter( TimerThread.TIMER_NAME,tm ) ;
            
        }catch( Exception t ){
            
            //t.printStackTrace() ;
            ret = null ;
            throw new AccessException( t ) ;
            
        }finally{
            
            flg = null ;
            tm = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * フラグの設定.
     */
    private final void setFlg( String name,boolean flg )
        throws NullPointerException
    {
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            obj[ 0 ] = flg ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
    }
    
    /**
     * フラグの取得.
     */
    private final boolean getFlg( String name )
        throws NullPointerException
    {
        boolean ret ;
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            ret = obj[ 0 ] ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
        
        return ret ;
    }
    
}

