/*
 * @(#)ArrayTable.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.NotExistException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.array.ObjectArray;


/**
 * 配列可能なテーブル情報をサポートするオブジェクト.
 * <BR><BR>
 * 配列可能なテーブル情報をサポートします.
 *
 * @version 1.00, 2003/10/27
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ArrayTable implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ArrayTable.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 配列を管理するオブジェクト.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * データサイズ格納値.
     */
    private int m_length = 0 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ArrayTable(){
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報のクリア.
     * <BR><BR>
     * 格納されている全ての情報をクリアします.
     */
    public final void clear()
    {
        int i ;
        int len ;
        
        String[] names = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        tb = m_table ;
        
        try{
            
            names = tb.getNames() ;
            len = names.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                ( ( ObjectArray )tb.get( names[i] ) ).clear() ;
                
                tb.remove( names[i] ) ;
                
            }
            
        }catch( Exception in ){
        }finally{
            
            names = null ;
            tb = null ;
            ls = null ;
            
            m_length = 0 ;
            
        }
        
    }
    
    /**
     * テーブル情報の追加.
     * <BR><BR>
     * 指定されたテーブル名に対する情報を新規生成／追加します.
     * <BR>
     * @param name 追加するテーブル名を指定します.
     * @param value 追加するテーブル情報を指定します.
     * @exception InputException 入力例外.
     */
    public synchronized final void add( String name,Object value )
        throws InputException
    {
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null || value == null ){
            if( name == null ){
                throw new InputException( "指定テーブル名情報は不正です" ) ;
            }
            throw new InputException( "指定value情報は不正です" ) ;
        }
        
        tb = m_table ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            ls.add( value ) ;
            
            m_length ++ ;
            
        }catch( Exception t ){
            
            try{
                ls = new ObjectArray() ;
                ls.add( value ) ;
                tb.add( name,ls ) ;
                
                m_length ++ ;
            }catch( Exception tt ){
            }
            
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
    }
    
    /**
     * 情報のセット.
     * <BR><BR>
     * 指定位置の情報を置き換えます.
     * <BR>
     * @param value セットする情報を設定します.
     * @param name セット対象のテーブル名を指定します.
     * @param no セット対象のテーブル要素項番を指定します.
     * @exception InputException 入力例外.
     */
    public synchronized final void set( Object value,String name,int no )
        throws InputException
    {
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null || value == null ){
            if( name == null ){
                throw new InputException( "指定テーブル名情報は不正です" ) ;
            }
            throw new InputException( "指定value情報は不正です" ) ;
        }
        
        tb = m_table ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            ls.set( no,value ) ;
            
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( 
                "指定テーブル名[" + name + "]に対する要素番号[" + no + "]は不正です" ) ;
        }catch( Exception t ){
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
    }
    
    /**
     * キー名の変更.
     * <BR><BR>
     * キー名を変更します.
     * <BR>
     * @param src 変更元のキー名を設定します.
     * @param no 変更元の要素項番を設定します.
     * @param dest 変更先のキー名を設定します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public synchronized final void rename( String src,int no,String dest )
        throws InputException,NotExistException
    {
        int len ;
        Object val = null ;
        
        if( src == null || no < 0 || dest == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( ( len = this.getElements( src ) ) <= 0 ){
                throw new NotExistException(
                    "対象の情報名(" + src +
                    ")は存在しません"
                ) ;
            }
            else if( len <= no ){
                throw new NotExistException(
                    "対象の情報名(" + src +
                    ")に対する対象項番(" + no +
                    ")に位置する要素は存在しません"
                ) ;
            }
            
            val = this.get( src,no ) ;
            this.remove( src,no ) ;
            this.add( dest,val ) ;
            
        }catch( InputException in ){
            throw new NotExistException( in ) ;
        }catch( NotExistException ne ){
            throw ne ;
        }finally{
            val = null ;
        }
        
    }
    
    /**
     * テーブル情報の削除.
     * <BR><BR>
     * テーブル情報を削除します.<BR>
     * 指定されたテーブル名に対するすべての情報を削除します.
     * <BR>
     * @param name 削除対象のテーブル名を指定します.
     * @return Object[] 削除された情報が全て返されます.
     * @exception InputException 入力例外.
     */
    public synchronized final Object[] remove( String name )
        throws InputException
    {
        int i ;
        int len ;
        Object[] ret = null ;
        
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        tb = m_table ;
        
        if( name == null ){
            throw new InputException( "指定テーブル名情報は不正です" ) ;
        }
        
        len = -1 ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            len = ls.size() ;
            
            if( len != 0 ){
                ret = new Object[ len ] ;
                for( i = 0 ; i < len ; i ++ ){
                    ret[ i ] = ls.get( i ) ;
                }
            }
            
            ls.clear() ;
            tb.remove( name ) ;
            
            m_length -= len ;
            
        }catch( InputException in ){
            throw in ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * テーブル情報の削除.
     * <BR><BR>
     * テーブル情報を削除します.<BR>
     * 指定されたテーブル名に対する1要素を削除します.
     * <BR>
     * @param name 削除対象のテーブル名を指定します.
     * @param no 削除対象の要素番号を指定します.
     * @return Object 削除された情報が返されます.
     * @exception InputException 入力例外.
     */
    public synchronized final Object remove( String name,int no )
        throws InputException
    {
        Object ret = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null || no < 0 ){
            if( name == null ){
                throw new InputException( "指定テーブル名情報は不正です" ) ;
            }
            throw new InputException( "対象要素番号は不正です" ) ;
        }
        
        tb = m_table ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            ret = ls.remove( no ) ;
            if( ls.size() == 0 ){
                tb.remove( name ) ;
            }
            m_length -- ;
            
        }catch( IndexOutOfBoundsException io ){
                throw new InputException( 
                    "指定テーブル名[" + name + "]に対する要素番号[" + no + "]は不正です" ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * テーブル情報の取得.
     * <BR><BR>
     * テーブル情報を取得します.<BR>
     * 指定されたテーブル名に対する全要素を取得します.
     * <BR>
     * @param name 取得対象のテーブル名を設定します.
     * @return Object[] 取得された単体( または複数 )のオブジェクトが返されます.<BR>
     *                  テーブル名に対する情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外.
     */
    public synchronized final Object[] get( String name )
        throws InputException
    {
        int i ;
        int len ;
        
        Object[] ret = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null ){
            throw new InputException( "指定テーブル名情報は不正です" ) ;
        }
        
        tb = m_table ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            len = ls.size() ;
            ret = new Object[len] ;
            
            for( i = 0 ; i < len ; i ++ ){
                ret[i] = ls.get( i ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * テーブル情報の取得.
     * <BR><BR>
     * テーブル情報を取得します.<BR>
     * 指定されたテーブル名に対する1要素を取得します.
     * <BR>
     * @param name 取得対象のテーブル名を指定します.
     * @param no 取得対象の要素番号を指定します.
     * @return Object[] 取得された単体のオブジェクトが返されます.<BR>
     *                  テーブル名に対する情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外.
     */
    public synchronized final Object get( String name,int no )
        throws InputException
    {
        Object ret = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null || no < 0 ){
            if( name == null ){
                throw new InputException( "指定テーブル名情報は不正です" ) ;
            }
            throw new InputException( "対象要素番号は不正です" ) ;
        }
        
        tb = m_table ;
        
        try{
            
            ls = ( ObjectArray )tb.get( name ) ;
            ret = ls.get( no ) ;
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( IndexOutOfBoundsException io ){
            ret = null ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 格納テーブル名群の取得.
     * <BR><BR>
     * 格納されているテーブル名群( 複数 )を取得します.
     * <BR>
     * @return String[] 格納テーブル名群( 複数 )が格納されます.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public synchronized final String[] getNames()
    {
        String[] ret = null ;
        CharTable tb = null ;
        
        try{
            tb = m_table ;
            ret = tb.getNames() ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            tb = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納キー数の取得.
     * <BR><BR>
     * 格納キーを取得します.<BR>
     * これは[ArrayTable#size()]とは違い、格納されているキーの数を
     * 返します。
     * 格納されている情報数を調べたい場合は[ArrayTable#size()]を利用してください。
     * <BR>
     * @return int 格納キー数が返されます.
     */
    public synchronized final int getKeySize()
    {
        int ret ;
        
        try{
            ret = m_table.size() ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 格納テーブル要素数の取得.
     * <BR><BR>
     * 格納テーブル要素数を取得します.<BR>
     * 指定したテーブル名に対する要素数を取得します.
     * <BR>
     * @param name 取得対象のテーブル名を設定します.
     * @return int 取得されたテーブル名に対する要素数が返されます.
     */
    public synchronized final int getElements( String name )
    {
        int ret ;
        
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        if( name == null ){
            return -1 ;
        }
        
        tb = m_table ;
        
        try{
            ls = ( ObjectArray )tb.get( name ) ;
            ret = ls.size() ;
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( IndexOutOfBoundsException io ){
            ret = 0 ;
        }catch( Exception t ){
            ret = 0 ;
        }finally{
            
            ls = null ;
            tb = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 設定されている全ての要素数の取得.
     * <BR><BR>
     * 設定されている全ての要素数を取得します.
     * <BR>
     * @return int 格納されている全ての要素数を取得します.
     */
    public synchronized final int size()
    {
        
        int ret ;
        
        try{
            ret = m_length ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 文字列に対する存在チェック.
     * <BR><BR>
     * 文字列に対する存在チェックを行います.
     * <BR>
     * @param name チェック対象の文字列を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public synchronized final boolean isData( String name )
    {
        boolean ret ;
        
        try{
            ret = m_table.isData( name ) ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * コピーオブジェクトの生成.
     * <BR><BR>
     * コピーオブジェクトの生成.
     * <BR>
     * @param table コピー対象のオブジェクトを設定します.
     */
    public synchronized final void getCopyObject( ArrayTable table )
    {
        int i,j ;
        int len,len2 ;
        
        String[] names = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        tb = m_table ;
        
        try{
            
            table.clear() ;
            
            names = tb.getNames() ;
            len = names.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                ls = ( ObjectArray )tb.get( names[i] ) ;
                len2 = ls.size() ;
                
                for( j = 0 ; j < len2 ; j ++ ){
                    
                    table.add( names[i] , ls.get( j ) ) ;
                    
                }
                
            }
            
        }catch( Exception t ){
            
            if( table != null ){
                table.clear() ;
            }
            
        }finally{
            
            names = null ;
            tb = null ;
            ls = null ;
            
        }
    }
    
    /**
     * クローンオブジェクトの生成.
     * <BR><BR>
     * クローンオブジェクトの生成.
     * <BR>
     * @return ArrayTable クローンオブジェクトが返されます.
     */
    public synchronized final ArrayTable getCloneObject()
    {
        int i,j ;
        int len,len2 ;
        
        String[] names = null ;
        ObjectArray ls = null ;
        CharTable tb = null ;
        
        ArrayTable ret = null ;
        
        ret = new ArrayTable() ;
        tb = m_table ;
        
        try{
            names = tb.getNames() ;
            len = names.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                ls = ( ObjectArray )tb.get( names[i] ) ;
                len2 = ls.size() ;
                
                for( j = 0 ; j < len2 ; j ++ ){
                    
                    ret.add( names[i] , ls.get( j ) ) ;
                    
                }
                
            }
        }catch( Exception t ){
            
            if( ret != null ){
                ret.clear() ;
            }
            
            ret = null ;
            
        }finally{
            
            names = null ;
            tb = null ;
            ls = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 格納情報内容出力.
     * <BR><BR>
     * 格納されている情報内容が返されます.
     * <BR>
     * @return String 情報内容が返されます.
     */
    public synchronized final String toString()
    {
        int i,j ;
        int len,len2 ;
        boolean flg = false ;
        
        String ret = null ;
        StringBuffer buf = null ;
        String[] list = null ;
        
        try{
            
            buf = new StringBuffer() ;
            list = this.getNames() ;
            
            len = list.length ;
            for( i = 0 ; i < len ; i ++ ){
                
                len2 = this.getElements( list[ i ] ) ;
                for( j = 0 ; j < len2 ; j ++ ){
                    if( flg == true ){
                        buf.append( "," ) ;
                    }
                    buf.append( list[ i ] ) ;
                    buf.append( "[" ) ;
                    buf.append( j ) ;
                    buf.append( "]:" ) ;
                    buf.append( this.get( list[ i ],j ) ) ;
                    flg = true ;
                }
                
            }
            
            ret = buf.toString().trim() ;
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            ret = null ;
        }finally{
            buf = null ;
            list = null ;
        }
        
        return ret ;
    }
    
}
