/*
 * @(#)DateCalc.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import org.maachang.commons.exception.InputException;

/**
 * 日付計算系処理.
 *
 * @version 1.00, 2004/02/15
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class DateCalc
{
    
    /**
     * 曜日情報 : 日曜日.
     */
    public static final int SUNDAY = 1 ;
    
    /**
     * 曜日情報 : 月曜日.
     */
    public static final int MONDAY = 2 ;
    
    /**
     * 曜日情報 : 火曜日.
     */
    public static final int TUESDAY = 3 ;
    
    /**
     * 曜日情報 : 水曜日.
     */
    public static final int WEDNESDAY = 4 ;
    
    /**
     * 曜日情報 : 木曜日.
     */
    public static final int THURSDAY = 5 ;
    
    /**
     * 曜日情報 : 金曜日.
     */
    public static final int FRIDAY = 6 ;
    
    /**
     * 曜日情報 : 土曜日.
     */
    public static final int SATURDAY = 7 ;
    
    
    
    /**
     * 月数係数.
     */
    private static final int[][] MONTH_TABLE = 
    {
        { 31,28,31,30,31,30,31,31,30,31,30,31 },
        { 31,29,31,30,31,30,31,31,30,31,30,31 }
    } ;
    
    /**
     * 曜日情報 for 日本語.
     */
    private static final String[] WEEK_JP = {
        "?","日","月","火","水","木","金","土"
    } ;
    
    /**
     * 曜日情報 for 英語.
     */
    private static final String[] WEEK_ENG = {
        "?","sunday","monday","tuesday",
        "wednesday","thursday","friday","saturday"
    } ;
    
    
    
    /**
     * コンストラクタ.
     */
    private DateCalc()
    {
    }
    
    
    /**
     * 閏年チェック.
     * <BR><BR>
     * 対象年に対する閏年チェックを行います.
     * <BR>
     * @param year チェック対象の年情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象年は閏年です.<BR>
     *                 [false]が返された場合、対象年は閏年ではありません.
     */
    public static final boolean checkLeapYear( int year )
    {
        
        return (
            ( year % 4 ) == 0 &&
            ( year % 100 ) != 0 ||
            ( year % 400 ) == 0
        ) ? true : false ;
        
    }
    
    /**
     * 0001年から、対象年までの閏年の数を取得.
     * <BR><BR>
     * 0001年から、対象年までの閏年の数を取得します.
     * <BR>
     * @param year 取得対象の年を設定します.
     * @return int 閏年の数が返されます.
     */
    public static final int getLeapYearSize( int year )
    {
        return ( year / 4 ) - ( year / 100 ) + ( year / 400 ) ;
    }
    
    /**
     * 対象の日付から、曜日を取得.
     * <BR><BR>
     * 対象の日付から、曜日を取得します.
     * <BR>
     * @param year 対象の年情報を設定します.
     * @param month 対象の月情報を設定します.
     * @param day 対象の日情報を設定します.
     * @return int 対象の曜日が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getDaysToWeek( int year,int month,int day )
        throws InputException
    {
        
        try{
            DateCalc.checkDays( year,month,day ) ;
        }catch( InputException in ){
            throw in ;
        }
        
        return DateCalc.daysToWeek( year,month,day ) ;
        
    }
    
    /**
     * 対象の日付から、曜日を取得.
     * <BR><BR>
     * 対象の日付から、曜日を取得します.
     * <BR>
     * @param lang 取得対象の言語を選択します.<BR>
     *             [true]を設定した場合、日本語で取得します.<BR>
     *             [false]を設定した場合、英語で取得します.
     * @param year 対象の年情報を設定します.
     * @param month 対象の月情報を設定します.
     * @param day 対象の日情報を設定します.
     * @return String 対象の曜日が返されます.
     * @exception InputException 入力例外.
     */
    public static final String getDaysToWeek( boolean lang,int year,int month,int day )
        throws InputException
    {
        
        try{
            DateCalc.checkDays( year,month,day ) ;
        }catch( InputException in ){
            throw in ;
        }
        
        return ( lang == true ) ?
            DateCalc.WEEK_JP[ DateCalc.daysToWeek( year,month,day ) ] :
            DateCalc.WEEK_ENG[ DateCalc.daysToWeek( year,month,day ) ] ;
        
    }
    
    /**
     * 月の最終日を取得.
     * <BR><BR>
     * 月の最終日を取得します.
     * <BR>
     * @param year 対象の年情報を設定します.
     * @param month 対象の月情報を設定します.
     * @return int 対象の最終日が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getDaysOfMonth( int year,int month )
        throws InputException
    {
        int ret ;
        
        try{
            
            ret = ( DateCalc.checkLeapYear( year ) == true ) ?
                DateCalc.MONTH_TABLE[ 1 ][ month-1 ] :
                DateCalc.MONTH_TABLE[ 0 ][ month-1 ] ;
            
        }catch( Exception t ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 0001年01月01日からの日数を求める.
     * <BR><BR>
     * 0001年01月01日からの日数を求めます.
     * <BR>
     * @param year 対象の年情報を設定します.
     * @param month 対象の月情報を設定します.
     * @param day 対象の日情報を設定します.
     * @return int 対象の日数が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getDaysOfAD( int year,int month,int day )
        throws InputException
    {
        
        try{
            DateCalc.checkDays( year,month,day ) ;
        }catch( InputException in ){
            throw in ;
        }
        
        if(month <= 2){
            year--;
            month += 12;
        }
        
        return (
            year * 365 + ( year / 4 ) - 
            ( year / 100 ) + ( year / 400 ) +
            ( ( 13 * month + 8 ) / 5 ) +
            ( 28 * month ) - 34 - 365 + day
        ) ;
        
    }
    
    /**
     * 日付の差を求める.
     * <BR><BR>
     * 日付の差を求めます.<BR>
     * 比較方法は、以下の方法で行われます.<BR>
     * 日付差 = src - dest.
     * <BR>
     * @param srcYear 比較元の年情報を指定します.
     * @param srcMonth 比較元の月情報を指定します.
     * @param srcDay 比較元の日情報を指定します.
     * @param dstYear 比較先の年情報を指定します.
     * @param dstMonth 比較先の月情報を指定します.
     * @param dstDay 比較先の日情報を指定します.
     * @return int 日付の差が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getDaysDiff(
        int srcYear,int srcMonth,int srcDay,
        int dstYear,int dstMonth,int dstDay
    ) throws InputException
    {
        int ret ;
        
        try{
            
            ret = DateCalc.getDaysOfAD(
                srcYear,srcMonth,srcDay
            ) - DateCalc.getDaysOfAD(
                dstYear,dstMonth,dstDay
            ) ;
            
        }catch( InputException in ){
            throw in ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の日付から、年月日を取得.
     * <BR><BR>
     * 対象の日付から、年月日を取得します.<BR>s
     * また、ad がマイナスの場合例外となります.
     * <BR>
     * @param ad 変換対象の日付を設定します.
     * @param year 変換された、年情報が返されます.
     * @param month 変換された、月情報が返されます.
     * @param day 変換された、日情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final void getADToDays( int[] year,int[] month,int[] day,int ad )
        throws InputException
    {
        int i ;
        int yt,yt2 ;
        int tpYear,tpYear2 ;
        int[] monTbl = null ;
        
        if( ad < 0 || year == null || month == null || day == null ){
            if( ad < 0 ){
                throw new InputException( "マイナス変換はサポートされていません" ) ;
            }
            throw new InputException( "引数は不正です" ) ;
        }
        
        tpYear = 1;
        ad -= 1;
        
        while(
            ad < 0 ||
            ad >= (
                (
                    ( tpYear % 4 ) == 0 &&
                    ( tpYear % 100 ) != 0 ||
                    ( tpYear % 400 ) == 0
                ) ? 366 : 365
            )
        )
        {
            
            yt = tpYear + ad / 365 - ( ( ad % 365 < 0 ) ? 1 : 0 ) ;
            
            yt2 = yt - 1 ;
            tpYear2 = tpYear - 1 ;
            
            ad -= (
                ( yt - tpYear ) * 365 +
                ( ( yt2 / 4 ) - ( yt2 / 100 ) + ( yt2 / 400 ) ) -
                ( ( tpYear2 / 4 ) - ( tpYear2 / 100 ) + ( tpYear2 / 400 ) )
            ) ;
            
            tpYear = yt ;
            
        }
        
        monTbl = DateCalc.MONTH_TABLE[
            (
                ( tpYear % 4 ) == 0 &&
                ( tpYear % 100 ) != 0 ||
                ( tpYear % 400 ) == 0
            ) ? 1 : 0
        ] ;
        
        for( i = 0 ; i < 12 ; i ++ ){
            
            if( ad < monTbl[ i ] ){
                break ;
            }
            
            ad -= monTbl[ i ] ;
            
        }
        
        year[ 0 ] = tpYear ;
        month[ 0 ] = i + 1 ;
        day[ 0 ] = ad + 1;
        monTbl = null ;
        
    }
    
    /**
     * ｎ日後の日数を取得.
     * <BR><BR>
     * ｎ日後の日数を取得します.
     * <BR>
     * @param year ｎ日を付加した年情報が返されます.
     * @param month ｎ日を付加した月情報が返されます.
     * @param day ｎ日を付加した日情報が返されます.
     * @param n 付加対象に日付を設定します.
     * @exception InputException 入力例外.
     */
    public static final void getDaysToAdd(
        int[] year,int[] month,int[] day,int n
    ) throws InputException
    {
        
        try{
            
            n += DateCalc.getDaysOfAD( year[ 0 ],month[ 0 ],day[ 0 ] ) ;
            DateCalc.getADToDays( year,month,day,n ) ; 
            
        }catch( NullPointerException nul ){
            throw new InputException( "引数は不正です" ) ;
        }catch( InputException in ){
            throw in ;
        }
        
    }
    
    
    
    /**
     * 日付から、曜日を取得.
     */
    private static final int daysToWeek( int year,int month,int day )
    {
        if( month <= 2 ){
            
            year -- ;
            month += 12 ;
            
        }
        
        return (
            (
                year + ( year / 4 ) - 
                ( year / 100 ) + ( year / 400 ) +
                ( ( 13 * month + 8 ) / 5 ) + day 
            ) % 7
        ) ;
        
    }
    
    /**
     * 引数不正チェック.
     */
    private static final void checkDays( int year,int month,int day )
        throws InputException
    {
        
        try{
            
            if( DateCalc.checkLeapYear( year ) == true ){
                
                if( day <= 0 || day > DateCalc.MONTH_TABLE[ 1 ][ month-1 ] ){
                    throw new InputException( "引数は不正です" ) ;
                }
                
            }else{
                
                if( day <= 0 || day > DateCalc.MONTH_TABLE[ 0 ][ month-1 ] ){
                    throw new InputException( "引数は不正です" ) ;
                }
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( Exception t ){
            throw new InputException( "引数は不正です" ) ;
        }
        
    }
    
}

