/*
 * @(#)GetterSetter.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.lang.reflect.Method;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;


/**
 * Getter/Setter処理.
 * 
 * @version 1.00, 2005/10/15
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class GetterSetter
{
    
    /**
     * 名前 : get.
     */
    private static final String NAME_GET = "get" ;
    
    /**
     * 名前 : is.
     * 戻り値がbooleanの場合のみ.
     */
    private static final String NAME_IS = "is" ;
    
    /**
     * 名前 : set.
     */
    private static final String NAME_SET = "set" ;
    
    /**
     * タイプ : get.
     */
    private static final int TYPE_GET = 1 ;
    
    /**
     * タイプ : is.
     */
    private static final int TYPE_IS = 2 ;
    
    /**
     * タイプ : set.
     */
    private static final int TYPE_SET = 0 ;
    
    
    
    private GetterSetter(){}
    
    /**
     * 指定Objectのフィールド内容を取得.
     * <BR><BR>
     * 指定Objectのフィールド内容を取得します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param in 取得対象のオブジェクト情報を設定します.
     * @param names 取得対象のフィールド名を設定します.
     * @return Object[] 取得されたフィールド情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Object[] getter( Object in,String[] names )
        throws InputException,AccessException
    {
        return GetterSetter.getter( in,false,names ) ;
    }
    
    /**
     * 指定Objectのフィールド内容を取得.
     * <BR><BR>
     * 指定Objectのフィールド内容を取得します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合でも、利用できます.
     * <BR>
     * @param in 取得対象のオブジェクト情報を設定します.
     * @param names 取得対象のフィールド名を設定します.
     * @return Object[] 取得されたフィールド情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Object[] getterByAll( Object in,String[] names )
        throws InputException,AccessException
    {
        return GetterSetter.getter( in,true,names ) ;
    }
    
    /**
     * Objectのフィールトに対してオブジェクトを挿入.
     * <BR><BR>
     * Objectのフィールドに対してオブジェクトを挿入します.<BR>
     * また、その時のフィールド名に対するSetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param out 挿入対象のオブジェクトを設定します.
     * @param names 追加対象のフィールド名を設定します.<BR>
     *              また、この条件は引数[objs]に紐付けされます.
     * @param objs 追加対象のオブジェクト群を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void setter(
        Object out,String[] names,Object[] objs )
        throws InputException,AccessException
    {
        GetterSetter.setter( out,false,names,objs ) ;
    }
    
    /**
     * Objectのフィールトに対してオブジェクトを挿入.
     * <BR><BR>
     * Objectのフィールドに対してオブジェクトを挿入します.<BR>
     * また、その時のフィールド名に対するSetterメソッドは
     * protected,privateの場合でも、利用できます.
     * <BR>
     * @param out 挿入対象のオブジェクトを設定します.
     * @param names 追加対象のフィールド名を設定します.<BR>
     *              また、この条件は引数[objs]に紐付けされます.
     * @param objs 追加対象のオブジェクト群を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void setterByAll(
        Object out,String[] names,Object[] objs )
        throws InputException,AccessException
    {
        GetterSetter.setter( out,true,names,objs ) ;
    }
    
    /**
     * Objectのフィールドをコピー.
     * <BR><BR>
     * Objectのフィールドで一致する内容をコピーします.<BR>
     * また、その時のフィールド名に対するGetter/Setterメソッドは
     * protected,privateの場合、コピーされません.
     * <BR>
     * @param src コピー元のオブジェクトを設定します.
     * @param dest コピー先のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void copy( Object src,Object dest )
        throws InputException,AccessException
    {
        GetterSetter.copy( false,src,dest ) ;
    }
    
    /**
     * Objectのフィールドをコピー.
     * <BR><BR>
     * Objectのフィールドで一致する内容をコピーします.<BR>
     * また、その時のフィールド名に対するGetter/Setterメソッドは
     * protected,privateの場合でも、コピーされます.
     * <BR>
     * @param src コピー元のオブジェクトを設定します.
     * @param dest コピー先のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void copyByAll( Object src,Object dest )
        throws InputException,AccessException
    {
        GetterSetter.copy( true,src,dest ) ;
    }
    
    /**
     * 指定Objectのフィールド型を取得.
     * <BR><BR>
     * 指定Objectのフィールド型を取得します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param in 取得対象のオブジェクト情報を設定します.
     * @param name 取得対象のフィールド名を設定します.
     * @return Class 対象のフィールド型が返されます.
     * @exception InputException 入力例外.
     */
    public static final Class getType( Object in,String name )
        throws InputException
    {
        //return GetterSetter.getterReturn( false,in,name ) ;
        return GetterSetter.setterParam( false,in,name ) ;
    }
    
    /**
     * 指定Objectのフィールド型を取得.
     * <BR><BR>
     * 指定Objectのフィールド型を取得します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合でも、利用できます.
     * <BR>
     * @param in 取得対象のオブジェクト情報を設定します.
     * @param name 取得対象のフィールド名を設定します.
     * @return Class 対象のフィールド型が返されます.
     * @exception InputException 入力例外.
     */
    public static final Class getTypeAll( Object in,String name )
        throws InputException
    {
        //return GetterSetter.getterReturn( true,in,name ) ;
        return GetterSetter.setterParam( true,in,name ) ;
    }
    
    
    
    /**
     * 指定Objectのフィールド内容を取得.
     */
    private static final Object[] getter( Object in,boolean mode,String[] names )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        String methodName = null ;
        Class[] cArgs = null ;
        Class thisClass = null ;
        Method md = null ;
        Object[] ret = null ;
        
        if(
            in == null ||
            names == null || ( len = names.length ) <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        thisClass = in.getClass() ;
        cArgs = new Class[ 1 ] ;
        ret = new Object[ len ] ;
        
        try{
            for( i = 0 ; i < len ; i ++ ){
                
                try{
                    
                    if( names[ i ] != null && names[ i ].length() > 0 ){
                        
                        methodName = GetterSetter.convertName(
                            TYPE_GET,names[ i ] ) ;
                        md = GetterSetter.getMethod(
                            mode,thisClass,cArgs,methodName,null ) ;
                            
                        if( md == null ){
                            
                            methodName = GetterSetter.convertName(
                                TYPE_IS,names[ i ] ) ;
                            md = GetterSetter.getMethod(
                                mode,thisClass,cArgs,methodName,null ) ;
                            
                            if( md == null ){
                                /*throw new AccessException(
                                    "オブジェクト[" + thisClass.getName() +
                                    "]に対して、メソッド名[" + methodName +
                                    "]の条件が不正です"
                                ) ;*/
                                continue ;
                            }
                            
                        }
                        
                        if( md != null ){
                            md.setAccessible( true ) ;
                            ret[ i ] = md.invoke( in,null ) ;
                        }
                        
                    }
                    
                }catch( Exception ee ){
                }
                
            }
            
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            methodName = null ;
            cArgs = null ;
            thisClass = null ;
            md = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * Objectのフィールトに対してオブジェクトを挿入.
     */
    private static final void setter(
        Object out,boolean mode,String[] names,Object[] objs )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        String methodName = null ;
        Object[] oArgs = null ;
        Class[] cArgs = null ;
        Class thisClass = null ;
        Method md = null ;
        
        if(
            out == null ||
            names == null || ( len = names.length ) <= 0 ||
            objs == null || objs.length <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( objs.length != len ){
            throw new InputException(
                "フィールド名とオブジェクト数が一致しません" ) ;
        }
        
        thisClass = out.getClass() ;
        cArgs = new Class[ 1 ] ;
        oArgs = new Object[ 1 ] ;
        
        try{
            for( i = 0 ; i < len ; i ++ ){
                
                if( names[ i ] != null && names[ i ].length() > 0 ){
                    
                    try{
                        
                        methodName = GetterSetter.convertName(
                            TYPE_SET,names[ i ] ) ;
                        md = GetterSetter.getMethod(
                            mode,thisClass,cArgs,methodName,objs[ i ] ) ;
                        if( md == null ){
                            /*throw new AccessException(
                                "オブジェクト[" + thisClass.getName() +
                                "]に対して、メソッド名[" + methodName +
                                "]の条件が不正です"
                            ) ;
                            */
                            continue ;
                        }
                        
                        oArgs[ 0 ] = objs[ i ] ;
                        md.setAccessible( true ) ;
                        md.invoke( out,oArgs ) ;
                        
                    }catch( Exception ee ){
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            methodName = null ;
            oArgs = null ;
            cArgs = null ;
            thisClass = null ;
            md = null ;
        }
        
    }
    
    /**
     * 指定オブジェクト内の指定Setterメソッドのパラメータ型を取得.
     */
    private static final Class setterParam( boolean mode,Object target,String name )
        throws InputException
    {
        Class[] params = null ;
        Class ret = null ;
        
        params = GetterSetter.getParam( 
            mode,target,name,1,0
        ) ;
        
        if( params != null && params.length == 1 ){
            ret = params[ 0 ] ;
        }
        
        return ret ;
    }
    
    /**
     * 指定オブジェクト内の指定Getterメソッドの戻り値型を取得.
     */
    private static final Class getterReturn( boolean mode,Object target,String name )
        throws InputException
    {
        return GetterSetter.getReturn( mode,target,name ) ;
    }
    
    
    /**
     * Objectのフィールドをコピー.
     */
    private static final void copy( boolean mode,Object src,Object dest )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        
        Method[] all = null ;
        Class objCls = null ;
        Class srcClass = null ;
        Class srcNext = null ;
        Class destClass = null ;
        
        if( src == null || dest == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            // publicのみかprotected,privateを含むか選択.
            if( mode == true ){
                
                ////////////////////////////////
                // protected,privateを含む場合.
                ////////////////////////////////
                
                objCls = Object.class ;
                srcClass = src.getClass() ;
                destClass = dest.getClass() ;
                
                for( srcNext = srcClass ;; ){
                
                    all = srcNext.getDeclaredMethods() ;
                    
                    GetterSetter.copyMethod( mode,src,srcNext,dest,destClass,all ) ;
                    
                    if(
                        objCls.equals( ( srcNext = srcNext.getSuperclass() ) ) == true ||
                        srcNext == null
                    )
                    {
                        break ;
                    }
                }
                
            }
            else{
                
                /////////////////////
                // publicのみの場合.
                /////////////////////
                
                srcClass = src.getClass() ;
                destClass = dest.getClass() ;
                
                all = srcClass.getMethods() ;
                
                GetterSetter.copyMethod( mode,src,srcClass,dest,destClass,all ) ;
            }
            
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            all = null ;
            srcClass = null ;
            srcNext = null ;
            destClass = null ;
        }
        
    }
    
    /**
     * setメソッドに対する引数の型が返されます.
     */
    private static final Class[] getParam( boolean mode,Object target,String name,int paramLen,int next )
        throws InputException
    {
        int i,j ;
        int len ;
        
        Method[] md = null ;
        Class[] ret = null ;
        
        if( target == null || name == null || name.length() <= 0 || next < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            name = GetterSetter.convertName( TYPE_SET,name ) ;
            
            // publicのみかprotected,privateを含むか選択.
            if( mode == true ){
                md = target.getClass().getDeclaredMethods() ;
            }
            // publicのみ.
            else{
                md = target.getClass().getMethods() ;
            }
            
            // パラメータ数指定が存在しない場合.
            if( paramLen <= 0 ){
            
                // 指定条件のメソッドを捜す.
                if( md != null && ( len = md.length ) > 0 ){
                    
                    for( i = 0,j = 0 ; i < len ; i ++ ){
                        
                        if( name.equals( md[ i ].getName() ) == true ){
                            
                            if( j >= next ){
                                ret = md[ i ].getParameterTypes() ;
                                break ;
                            }
                            
                            j ++ ;
                            
                        }
                        
                    }
                    
                }
                
            }
            // パラメータ数指定が存在する場合.
            else{
                
                // 指定条件のメソッドを捜す.
                if( md != null && ( len = md.length ) > 0 ){
                    
                    for( i = 0,j = 0 ; i < len ; i ++ ){
                        
                        if( name.equals( md[ i ].getName() ) == true ){
                            
                            if(
                                ( ret = md[ i ].getParameterTypes() ) != null &&
                                ret.length == paramLen
                            )
                            {
                                
                                if( j >= next ){
                                    break ;
                                }
                                
                                j ++ ;
                                
                            }
                            
                            ret = null ;
                            
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            md = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * get又はis条件の戻り値の型を取得.
     */
    private static final Class getReturn( boolean mode,Object target,String name )
        throws InputException
    {
        
        String methodName = null ;
        Method md = null ;
        Class thisClass = null ;
        Class[] cArgs = null ;
        Class ret = null ;
        
        if( target == null || name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            cArgs = new Class[ 1 ] ;
            thisClass = target.getClass() ;
            
            methodName = GetterSetter.convertName(
                TYPE_GET,name ) ;
            md = GetterSetter.getMethod(
                mode,thisClass,cArgs,methodName,null ) ;
                
            if( md == null ){
                
                methodName = GetterSetter.convertName(
                    TYPE_IS,name ) ;
                md = GetterSetter.getMethod(
                    mode,thisClass,cArgs,methodName,null ) ;
                
            }
            
            if( md != null ){
                ret = md.getReturnType() ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            methodName = null ;
            md = null ;
            thisClass = null ;
            cArgs = null ;
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象フィールド名をgetter/setter名に変換.
     */
    private static final String convertName( int mode,String name )
    {
        
        String dat = null ;
        StringBuffer buf = null ;
        
        if( name == null || name.length() <= 0 ){
            return null ;
        }
        
        if( mode == TYPE_SET ){
            dat = NAME_SET ;
        }
        else if( mode == TYPE_GET ){
            dat = NAME_GET ;
        }
        else if( mode == TYPE_IS ){
            dat = NAME_IS ;
        }
        else{
            dat = NAME_GET ;
        }
        
        buf = new StringBuffer() ;
        buf.append( dat ) ;
        buf.append( name.substring(0,1).toUpperCase() ) ;
        buf.append( name.substring( 1 ) ) ;
        
        return buf.toString() ;
        
    }
    
    /**
     * 対象フィールド名を変換.
     */
    private static final String changeName( String name )
    {
        if( name == null || name.length() <= 0 ){
            return null ;
        }
        
        if( name.indexOf( NAME_GET ) == 0 ){
            return NAME_SET + name.substring( NAME_GET.length() ) ;
        }
        else if( name.indexOf( NAME_SET ) == 0 ){
            return NAME_GET + name.substring( NAME_SET.length() ) ;
        }
        else if( name.indexOf( NAME_IS ) == 0 ){
            return NAME_SET + name.substring( NAME_IS.length() ) ;
        }
        
        return null ;
    }
    
    /**
     * 指定メソッド群の内容を、オブジェクトにコピー.
     */
    private static final void copyMethod(
        boolean mode,Object src,Class srcClass,Object dest,Class destClass,Method[] all )
        throws AccessException
    {
        int i ;
        int len ;
        
        Method md = null ;
        String methodSrc = null ;
        String methodDest = null ;
        
        Object[] oArgs = new Object[ 1 ] ;
        Class[] cArgs = new Class[ 1 ] ;
        
        try{
            if( all != null && ( len = all.length ) > 0 ){
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    methodSrc = all[ i ].getName() ;
                    
                    // 対象名はGetterである場合は、処理する.
                    if(
                        Void.TYPE.equals( all[ i ].getReturnType() ) == false &&
                        all[ i ].getParameterTypes().length == 0 &&
                        ( methodDest = GetterSetter.changeName( methodSrc ) ) != null
                    )
                    {
                        
                        try{
                            
                            // コピー元のgetterオブジェクトから、情報を取得.
                            all[ i ].setAccessible( true ) ;
                            oArgs[ 0 ] = all[ i ].invoke( src,null ) ;
                            
                            // コピー先にsetterが存在する場合、コピー元から取得した
                            // オブジェクトを設定.
                            md = GetterSetter.getMethod(
                                mode,destClass,cArgs,methodDest,oArgs[ 0 ] ) ;
                            if( md != null ){
                                
                                md.setAccessible( true ) ;
                                md.invoke( dest,oArgs ) ;
                                
                            }
                            
                        }catch( Exception ee ){
                        }
                        
                    }
                    
                }
                
            }
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            md = null ;
            methodSrc = null ;
            methodDest = null ;
            oArgs = null ;
            cArgs = null ;
        }
    }
    
    /**
     * メソッドを取得.
     */
    private static final Method getMethod(
        boolean mode,Class thisClass,Class[] args,String name,Object obj
    )
        throws AccessException
    {
        int pm = 0 ;
        
        Class objCls = null ;
        Class next = null ;
        Method ret = null ;
        
        objCls = Object.class ;
        next = thisClass ;
        pm = GetterSetter.getPMType( obj ) ;
        
        // publicのみかprotected,privateを含むか選択.
        if( mode == true ){
            
            ////////////////////////////////
            // protected,privateを含む場合.
            ////////////////////////////////
            
            for( ;; ){
                try{
                    
                    // 引数がnull指定の場合.
                    if( pm == -1 ){
                        try{
                            ret = next.getDeclaredMethod(
                                name,null
                            ) ;
                            if( Void.TYPE.equals( ret.getReturnType() ) == true ){
                                return null ;
                            }
                            break ;
                        }catch( Exception e ){
                            throw e ;
                        }
                    }
                    // 引数が通常オブジェクト指定の場合.
                    else if( pm == 0 ){
                        try{
                            args[ 0 ] = obj.getClass() ;
                            ret = next.getDeclaredMethod(
                                name,args
                            ) ;
                            if( Void.TYPE.equals( ret.getReturnType() ) == true ){
                                break ;
                            }
                        }catch( Exception e ){
                            throw e ;
                        }
                    }
                    // 引数がプリミティブ型指定の場合.
                    else{
                        try{
                            args[ 0 ] = obj.getClass() ;
                            ret = next.getDeclaredMethod(
                                name,args
                            ) ;
                            if( Void.TYPE.equals( ret.getReturnType() ) == true ){
                                break ;
                            }
                            else{
                                ret = null ;
                            }
                        }catch( Exception e ){
                            ret = null ;
                        }
                        if( ret == null ){
                            
                            args[ 0 ] = GetterSetter.getPMClass( pm ) ;
                            
                            try{
                                ret = next.getDeclaredMethod(
                                    name,args
                                ) ;
                                if( Void.TYPE.equals( ret.getReturnType() ) == true ){
                                    break ;
                                }
                            }catch( Exception e ){
                                throw e ;
                            }
                            
                        }
                    }
                    
                }catch( Exception ee ){
                    ret = null ;
                }
                
                // 基底クラスまで検索したが検知出来ない場合.
                if(
                    objCls.equals( ( next = next.getSuperclass() ) ) == true ||
                    next == null
                )
                {
                    throw new AccessException(
                        "オブジェクト[" + thisClass.getName() +
                        "]に対して、メソッド名[" + name +
                        "]の処理条件は存在しません"
                    ) ;
                }
                
            }
            
        }
        else{
            
            /////////////////////
            // publicのみの場合.
            /////////////////////
            
            // 引数がnull指定の場合.
            if( pm == -1 ){
                try{
                    ret = thisClass.getMethod(
                        name,null
                    ) ;
                    if( Void.TYPE.equals( ret.getReturnType() ) == true ){
                        return null ;
                    }
                }catch( Exception e ){
                    throw new AccessException( e ) ;
                }
            }
            // 引数が通常オブジェクト指定の場合.
            else if( pm == 0 ){
                try{
                    args[ 0 ] = obj.getClass() ;
                    ret = thisClass.getMethod(
                        name,args
                    ) ;
                    if( Void.TYPE.equals( ret.getReturnType() ) == false ){
                        throw new AccessException(
                            "対象名[" + thisClass.getName() +
                            "]のsetter[" + name + "]に対して戻り値が存在します"
                        ) ;
                    }
                }catch( AccessException ae ){
                    throw ae ;
                }catch( Exception e ){
                    throw new AccessException( e ) ;
                }
            }
            // 引数がプリミティブ型指定の場合.
            else{
                try{
                    args[ 0 ] = obj.getClass() ;
                    ret = thisClass.getMethod(
                        name,args
                    ) ;
                    if( Void.TYPE.equals( ret.getReturnType() ) == false ){
                        ret = null ;
                    }
                }catch( Exception e ){
                    ret = null ;
                }
                if( ret == null ){
                    
                    args[ 0 ] = GetterSetter.getPMClass( pm ) ;
                    
                    try{
                        ret = thisClass.getMethod(
                            name,args
                        ) ;
                        if( Void.TYPE.equals( ret.getReturnType() ) == false ){
                            throw new AccessException(
                                "対象名[" + thisClass.getName() +
                                "]のsetter[" + name + "]に対して戻り値が存在します"
                            ) ;
                        }
                    }catch( AccessException ae ){
                        throw ae ;
                    }catch( Exception e ){
                        throw new AccessException( e ) ;
                    }
                    
                }
            }
            
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトがプリミティブ型かチェック.
     */
    private static final int getPMType( Object obj )
    {
        if( obj == null ){
            return -1 ;
        }
        else if( obj instanceof Boolean ){
            return 1 ;
        }
        else if( obj instanceof Byte ){
            return 2 ;
        }
        else if( obj instanceof Character ){
            return 3 ;
        }
        else if( obj instanceof Short ){
            return 4 ;
        }
        else if( obj instanceof Integer ){
            return 5 ;
        }
        else if( obj instanceof Long ){
            return 6 ;
        }
        else if( obj instanceof Float ){
            return 7 ;
        }
        else if( obj instanceof Double ){
            return 8 ;
        }
        
        return 0 ;
    }
    
    /**
     * プリミティブ項番から、クラスオブジェクトを取得.
     */
    private static final Class getPMClass( int pm )
    {
        Class ret = null ;
        switch( pm ){
            case -1 : ret = Void.TYPE ; break ;
            case 1 : ret = Boolean.TYPE ; break ;
            case 2 : ret = Byte.TYPE ; break ;
            case 3 : ret = Character.TYPE ; break ;
            case 4 : ret = Short.TYPE ; break ;
            case 5 : ret = Integer.TYPE ; break ;
            case 6 : ret = Long.TYPE ; break ;
            case 7 : ret = Float.TYPE ; break ;
            case 8 : ret = Double.TYPE ; break ;
        }
        return ret ;
    }
    
}

