/*
 * @(#)MemoryIndex.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.array.ObjectArray;


/**
 * メモリインデックス情報.
 * <BR><BR>
 * メモリ上にインデックス情報を管理するオブジェクトを提供します.
 *
 * @version     1.00, 2004/05/13
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class MemoryIndex implements ReadIndex
{
    
    /**
     * 配列管理.
     */
    private final ObjectArray m_array = new ObjectArray() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public MemoryIndex()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_array.clear() ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param value 追加対象の情報をセットします.
     * @exception InputException 入力例外.
     */
    public final void addIndex( String value )
        throws InputException
    {
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.createSync() ;
        try{
            synchronized( m_sync.get() ){
                m_array.add( value ) ;
            }
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * インデックス情報の取得.
     * <BR><BR>
     * インデックス情報を取得します.
     * <BR>
     * @param no 取得対象のインデックス位置を設定します.
     * @return Object インデックス位置に対する情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final Object getIndex( int no )
        throws InputException,AccessException
    {
        Object ret = null ;
        
        this.createSync() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_array.get( no ) ;
            }
            if( ret == null ){
                throw new AccessException(
                    "対象の項番(" + no +
                    ")に対して情報は存在しません"
                ) ;
            }
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 管理インデックス数の取得.
     * <BR><BR>
     * 管理しているインデックス数を取得します.
     * <BR>
     * @return int 管理インデックス数が返されます.<BR>
     *             情報が存在しない場合[ReadIndex.NOT_INDEX]が返されます.
     */
    public final int size()
    {
        int ret ;
        
        this.createSync() ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_array.size() ;
            }
            
            ret = ( ret == 0 ) ? ReadIndex.NOT_INDEX : ret ;
            
        }catch( Exception t ){
            ret = ReadIndex.NOT_INDEX ;
        }
        
        return ret ;
    }
    
    /**
     * インデックス利用可能状態チェック.
     * <BR><BR>
     * インデックス情報が利用可能かチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、インデックスは利用可能です.
     *                 [false]が返された場合、インデックスは利用不可能です.
     */
    public final boolean isIndex()
    {
        int len ;
        boolean ret ;
        
        this.createSync() ;
        
        try{
            
            synchronized( m_sync.get() ){
                len = m_array.size() ;
            }
            
            ret = ( len == 0 ) ? false : true ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    /**
     * 同期オブジェクト生成.
     */
    private final void createSync()
    {
        if( m_sync.isUse() == false ){
            m_sync.create() ;
        }
    }
    
}
