/*
 * @(#)Parameter.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.math.BigDecimal;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.exception.UnsupportException;
import org.maachang.commons.util.array.ObjectArray;


/**
 * パラメータオブジェクト.
 * <BR><BR>
 * パラメータ情報を定義するオブジェクトです.
 *
 * @version 1.00,2004/02/16
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class Parameter
{
    
    /**
     * タイプ : Boolean.
     */
    public static final byte TYPE_BOOLEAN = 0x01 ;
    
    /**
     * タイプ : Byte.
     */
    public static final byte TYPE_BYTE = 0x02 ;
    
    /**
     * タイプ : Char.
     */
    public static final byte TYPE_CHAR = 0x03 ;
    
    /**
     * タイプ : Short.
     */
    public static final byte TYPE_SHORT = 0x04 ;
    
    /**
     * タイプ : Int.
     */
    public static final byte TYPE_INT = 0x05 ;
    
    /**
     * タイプ : Long.
     */
    public static final byte TYPE_LONG = 0x06 ;
    
    /**
     * タイプ : Float.
     */
    public static final byte TYPE_FLOAT = 0x07 ;
    
    /**
     * タイプ : Double.
     */
    public static final byte TYPE_DOUBLE = 0x09 ;
    
    /**
     * タイプ : Decimal.
     */
    public static final byte TYPE_DECIMAL = 0x11 ;
    
    /**
     * タイプ : Date.
     */
    public static final byte TYPE_DATE = 0x21 ;
    
    /**
     * タイプ : Time.
     */
    public static final byte TYPE_TIME = 0x22 ;
    
    /**
     * タイプ : Timestamp.
     */
    public static final byte TYPE_TIMESTAMP = 0x23 ;
    
    /**
     * タイプ : Binary.
     */
    public static final byte TYPE_BINARY = 0x31 ;
    
    /**
     * タイプ : Chars.
     */
    public static final byte TYPE_CHARS = 0x32 ;
    
    /**
     * タイプ : String.
     */
    public static final byte TYPE_STRING = 0x41 ;
    
    /**
     * タイプ : Boolean-Object.
     */
    public static final byte TYPE_BOOLEAN_OBJECT = 0x51 ;
    
    /**
     * タイプ : Byte-Object.
     */
    public static final byte TYPE_BYTE_OBJECT = 0x52 ;
    
    /**
     * タイプ : Character-Object.
     */
    public static final byte TYPE_CHARACTER_OBJECT = 0x53 ;
    
    /**
     * タイプ : Short-Object.
     */
    public static final byte TYPE_SHORT_OBJECT = 0x54 ;
    
    /**
     * タイプ : Integer-Object.
     */
    public static final byte TYPE_INTEGER_OBJECT = 0x55 ;
    
    /**
     * タイプ : Long-Object.
     */
    public static final byte TYPE_LONG_OBJECT = 0x56 ;
    
    /**
     * タイプ : Float-Object.
     */
    public static final byte TYPE_FLOAT_OBJECT = 0x57 ;
    
    /**
     * タイプ : Double-Object.
     */
    public static final byte TYPE_DOUBLE_OBJECT = 0x58 ;
    
    /**
     * タイプ : Group.
     */
    public static final byte TYPE_GROUP = 0x7f ;
    
    
    
    /**
     * １パラメータ値.
     */
    class ParamPause{
        
        public byte type = 0x00 ;
        public Object value = null ;
        
    } ;
    
    /**
     * 終了テーブル値.
     */
    private static final byte[] END_TBL = {
        ( byte )0x00,( byte )0x00,( byte )0x00,( byte )0x00
    } ;
    
    /**
     * 終了テーブルチェック値.
     */
    private static final int CKEND_TBL = 0x00000000 ;
    
    
    
    /**
     * データ管理オブジェクト.
     */
    private final ArrayTable m_table = new ArrayTable() ;
    
    /**
     * 配下ParameterTbl値.
     */
    private final ObjectArray m_inParamTbl = new ObjectArray() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public Parameter()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 復元可能なバイナリ情報を設定して、情報を生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 情報読み込み失敗例外.
     */
    public Parameter( byte[] binary )
        throws InputException,ReadSyntaxException
    {
        try{
            
            this.create( binary ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
    }
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 復元可能なバイナリ情報を設定して、情報を生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 情報読み込み失敗例外.
     */
    public Parameter( byte[] binary,int offset )
        throws InputException,ReadSyntaxException
    {
        try{
            
            this.create( binary,offset ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
    }
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のArrayTableオブジェクトを当オブジェクトに反映します.
     * <BR>
     * @param table 反映対象のArrayTableオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public Parameter( ArrayTable table )
        throws InputException,UnsupportException
    {
        int i,j ;
        int len ;
        int lenJ ;
        
        String[] keys = null ;
        String key = null ;
        Object value = null ;
        
        try{
            
            keys = table.getNames() ;
            len = keys.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                key = keys[ i ] ;
                keys[ i ] = null ;
                lenJ = table.getElements( key ) ;
                
                for( j = 0 ; j < lenJ ; j ++ ){
                    
                    value = table.get( key,j ) ;
                    this.put( key,value ) ;
                    value = null ;
                    
                }
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException ue ){
            throw ue ;
        }finally{
            
            keys = null ;
            key = null ;
            value = null ;
        }
        
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.s
     * 具体的には[Parameter#clear()]と同様の処理を定義します.
     */
    public final void create()
    {
        this.clear() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 復元可能なバイナリ情報を設定して、情報を生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 情報読み込み失敗例外.
     */
    public final void create( byte[] binary )
        throws InputException,ReadSyntaxException
    {
        
        try{
            
            this.clear() ;
            
            Parameter.binaryToParameterTbl( this,0,binary ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 復元可能なバイナリ情報を設定して、情報を生成します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 情報読み込み失敗例外.
     */
    public final void create( byte[] binary,int offset )
        throws InputException,ReadSyntaxException
    {
        
        try{
            
            this.clear() ;
            
            Parameter.binaryToParameterTbl( this,offset,binary ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
    }
    
    /**
     * 対象情報のクリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        
        m_table.clear() ;
        m_inParamTbl.clear() ;
        
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * 格納データに対するバイナリ情報を取得します.
     * <BR>
     * @return byte[] 変換されたバイナリ情報が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final byte[] getBinary()
    {
        return Parameter.ParameterTblToBinary( m_table ) ;
    }
    
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final void put( String name,Object info )
        throws InputException,UnsupportException
    {
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( info instanceof Boolean ){
                this.putBooleanObject( name,( Boolean )info ) ;
            }else if( info instanceof Byte ){
                this.putByteObject( name,( Byte )info ) ;
            }else if( info instanceof Character ){
                this.putCharacterObject( name,( Character )info ) ;
            }else if( info instanceof Short ){
                this.putShortObject( name,( Short )info ) ;
            }else if( info instanceof Integer ){
                this.putIntegerObject( name,( Integer )info ) ;
            }else if( info instanceof Long ){
                this.putLongObject( name,( Long )info ) ;
            }else if( info instanceof Float ){
                this.putFloatObject( name,( Float )info ) ;
            }else if( info instanceof Double ){
                this.putDoubleObject( name,( Double )info ) ;
            }else if( info instanceof BigDecimal ){
                this.putDecimal( name,( BigDecimal )info ) ;
            }else if( info instanceof java.sql.Date ){
                this.putDate( name,( java.sql.Date )info ) ;
            }else if( info instanceof java.sql.Time ){
                this.putTime( name,( java.sql.Time )info ) ;
            }else if( info instanceof java.sql.Timestamp ){
                this.putTimestamp( name,( java.sql.Timestamp )info ) ;
            }else if( info instanceof byte[] ){
                this.putBinary( name,( byte[] )info ) ;
            }else if( info instanceof char[] ){
                this.putChars( name,( char[] )info ) ;
            }else if( info instanceof String ){
                this.putString( name,( String )info ) ;
            }else if( info instanceof Parameter ){
                this.putGroup( name,( Parameter )info ) ;
            }else{
                throw new UnsupportException( "設定されたオブジェクトは、対応外です" ) ;
            }
            
        }catch( InputException in ){
            throw in ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putBoolean( String name,boolean info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_BOOLEAN ;
        pause.value = new Boolean( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putByte( String name,byte info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_BYTE ;
        pause.value = new Byte( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putChar( String name,char info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_CHAR ;
        pause.value = new Character( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putShort( String name,short info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_SHORT ;
        pause.value = new Short( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putInt( String name,int info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_INT ;
        pause.value = new Integer( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putLong( String name,long info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_LONG ;
        pause.value = new Long( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putFloat( String name,float info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_FLOAT ;
        pause.value = new Float( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putDouble( String name,double info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_DOUBLE ;
        pause.value = new Double( info ) ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putDecimal( String name,BigDecimal info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_DECIMAL ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putDate( String name,java.sql.Date info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_DATE ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putTime( String name,java.sql.Time info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_TIME ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putTimestamp( String name,java.sql.Timestamp info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_TIMESTAMP ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putBinary( String name,byte[] info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_BINARY ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putChars( String name,char[] info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_CHARS ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putString( String name,String info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_STRING ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putBooleanObject( String name,Boolean info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_BOOLEAN_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putByteObject( String name,Byte info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_BYTE_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putCharacterObject( String name,Character info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_CHARACTER_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putShortObject( String name,Short info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_SHORT_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putIntegerObject( String name,Integer info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_INTEGER_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putLongObject( String name,Long info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_LONG_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putFloatObject( String name,Float info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_FLOAT_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putDoubleObject( String name,Double info )
        throws InputException
    {
        
        ParamPause pause = null ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_DOUBLE_OBJECT ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            pause = null ;
        }
        
    }
    
    /**
     * 対象パラメータ情報セット.
     * <BR><BR>
     * 対象のパラメータ情報を設定します.
     * <BR>
     * @param name 設定対象のキー名を設定します.
     * @param info 設定対象のパラメータ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void putGroup( String name,Parameter info )
        throws InputException
    {
        int i ;
        int len ;
        
        ParamPause pause = null ;
        ObjectArray chk = null ;
        
        chk = m_inParamTbl ;
        
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        len = chk.size() ;
        for( i = 0 ; i < len ; i ++ ){
            if( chk.get( i ) == info ){
                throw new InputException( "対象のParameterImplオブジェクトは追加できません" ) ;
            }
        }
        if( this == info ){
            throw new InputException( "対象のParameterImplオブジェクトは追加できません" ) ;
        }
        
        chk.add( info ) ;
        
        pause = new ParamPause() ;
        pause.type = Parameter.TYPE_GROUP ;
        pause.value = info ;
        
        try{
            m_table.add( name,pause ) ;
        }catch( InputException in ){
            
            pause.value = null ;
            pause = null ;
            
            throw in ;
            
        }finally{
            
            pause = null ;
            chk = null ;
            
        }
        
    }
    
    
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Object 取得された情報が返されます.
     * @exception InputException 入力例外.
     */
    public final Object get( String name,int no )
        throws InputException
    {
        Object ret = null ;
        
        try{
            
            ret = ( ( ParamPause )m_table.get( name,no ) ).value ;
            
        }catch( InputException in ){
            throw in ;
        }finally{
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return boolean 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final boolean getBoolean( String name,int no )
        throws InputException,UnsupportException
    {
        
        boolean ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_BOOLEAN &&
                pause.type != Parameter.TYPE_BOOLEAN_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_BOOLEAN_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Boolean )pause.value ).booleanValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return byte 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final byte getByte( String name,int no )
        throws InputException,UnsupportException
    {
        
        byte ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_BYTE &&
                pause.type != Parameter.TYPE_BYTE_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_BYTE ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Byte )pause.value ).byteValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return char 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final char getChar( String name,int no )
        throws InputException,UnsupportException
    {
        
        char ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_CHAR &&
                pause.type != Parameter.TYPE_CHARACTER_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_CHAR ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Character )pause.value ).charValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return short 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final short getShort( String name,int no )
        throws InputException,UnsupportException
    {
        
        short ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_SHORT &&
                pause.type != Parameter.TYPE_SHORT_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_SHORT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Short )pause.value ).shortValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return int 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final int getInt( String name,int no )
        throws InputException,UnsupportException
    {
        
        int ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_INT &&
                pause.type != Parameter.TYPE_INTEGER_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_INT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Integer )pause.value ).intValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return long 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final long getLong( String name,int no )
        throws InputException,UnsupportException
    {
        
        long ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_LONG &&
                pause.type != Parameter.TYPE_LONG_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_LONG ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Long )pause.value ).longValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return float 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final float getFloat( String name,int no )
        throws InputException,UnsupportException
    {
        
        float ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_FLOAT &&
                pause.type != Parameter.TYPE_FLOAT_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_FLOAT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Float )pause.value ).floatValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return double 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final double getDouble( String name,int no )
        throws InputException,UnsupportException
    {
        
        double ret ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_DOUBLE &&
                pause.type != Parameter.TYPE_DOUBLE_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_DOUBLE ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( ( Double )pause.value ).doubleValue() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return BigDecimal 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final BigDecimal getDecimal( String name,int no )
        throws InputException,UnsupportException
    {
        
        BigDecimal ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type != Parameter.TYPE_DECIMAL ){
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_DECIMAL ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( BigDecimal )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return java.sql.Date 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final java.sql.Date getDate( String name,int no )
        throws InputException,UnsupportException
    {
        
        java.sql.Date ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_TIMESTAMP ){
                
                ret = new java.sql.Date( ( ( java.sql.Timestamp )pause.value ).getTime() ) ;
                
            }else if( pause.type == Parameter.TYPE_TIME ){
                
                ret = new java.sql.Date( ( ( java.sql.Time )pause.value ).getTime() ) ;
                
            }else if( pause.type == Parameter.TYPE_DATE ){
                
                ret = ( java.sql.Date )pause.value ;
                
            }else{
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_DATE ) +
                    "]ではありません"
                ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return java.sql.Time 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final java.sql.Time getTime( String name,int no )
        throws InputException,UnsupportException
    {
        
        java.sql.Time ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_TIMESTAMP ){
                
                ret = new java.sql.Time( ( ( java.sql.Timestamp )pause.value ).getTime() ) ;
                
            }else if( pause.type == Parameter.TYPE_TIME ){
                
                ret = ( java.sql.Time )pause.value ;
                
            }else if( pause.type == Parameter.TYPE_DATE ){
                
                ret = new java.sql.Time( ( ( java.sql.Date )pause.value ).getTime() ) ;
                
            }else{
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_TIME ) +
                    "]ではありません"
                ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return java.sql.Timestamp 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final java.sql.Timestamp getTimestamp( String name,int no )
        throws InputException,UnsupportException
    {
        
        java.sql.Timestamp ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_TIMESTAMP ){
                
                ret = ( java.sql.Timestamp )pause.value ;
                
            }else if( pause.type == Parameter.TYPE_TIME ){
                
                ret = new java.sql.Timestamp( ( ( java.sql.Time )pause.value ).getTime() ) ;
                
            }else if( pause.type == Parameter.TYPE_DATE ){
                
                ret = new java.sql.Timestamp( ( ( java.sql.Date )pause.value ).getTime() ) ;
                
            }else{
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_TIMESTAMP ) +
                    "]ではありません"
                ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return byte[] 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final byte[] getBinary( String name,int no )
        throws InputException,UnsupportException
    {
        int i,j ;
        int len ;
        
        byte[] ret = null ;
        char[] chr = null ;
        String str = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_GROUP ){
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_BINARY ) +
                    "]ではありません"
                ) ;
                
            }else if( pause.type == Parameter.TYPE_BINARY ){
                
                ret = ( byte[] )pause.value ;
                
            }else if( pause.type == Parameter.TYPE_CHARS ){
                
                chr = ( char[] )pause.value ;
                len = chr.length ;
                ret = new byte[ len * 2 ] ;
                
                for( i = 0,j = 0 ; i < len ; i ++,j += 2 ){
                    
                    ret[ j ] = ( byte )( chr[ i ] & 0x000000ff ) ;
                    ret[ j+1 ] = ( byte )( ( chr[ i ] & 0x0000ff00 ) >> 8 ) ;
                    
                }
                
            }else if( pause.type == Parameter.TYPE_STRING ){
                
                str = ( String )pause.value ;
                ret = str.getBytes() ;
                
            }else{
                
                str = pause.value.toString() ;
                ret = str.getBytes() ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
            str = null ;
            chr = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return char[] 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final char[] getChars( String name,int no )
        throws InputException,UnsupportException
    {
        
        int len ;
        
        char[] ret = null ;
        String str = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_GROUP ){
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_CHARS ) +
                    "]ではありません"
                ) ;
                
            }else if( pause.type == Parameter.TYPE_CHARS ){
                
                ret = ( char[] )pause.value ;
                
            }else if( pause.type == Parameter.TYPE_STRING ){
                
                str = ( String )pause.value ;
                
                len = str.length() ;
                ret = new char[ len ] ;
                str.getChars( 0,len,ret,0 ) ;
                
            }else{
                
                str = pause.value.toString() ;
                
                len = str.length() ;
                ret = new char[ len ] ;
                str.getChars( 0,len,ret,0 ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
            str = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return String 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final String getString( String name,int no )
        throws InputException,UnsupportException
    {
        
        String ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type == Parameter.TYPE_GROUP ){
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_STRING ) +
                    "]ではありません"
                ) ;
                
            }else if( pause.type == Parameter.TYPE_STRING ){
                
                ret = ( String )pause.value ;
                
            }else{
                
                ret = pause.value.toString() ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Boolean 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Boolean getBooleanObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Boolean ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_BOOLEAN &&
                pause.type != Parameter.TYPE_BOOLEAN_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_BOOLEAN_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Boolean )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Byte 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Byte getByteObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Byte ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_BYTE &&
                pause.type != Parameter.TYPE_BYTE_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_BYTE_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Byte )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Character 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Character getCharacterObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Character ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_CHAR &&
                pause.type != Parameter.TYPE_CHARACTER_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_CHARACTER_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Character )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Short 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Short getShortObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Short ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_SHORT &&
                pause.type != Parameter.TYPE_SHORT_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_SHORT_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Short )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Integer 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Integer getIntegerObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Integer ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_INT &&
                pause.type != Parameter.TYPE_INTEGER_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_INTEGER_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Integer )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Long 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Long getLongObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Long ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_LONG &&
                pause.type != Parameter.TYPE_LONG_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_LONG_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Long )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Float 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Float getFloatObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Float ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_FLOAT &&
                pause.type != Parameter.TYPE_FLOAT_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_FLOAT_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Float )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Double 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Double getDoubleObject( String name,int no )
        throws InputException,UnsupportException
    {
        
        Double ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if(
                pause.type != Parameter.TYPE_DOUBLE &&
                pause.type != Parameter.TYPE_DOUBLE_OBJECT
            )
            {
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_DOUBLE_OBJECT ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Double )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象パラメータ情報の取得.
     * <BR><BR>
     * 対象の格納されているパラメータ情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を指定します.
     * @return Parameterput 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public final Parameter getGroup( String name,int no )
        throws InputException,UnsupportException
    {
        Parameter ret = null ;
        ParamPause pause = null ;
        
        try{
            
            pause = ( ParamPause )m_table.get( name,no ) ;
            
            if( pause.type != Parameter.TYPE_GROUP ){
                
                throw new UnsupportException(
                    "対象の格納パラメータ( name :" + name +
                    ", no :" + no +
                    ", type :" + Parameter.getTypeName( pause.type ) +
                    ")は、[" + Parameter.getTypeName( Parameter.TYPE_GROUP ) +
                    "]ではありません"
                ) ;
                
            }
            
            ret = ( Parameter )pause.value ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportException us ){
            throw us ;
        }finally{
            pause = null ;
        }
        
        return ret ;
        
    }
    
    
    /**
     * 格納されているデータ項番に対する、
     * パラメータタイプを取得.
     * <BR><BR>
     * 格納されているデータ項番に対する、
     * パラメータタイプを取得します.
     * <BR>
     * @param name 取得対象のキー名を設定します.
     * @param no キー名に対する要素位置を指定します.
     * @return byte 対象のパラメータタイプ情報が返されます.
     * @exception InputException 入力例外.
     */
    public final byte getType( String name,int no ) throws InputException
    {
        byte ret ;
        
        try{
            
            ret = ( ( ParamPause )m_table.get( name,no ) ).type ;
            
        }catch( InputException in ){
            throw in ;
        }
        
        return ret ;
    }
    
    /**
     * 指定キー名に対する、格納要素数の取得.
     * <BR><BR>
     * 指定されているキー名に対する、格納要素数を取得します.
     * <BR>
     * @param name 対象のキー名を指定します.
     * @return int 対象のキー名に対する格納データ数が返されます.
     */
    public final int getElements( String name )
    {
        return m_table.getElements( name ) ;
    }
    
    /**
     * 設定されているキー名群の取得.
     * <BR><BR>
     * 設定されているキー名群を取得します.
     * <BR>
     * @return String[] 設定されているキー名群が返されます.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getNames()
    {
        return m_table.getNames() ;
    }
    
    /**
     * 現在格納されているパラメータ数を取得.
     * <BR><BR>
     * 現在格納されているパラメータ数を取得します.
     * <BR>
     * @return int 格納されているパラメータ数が返されます.
     */
    public final int size()
    {
        return m_table.size() ;
    }
    
    
    
    /**
     * １パラメータオブジェクトの生成.
     */
    private final ParamPause createParamPause()
    {
        return new ParamPause() ;
    }
    
    /**
     * タイプ定義から、名前を取得.
     */
    private static final String getTypeName( byte type )
    {
        String ret = null ;
        
        switch( type ){
            
            case Parameter.TYPE_BOOLEAN : 
                ret = "boolean" ;
                break ;
            case Parameter.TYPE_BYTE : 
                ret = "byte" ;
                break ;
            case Parameter.TYPE_CHAR : 
                ret = "char" ;
                break ;
            case Parameter.TYPE_SHORT : 
                ret = "short" ;
                break ;
            case Parameter.TYPE_INT : 
                ret = "int" ;
                break ;
            case Parameter.TYPE_LONG : 
                ret = "long" ;
                break ;
            case Parameter.TYPE_FLOAT : 
                ret = "float" ;
                break ;
            case Parameter.TYPE_DOUBLE : 
                ret = "double" ;
                break ;
            case Parameter.TYPE_DECIMAL : 
                ret = "BigDecimal" ;
                break ;
            case Parameter.TYPE_DATE : 
                ret = "java.sql.Date" ;
                break ;
            case Parameter.TYPE_TIME : 
                ret = "java.sql.Time" ;
                break ;
            case Parameter.TYPE_TIMESTAMP : 
                ret = "java.sql.Timestamp" ;
                break ;
            case Parameter.TYPE_BINARY : 
                ret = "byte[]" ;
                break ;
            case Parameter.TYPE_CHARS : 
                ret = "char[]" ;
                break ;
            case Parameter.TYPE_STRING : 
                ret = "String" ;
                break ;
            case Parameter.TYPE_BOOLEAN_OBJECT : 
                ret = "Boolean" ;
                break ;
            case Parameter.TYPE_BYTE_OBJECT : 
                ret = "Byte" ;
                break ;
            case Parameter.TYPE_CHARACTER_OBJECT : 
                ret = "Character" ;
                break ;
            case Parameter.TYPE_SHORT_OBJECT : 
                ret = "Short" ;
                break ;
            case Parameter.TYPE_INTEGER_OBJECT : 
                ret = "Integer" ;
                break ;
            case Parameter.TYPE_LONG_OBJECT : 
                ret = "Long" ;
                break ;
            case Parameter.TYPE_FLOAT_OBJECT : 
                ret = "Float" ;
                break ;
            case Parameter.TYPE_DOUBLE_OBJECT : 
                ret = "Double" ;
                break ;
            case Parameter.TYPE_GROUP : 
                ret = "Parameter" ;
                break ;
            default :
                ret = "Unknown??" ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のパラメータ情報群をバイナリ化.
     */
    private static final byte[] ParameterTblToBinary( ArrayTable table )
    {
        int i,j ;
        int len ;
        int lenJ ;
        int length ;
        
        ByteUtil bu = null ;
        String key = null ;
        String[] keys = null ;
        byte[] binary = null ;
        byte[] ret = null ;
        
        if( table == null || table.size() == 0 ){
            
            if( table == null ){
                return null ;
            }else{
                return Parameter.END_TBL ;
            }
            
        }
        
        try{
            
            keys = table.getNames() ;
            len = keys.length ;
            
            bu = new ByteUtil() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                key = keys[ i ] ;
                keys[ i ] = null ;
                
                lenJ = table.getElements( key ) ;
                
                binary = Parameter.getBinaryName( key ) ;
                length = binary.length ;
                
                // キー名に対するキー名長[ binary - length ](4)
                bu.put( ConvertParam.convertInt( length ) ) ;
                
                // キー名.( String[Unicode] )
                bu.put( binary ) ;
                
                // キーに対する格納データ数.(4)
                bu.put( ConvertParam.convertInt( lenJ ) ) ;
                
                for( j = 0 ; j < lenJ ; j ++ ){
                    
                    // 設定データタイプ.(1)
                    // 設定データ長(4)
                    // 格納データ.
                    binary = Parameter.getBinaryValue( table.get( key, j ) ) ;
                    bu.put( binary ) ;
                    binary = null ;
                    
                }
                
                key = null ;
                
            }
            
            // エンドテーブル値(4).
            bu.add( Parameter.END_TBL ) ;
            
            // バイナリ変換.
            ret = bu.get() ;
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            try{
                bu.clear() ;
            }catch( Exception t ){
            }
            
            bu = null ;
            key = null ;
            keys = null ;
            binary = null ;
            
        }
        
        return ret ;
        
    }
    
    
    /**
     * 対象のバイナリをパラメータ情報群に変換.
     */
    private static final void binaryToParameterTbl( Parameter out,int off,byte[] in )
        throws InputException,ReadSyntaxException
    {
        int i ;
        
        int length ;
        int keyLen ;
        int elementLen ;
        int pauseLen ;
        int pnt ;
        
        String key = null ;
        ParamPause pause = null ;
        ArrayTable tbl = null ;
        ObjectArray oa = null ;
        
        if( out == null || in == null || ( length = in.length ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        pnt = off ;
        i = -1 ;
        key = null ;
            
        try{
            
            tbl = out.m_table ;
            oa = out.m_inParamTbl ;
            
            for( ;; ){
                
                // キー名に対するキー名長[ binary - length ](4)
                keyLen = ConvertParam.convertInt( pnt,in ) ;
                pnt += ConvertParam.LENGTH_INT ;
                
                //<><><><><><><><><><><><><><><><><><>
                // 格納データが、最終データであるか.
                //<><><><><><><><><><><><><><><><><><>
                if( keyLen == CKEND_TBL ){
                    break ;
                }
                
                // キー名.( String[Unicode] )
                key = Parameter.getStringName( pnt,keyLen,in ) ;
                pnt += keyLen ;
                
                // キーに対する格納データ数.(4)
                elementLen = ConvertParam.convertInt( pnt,in ) ;
                pnt += ConvertParam.LENGTH_INT ;
                
                // 各データ変換.
                for( i = 0 ; i < elementLen ; i ++ ){
                    
                    // 設定データタイプ.(1)
                    // 設定データ長(4)
                    // 格納データ.
                    pause =  out.createParamPause() ;
                    pauseLen = Parameter.getConvertValue( pause,oa,pnt,in ) ;
                    pnt += pauseLen ;
                    
                    // データ登録.
                    tbl.add( key,pause ) ;
                    
                }
                
            }
            
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }catch( Exception t ){
            
            throw new ReadSyntaxException(
                "バイナリPoint(" + pnt +
                ")キー名(" + key +
                "):登録項番(" + i +
                ")で情報不正が確認されました"
            ) ;
            
        }finally{
            
            key = null ;
            pause = null ;
            tbl = null ;
            oa = null ;
            
        }
        
    }
    
    
    
    /**
     * コンバートバイナリ名.
     * name.getBytes( Unicode ) ;
     */
    private static final byte[] getBinaryName( String name )
    {
        return ConvertParam.convertString( name ) ;
    }
    
    /**
     * コンバート文字列.
     * new String( binary,Unicode ) ;
     * バイナリを、文字列に変換.
     */
    private static final String getStringName( int off,int length,byte[] binary )
    {
        String ret = null ;
        
        try{
            ret = ConvertParam.convertString( off,length,binary ) ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * コンバート設定値.
     * 1.パラメータタイプ.
     * 2.バイナリデータ長.
     * 3.バイナリ.
     */
    private static final byte[] getBinaryValue( Object value )
    {
        
        int len ;
        
        ParamPause pause = null ;
        byte[] bin = null ;
        byte[] ret = null ;
        
        pause = ( ParamPause )value ;
        
        switch( pause.type ){
            
            case Parameter.TYPE_BOOLEAN : 
            case Parameter.TYPE_BOOLEAN_OBJECT : 
                bin = ConvertParam.convertBoolean( ( ( Boolean )pause.value ).booleanValue() ) ;
                break ;
            case Parameter.TYPE_BYTE : 
            case Parameter.TYPE_BYTE_OBJECT : 
                bin = ConvertParam.convertByte( ( ( Byte )pause.value ).byteValue() ) ;
                break ;
            case Parameter.TYPE_CHAR : 
            case Parameter.TYPE_CHARACTER_OBJECT : 
                bin = ConvertParam.convertChar( ( ( Character )pause.value ).charValue() ) ;
                break ;
            case Parameter.TYPE_SHORT : 
            case Parameter.TYPE_SHORT_OBJECT : 
                bin = ConvertParam.convertShort( ( ( Short )pause.value ).shortValue() ) ;
                break ;
            case Parameter.TYPE_INT : 
            case Parameter.TYPE_INTEGER_OBJECT : 
                bin = ConvertParam.convertInt( ( ( Integer )pause.value ).intValue() ) ;
                break ;
            case Parameter.TYPE_LONG : 
            case Parameter.TYPE_LONG_OBJECT : 
                bin = ConvertParam.convertLong( ( ( Long )pause.value ).longValue() ) ;
                break ;
            case Parameter.TYPE_FLOAT : 
            case Parameter.TYPE_FLOAT_OBJECT : 
                bin = ConvertParam.convertFloat( ( ( Float )pause.value ).floatValue() ) ;
                break ;
            case Parameter.TYPE_DOUBLE : 
            case Parameter.TYPE_DOUBLE_OBJECT : 
                bin = ConvertParam.convertDouble( ( ( Double )pause.value ).doubleValue() ) ;
                break ;
            case Parameter.TYPE_DECIMAL : 
                bin = ConvertParam.convertDecimal( ( BigDecimal )pause.value ) ;
                break ;
            case Parameter.TYPE_DATE : 
                bin = ConvertParam.convertLong( ( ( java.sql.Date )pause.value ).getTime() ) ;
                break ;
            case Parameter.TYPE_TIME : 
                bin = ConvertParam.convertLong( ( ( java.sql.Time )pause.value ).getTime() ) ;
                break ;
            case Parameter.TYPE_TIMESTAMP : 
                bin = ConvertParam.convertLong( ( ( java.sql.Timestamp )pause.value ).getTime() ) ;
                break ;
            case Parameter.TYPE_BINARY : 
                bin = ( byte[] )pause.value ;
                break ;
            case Parameter.TYPE_CHARS : 
                bin = ConvertParam.convertChars( ( char[] )pause.value ) ;
                break ;
            case Parameter.TYPE_STRING : 
                bin = ConvertParam.convertString( ( String )pause.value ) ;
                break ;
            case Parameter.TYPE_GROUP : 
                bin = Parameter.ParameterTblToBinary( ( ( Parameter)pause.value ).m_table ) ;
                break ;
            default :
                bin = null ;
        }
        
        if( bin == null ){
            
            ret = new byte[ 5 ] ;
            ret[ 0 ] = pause.type ;
            ret[ 1 ] = 0x00 ;
            ret[ 2 ] = 0x00 ;
            ret[ 3 ] = 0x00 ;
            ret[ 4 ] = 0x00 ;
            
        }else{
            
            len = bin.length ;
            ret = new byte[ len + 5 ] ;
            
            ret[ 0 ] = pause.type ;
            ret[ 1 ] = ( byte )( len & 0x000000ff ) ;
            ret[ 2 ] = ( byte )( ( len & 0x0000ff00 ) >> 8 ) ;
            ret[ 3 ] = ( byte )( ( len & 0x00ff0000 ) >> 16 ) ;
            ret[ 4 ] = ( byte )( ( len & 0xff000000 ) >> 24 ) ;
            
            System.arraycopy( bin,0,ret,5,len ) ;
            
        }
        
        pause = null ;
        bin = null ;
        
        return ret ;
        
    }
    
    /**
     * バイナリ情報から、対象のオブジェクトを取得.
     */
    private static final int getConvertValue( ParamPause out,ObjectArray oa,int off,byte[] binary )
        throws ReadSyntaxException
    {
        
        int length ;
        int pnt ;
        int ret ;
        
        pnt = off ;
        
        try{
            
            out.type = binary[ pnt ] ;
            pnt += 1 ;
            
            length = ConvertParam.convertInt( pnt,binary ) ;
            pnt += ConvertParam.LENGTH_INT ;
            
            if( length == 0 ){
                
                out.value = null ;
                
            }else{
                
                switch( out.type ){
                    
                    case Parameter.TYPE_BOOLEAN : 
                    case Parameter.TYPE_BOOLEAN_OBJECT : 
                        out.value = new Boolean( ConvertParam.convertBoolean( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_BYTE : 
                    case Parameter.TYPE_BYTE_OBJECT : 
                        out.value = new Byte( ConvertParam.convertByte( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_CHAR : 
                    case Parameter.TYPE_CHARACTER_OBJECT : 
                        out.value = new Character( ConvertParam.convertChar( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_SHORT : 
                    case Parameter.TYPE_SHORT_OBJECT : 
                        out.value = new Short( ConvertParam.convertShort( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_INT : 
                    case Parameter.TYPE_INTEGER_OBJECT : 
                        out.value = new Integer( ConvertParam.convertInt( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_LONG : 
                    case Parameter.TYPE_LONG_OBJECT : 
                        out.value = new Long( ConvertParam.convertLong( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_FLOAT : 
                    case Parameter.TYPE_FLOAT_OBJECT : 
                        out.value = new Float( ConvertParam.convertFloat( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_DOUBLE : 
                    case Parameter.TYPE_DOUBLE_OBJECT : 
                        out.value = new Double( ConvertParam.convertDouble( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_DECIMAL : 
                        out.value = ConvertParam.convertDecimal( pnt,binary ) ;
                        break ;
                    case Parameter.TYPE_DATE : 
                        out.value = new java.sql.Date( ConvertParam.convertLong( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_TIME : 
                        out.value = new java.sql.Time( ConvertParam.convertLong( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_TIMESTAMP : 
                        out.value = new java.sql.Timestamp( ConvertParam.convertLong( pnt,binary ) ) ;
                        break ;
                    case Parameter.TYPE_BINARY : 
                        out.value = new byte[ length ] ;
                        System.arraycopy( binary,pnt,( byte[] )out.value,0,length ) ;
                        break ;
                    case Parameter.TYPE_CHARS : 
                        out.value = ConvertParam.convertChars( pnt,length,binary ) ;
                        break ;
                    case Parameter.TYPE_STRING : 
                        out.value = ConvertParam.convertString( pnt,length,binary ) ;
                        break ;
                    case Parameter.TYPE_GROUP : 
                        out.value = new Parameter() ;
                        Parameter.binaryToParameterTbl( ( Parameter)out.value,pnt,binary ) ;
                        oa.add( out.value ) ;
                        break ;
                    default :
                        
                        throw new ReadSyntaxException(
                            "バイナリPoint(" + pnt +
                            ")で範囲外のパラメータタイプ[" + out.value +
                            "]が検出されました"
                        ) ;
                        
                }
                
            }
            
            ret = length + 5 ;
            
        }catch( ReadSyntaxException rs ){
            out.value = null ;
            throw rs ;
        }catch( Exception t ){
            
            out.value = null ;
            
            throw new ReadSyntaxException(
                "バイナリPoint(" + pnt +
                ")パラメータタイプ[" + out.value +
                "]で例外が検出されました"
            ) ;
            
        }
        
        return ret ;
        
    }
    
}

