/*
 * @(#)SearchLong.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.util.HashSet;
import java.util.Iterator;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.array.LongArray;


/**
 * long型検索管理.
 * <BR><BR>
 * long型の情報の検索を管理します.
 *
 * @version     1.00, 2004/10/28
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class SearchLong
{
    /**
     * シングルデータ管理.
     */
    private HashSet m_hash = null ;
    
    /**
     * コンストラクタ.
     */
    public SearchLong()
    {
        this.create() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     */
    public final void create()
    {
        m_hash = new HashSet() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        if( m_hash != null ){
            m_hash.clear() ;
        }
        m_hash = null ;
    }
    
    /**
     * 情報を追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param el 追加対象の要素情報を設定します.
     */
    public final void add( long el )
    {
        if( m_hash != null ){
            m_hash.add( new Long( el ) ) ;
        }
    }
    
    /**
     * 情報を追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param el 追加対象の要素情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void add( Long el )
        throws InputException
    {
        if( el == null ){
            throw new InputException( "引数は存在しません" ) ;
        }
        if( m_hash != null ){
            m_hash.add( el ) ;
        }
    }
    
    /**
     * 情報を削除.
     * <BR><BR>
     * 情報を削除します.
     * <BR>
     * @param el 削除対象の要素情報を設定します.
     */
    public final void remove( long el )
    {
        if( m_hash != null ){
            m_hash.remove( new Long( el ) ) ;
        }
    }
    
    /**
     * 情報を削除.
     * <BR><BR>
     * 情報を削除します.
     * <BR>
     * @param el 削除対象の要素情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void remove( Long el )
        throws InputException
    {
        if( el == null ){
            throw new InputException( "引数は存在しません" ) ;
        }
        if( m_hash != null ){
            m_hash.remove( el ) ;
        }
    }
    
    /**
     * 格納情報長を取得.
     * <BR><BR>
     * 格納されている情報長が返されます.
     * <BR>
     * @return int 格納されている情報長が返されます.
     */
    public final int size()
    {
        if( m_hash != null ){
            return m_hash.size() ;
        }
        return 0 ;
    }
    
    /**
     * 格納情報を全て取得.
     * <BR><BR>
     * 格納されている情報を全て取得します.
     * <BR>
     * @return long[] 格納されている全ての要素情報が返されます.
     */
    public final long[] getAll()
    {
        int len ;
        Long tmp = null ;
        Iterator it = null ;
        LongArray ary = null ;
        long[] ret = null ;
        
        if( m_hash != null ){
            
            it = m_hash.iterator() ;
            if( it != null ){
                ary = new LongArray() ;
                while( it.hasNext() ){
                    
                    if( ( tmp = ( Long )it.next() ) != null ){
                        
                        ary.add( tmp.longValue() ) ;
                        
                    }
                    tmp = null ;
                    
                }
                
                if( ( len = ary.size() ) > 0 ){
                    
                    ret = new long[ len ] ;
                    System.arraycopy( ary.getObjects(),0,ret,0,len ) ;
                    ary.clear() ;
                    
                }
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 対象情報が存在するかチェック.
     * <BR><BR>
     * 対象の情報が存在するかチェックします.
     * <BR>
     * @param el チェック対象の情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isData( long el )
    {
        if( m_hash != null ){
            return m_hash.contains( new Long( el ) ) ;
        }
        return false ;
    }
    
    /**
     * 対象情報が存在するかチェック.
     * <BR><BR>
     * 対象の情報が存在するかチェックします.
     * <BR>
     * @param el チェック対象の情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isData( Long el )
    {
        if( el != null && m_hash != null ){
            return m_hash.contains( el ) ;
        }
        return false ;
    }
}
