/*
 * @(#)ObjectArray.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.array;

import java.io.Serializable;
import java.util.Arrays;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;


/**
 * ObjectArrayオブジェクト.
 * <BR><BR>
 * オブジェクト配列を管理するオブジェクトです.<BR>
 * また、このオブジェクトは同期されていません.
 *
 * @version     1.00, 2004/02/02
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ObjectArray implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ObjectArray.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 配列管理.
     */
    private Object[] m_array = null ;
    
    /**
     * 開始時管理配列数.
     */
    private int m_startLength = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 配列管理数.
     */
    private int m_length = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 現在格納配列数.
     */
    private int m_nowSize = 0 ;
    
    /**
     * 情報保全タイマー値.
     */
    private long m_timer = 0L ;
    
    /**
     * コンストラクタ.
     */
    public ObjectArray()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のバッファ係数を設定します.
     * <BR>
     * @param length 対象のバッファ係数を設定します.<BR>
     *               設定可能な最大値は[9999]です.<BR>
     *               設定可能な最小値は[2]です.<BR>
     *               また、これらの設定範囲外を設定した場合、
     *               デフォルトの値[16]となります.
     */
    public ObjectArray( int length )
    {
        
        m_startLength = 
            (
                length < ArrayDef.MIN_START_LENGTH ||
                length > ArrayDef.MAX_START_LENGTH
            ) ? ArrayDef.DEFAULT_START_LENGTH :length ;
        m_length = m_startLength ;
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        m_array = null ;
        m_length = m_startLength ;
        m_nowSize = 0 ;
        //this.createTimer() ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象のオブジェクト情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( Object value )
        throws InputException
    {
        int length ;
        int nowSize ;
        
        Object[] info = null ;
        Object[] tmp = null ;
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        info = m_array ;
        length = m_length ;
        nowSize = m_nowSize ;
        
        if( info == null ) {
            info = new Object[ length ] ;
            info[ nowSize ] = value ;
            
            m_array = info ;
            //this.createTimer() ;
        }
        else if( info.length <= nowSize ) {
            
            int i ;
            
            length *= 2 ;
            tmp = info ;
            info = new Object[ length ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            for( i = 0 ; i < nowSize ; i ++ ){
                tmp[ i ] = null ;
            }
            
            m_length = length ;
            m_array = info ;
            //this.createTimer() ;
            
        }
        else {
            info[ nowSize ] = value ;
        }
        
        m_nowSize ++ ;
        
        /*
        try{
            
            info[ nowSize ] = value ;
            
        }catch( NullPointerException nul ){
            
            info = new Object[ length ] ;
            info[ nowSize ] = value ;
            
            m_array = info ;
            //this.createTimer() ;
            
        }catch( IndexOutOfBoundsException io ){
            
            int i ;
            
            length *= 2 ;
            tmp = info ;
            info = new Object[ length ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            for( i = 0 ; i < nowSize ; i ++ ){
                tmp[ i ] = null ;
            }
            
            m_length = length ;
            m_array = info ;
            //this.createTimer() ;
            
        }finally{
            
            m_nowSize ++ ;
            
            tmp = null ;
            info = null ;
            
        }
        */
        
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,Object value )
        throws InputException
    {
        int nowLen ;
        
        nowLen = m_nowSize ;
        
        if( value == null || no < 0 || no >= nowLen ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_array[ no ] = value ;
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return Object 削除されたオブジェクト情報が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final Object remove( int no )
    {
        int i ;
        int nowSize ;
        int length ;
        int newLength ;
        Object[] info = null ;
        Object[] tmp = null ;
        
        Object ret = null ;
        
        nowSize = m_nowSize ;
        length = m_length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return null ;
        }
        
        info = m_array ;
        
        ret = info[ no ] ;
        info[ no ] = null ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,1,info,0,( nowSize-1 ) ) ;
            
            info[ nowSize - 1 ] = null ;
            
        }else if( ( nowSize - no ) != 1 ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+1,info,no,nowSize-(no+1) ) ;
            
            info[ nowSize - 1 ] = null ;
            
        }
        
        nowSize -- ;
        
        if(
            nowSize != 0 &&
            ( length * ArrayDef.REDUCTION_ARRAY ) >= nowSize
        )
        {
            
            newLength = length / 2 ;
            tmp = new Object[ newLength ] ;
            System.arraycopy( info,0,tmp,0,newLength ) ;
            
            for( i = 0 ; i < nowSize ; i ++ ){
                info[ i ] = null ;
            }
            
            info = null ;
            info = tmp ;
            m_length = newLength ;
            
            //this.createTimer() ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        m_array = info ;
        m_nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return Object 取得された情報が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final Object get( int no )
    {
        
        if( no < 0 || no >= m_nowSize ){
            return null ;
        }
        
        return m_array[ no ] ;
    }
    
    /**
     * データを昇順でソート.
     * <BR><BR>
     * データを昇順でソートします.
     */
    public final void sort()
    {
        if( m_nowSize != 0 ){
            Arrays.sort( m_array,0,m_nowSize ) ;
        }
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( Object key )
    {
        return this.indexOf( key,0 ) ;
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( Object key,int index )
    {
        int len ;
        int ret ;
        Object[] tmp = null ;
        
        if( m_nowSize != 0 ){
            this.sort() ;
            len = m_nowSize - index ;
            tmp = new Object[ len ] ;
            System.arraycopy( m_array,index,tmp,0,len ) ;
            ret = Arrays.binarySearch( tmp,key ) ;
            tmp = null ;
            ret = ( ret < 0 || ret >= m_nowSize ) ? -1 : ret ;
        }
        else{
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のオブジェクト配列を取得.
     * <BR><BR>
     * 対象のオブジェクト配列を取得します.<BR>
     * また、このオブジェクト配列は配列の再構成を行った場合、
     * 情報の保証は行われません.<BR>
     * また、このオブジェクト群は基本的の読み込み専用で利用する
     * ことを進めます.
     * <BR>
     * @return Object[] 対象のオブジェクト配列が返されます.
     */
    public final Object[] getObjects()
    {
        return m_array ;
    }
    
    /**
     * 情報保全タイマー値の取得.
     * <BR><BR>
     * 情報保全タイマー値を取得します.<BR>
     * この情報は、メソッド[ObjectArray#getObjects()]のデータ安全有無を
     * 示唆するために利用します.<BR>
     * 方法として、メソッド[ObjectArray#getObjects()]実施後、
     * 当メソッドを呼び出し、タイマー値を保持します.<BR>
     * その後、メソッド[ObjectArray#getObjects()]を呼び出し、
     * 値が変化している場合、そのオブジェクト群は有効でなくなった意味を持ちます.
     * <BR>
     * @return int 情報保全タイマー値が返されます.
     */
    public final long getTimer()
    {
        return m_timer ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public final int size()
    {
        return m_nowSize ;
    }
    
    
    
    
    /**
     * 情報保全タイマー値の更新.
     */
    private final void createTimer()
    {
        m_timer = System.currentTimeMillis() ;
    }
    
}

