/*
 * @(#)SimpleFlagBox.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.box;

import java.io.Serializable;
import java.util.Arrays;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.array.IntArray;


/**
 * 単純フラグ固定配列.
 * <BR><BR>
 * 単純フラグ固定配列を管理します.
 * 
 * @version 2006/07/28
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class SimpleFlagBox implements Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            SimpleFlagBox.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * １データ内要素数.
     */
    private static final int ELEMENT_LENGTH = 64 ;
    
    /**
     * １データ内要素内単位.
     */
    private static final int ELEMENT_MASK = ELEMENT_LENGTH - 1 ;
    
    /**
     * データビット数.
     */
    private static final int ELEMENT_BITCOUNT = 6 ;
    
    
    
    /**
     * フラグ管理.
     */
    private long[] flagTable = null ;
    
    /**
     * フラグ個数.
     */
    private int maxTableSize = 0 ;
    
    /**
     * 有効フラグ個数.
     */
    private int nowTableSize = 0 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public SimpleFlagBox() {
        
        
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public SimpleFlagBox( int size )
        throws InputException {
        
        this.create( size ) ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.clear() ;
        } catch( Exception t ) {
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public void create( int size )
        throws InputException {
        
        int no ;
        
        if ( size < 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        no = ( ( ( size & SimpleFlagBox.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( size & (~SimpleFlagBox.ELEMENT_MASK) ) >> SimpleFlagBox.ELEMENT_BITCOUNT ) ;
        
        flagTable = new long[ no ] ;
        Arrays.fill( flagTable,0L ) ;
        
        maxTableSize = size ;
        nowTableSize = 0 ;
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        
        flagTable = null ;
        maxTableSize = 0 ;
        nowTableSize = 0 ;
        
    }
    
    /**
     * 対象要素を有効/無効に設定.
     * <BR><BR>
     * 対象の要素を有効/無効に設定します.
     * <BR>
     * @param flg 対象の条件を設定します.<BR>
     *            [true]を設定した場合有効となります.<BR>
     *            [false]を設定した場合無効となります.
     * @param no 設定対象の項番を設定します.
     */
    public void add( boolean flg,int no ) {
        
        int indNo ;
        int indOff ;
        
        long[] flags = null ;
        
        flags = flagTable ;
        
        if ( flags == null || no < 0 || no >= maxTableSize ) {
            return ;
        }
        
        indNo = ( no & (~SimpleFlagBox.ELEMENT_MASK) ) >> SimpleFlagBox.ELEMENT_BITCOUNT ;
        indOff = no & SimpleFlagBox.ELEMENT_MASK ;
        
        if ( flg == true ) {
            
            if ( ( flags[ indNo ] & ( 1L << indOff ) ) == 0 ) {
                nowTableSize ++ ;
            }
            
            flags[ indNo ] |= ( ( 1L << indOff ) & 0xffffffffffffffffL ) ;
            
        }
        else {
            
            if ( ( flags[ indNo ] & ( 1L << indOff ) ) != 0 ) {
                nowTableSize -- ;
            }
            
            flags[ indNo ] &= ( ( ~( 1L << indOff ) ) & 0xffffffffffffffffL ) ;
            
        }
        
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象の要素を取得します.
     * <BR>
     * @param no 取得対象の要素番号を設定します.
     * @return boolean 取得された内容が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public boolean get( int no ) {
        
        long[] flags = null ;
        
        flags = flagTable ;
        
        if ( flags == null || no < 0 || no >= maxTableSize ) {
            return false ;
        }
        
        return (
            (
                flags[ ( no & (~SimpleFlagBox.ELEMENT_MASK) ) >> SimpleFlagBox.ELEMENT_BITCOUNT ] &
                ( 1L << ( no & SimpleFlagBox.ELEMENT_MASK ) ) 
            ) == 0
        ) ? false : true ;
    }
    
    /**
     * 有効領域を検索.
     * <BR><BR>
     * 有効領域を検索します.
     * <BR>
     * @return int 有効な項番が返されます.<BR>
     *             空き領域が存在しない場合[-1]が返されます.
     */
    public int useSearchSector() {
        
        int i,j ;
        int len ;
        int max ;
        int size ;
        boolean flg ;
        int ret ;
        
        long[] flags = null ;
        
        ret = -1 ;
        size = 0 ;
        
        size = maxTableSize ;
        flags = flagTable ;
        
        if( flags == null ){
            return -1 ;
        }
        
        len = SimpleFlagBox.ELEMENT_LENGTH ;
        max = flags.length ;
        
        // 情報を検索.
        for ( i = 0,ret = -1,flg = false ; ( i < max && flg == false ) ; i ++ ) {
            
            // 空き領域が存在する場合.
            if ( flags[ i ] != 0xffffffffffffffffL ) {
                
                // 空き条件を調べる.
                for ( j = 0 ; j < len ; j ++ ) {
                    
                    // 空きデータが検知された場合.
                    if ( ( flags[ i ] & ( 1L << j ) ) == 0 ) {
                        
                        ret = ( i * SimpleFlagBox.ELEMENT_LENGTH ) + j ;
                        
                        if ( ret >= size ) {
                            ret = -1 ;
                        }
                        else {
                            flg = true ;
                        }
                        
                        break ;
                        
                    }
                    
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 有効領域群を検索.
     * <BR><BR>
     * 有効領域群を検索します.
     * <BR>
     * @return int[] 有効な項番群が返されます.<BR>
     *              空き領域が存在しない場合[null]が返されます.
     */
    public int[] useSearchSectors() {
        return this.useSearchSectors( -1 ) ;
    }
    
    /**
     * 有効領域群を検索.
     * <BR><BR>
     * 有効領域群を検索します.
     * <BR>
     * @param length 領域取得長を設定します.<BR>
     *               [-1]の場合、全ての条件で取得します.
     * @return int[] 有効な項番群が返されます.<BR>
     *              空き領域が存在しない場合[null]が返されます.
     */
    public int[] useSearchSectors( int length ) {
        
        int i,j ;
        int len ;
        int max ;
        int size = 0 ;
        int cnt = 0 ;
        boolean flg ;
        int point ;
        
        long[] flags = null ;
        
        size = maxTableSize ;
        flags = flagTable ;
        
        if( flags == null ){
            return null ;
        }
        
        if( length <= 0 ) {
            length = Integer.MAX_VALUE ;
        }
        
        len = SimpleFlagBox.ELEMENT_LENGTH ;
        max = flags.length ;
        
        IntArray array = new IntArray() ;
        
        // 情報を検索.
        for ( i = 0,point = -1,flg = false ; ( i < max && flg == false ) ; i ++ ) {
            
            // 空き領域が存在する場合.
            if ( flags[ i ] != 0xffffffffffffffffL ) {
                
                // 空き条件を調べる.
                for ( j = 0 ; j < len ; j ++ ) {
                    
                    // 空きデータが検知された場合.
                    if ( ( flags[ i ] & ( 1L << j ) ) == 0 ) {
                        
                        point = ( i * SimpleFlagBox.ELEMENT_LENGTH ) + j ;
                        
                        if ( point < size ) {
                            array.add( point ) ;
                            cnt ++ ;
                            
                            if( cnt >= length ) {
                                break ;
                            }
                        }
                        else {
                            flg = true ;
                            break ;
                        }
                        
                    }
                    
                }
                
            }
            
        }
        
        // 空き情報郡が存在する場合.
        if( ( len = array.size() ) > 0 ) {
            int[] ret = new int[ len ] ;
            System.arraycopy( array.getObjects(),0,ret,0,len ) ;
            return ret ;
        }
        
        return null ;
        
    }
    
    /**
     * 管理フラグ個数を取得.
     * <BR><BR>
     * 管理されているフラグ個数を取得します.
     * <BR>
     * @return int 管理されているフラグ個数が返されます.
     */
    public int getMaxSize() {
        
        return maxTableSize ;
        
    }
    
    /**
     * 有効となっているフラグ個数を取得.
     * <BR><BR>
     * 有効となっているフラグ個数を取得します.
     * <BR>
     * @return int 有効となっているフラグ個数が返されます.
     */
    public int getUseSize() {
        
        return nowTableSize ;
        
    }
    
    /**
     * 空き条件が存在するかチェック.
     * <BR><BR>
     * 空き条件が存在するかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、空き条件は存在します.<BR>
     *                 [false]が返された場合、空き条件は存在しません.
     */
    public boolean isSpace() {
        return ( maxTableSize == nowTableSize ) ? false : true ;
    }
    
    /**
     * 利用可能チェック.
     * <BR><BR>
     * オブジェクトが利用可能であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合利用可能です.<BR>
     *                 [false]が返された場合利用不可能です.
     */
    public boolean isUseObject() {
        
        return ( flagTable != null ) ? true : false ;
        
    }
    
}

