/*
 * @(#)AbstractMaachangConnect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect ;

import java.net.InetAddress;

import org.maachang.connect.table.Cb32Table;
import org.maachang.connect.table.CompletionBean;
import org.maachang.connect.table.CompletionTable;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.thread.MaachangConnectThreadFactory;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.TimeoutException;
import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;

/**
 * AbstractMaachangConnectプロトコル.
 * <BR><BR>
 * MaachangConnectプロトコルAbstract実装.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public abstract class AbstractMaachangConnect implements MaachangConnectInterface {
    
    /**
     * デフォルトバッファ長.
     * 512Kbyte.
     */
    protected static final int DEFAULT_BUFFER = 524288 ;
    
    /**
     * 最小バッファ長.
     * 128Kbyte.
     */
    protected static final int MIN_BUFFER = 131072 ;
    
    /**
     * 最小バッファ長.
     * 8Mbyte.
     */
    protected static final int MAX_BUFFER = 8388608 ;
    
    /**
     * スレッドFactory.
     */
    protected MaachangConnectThreadFactory factory = null ;
    
    /**
     * 同期オブジェクト.
     */
    protected final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public AbstractMaachangConnect() {
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * 接続状態をクローズします.
     */
    public void close() {
        if( factory != null ) {
            factory.destroy() ;
            factory = null ;
        }
        sync.clear() ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,byte[] binary )
        throws InputException,AccessException {
        this.send( address,port,null,binary ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,BinResource binary )
        throws InputException,AccessException {
        this.send( address,port,null,binary ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,String cb32Word,byte[] binary )
        throws InputException,AccessException {
        
        if( address == null || port < 0 || port > 65535 ||
            binary == null || binary.length <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.isOpen() == false ) {
            throw new AccessException( "バインドされていません" ) ;
        }
        
        try {
            
            MaachangConnectTableFactory table = null ;
            
            synchronized( sync.get() ) {
                
                table = factory.getTableFactory() ;
                
                if( table.isCb32Flag() == true && cb32Word != null &&
                    cb32Word.length() > 0 ) {
                    Cb32Table cb32Table = table.getCb32Table() ;
                    if( cb32Table.isWord( cb32Word ) == false ) {
                        cb32Word = null ;
                    }
                }
                else {
                    cb32Word = null ;
                }
                
                
            }
            
            ResourceType resType = table.getUnderReceiveTable().getResourceType() ;
            BinResource bin = Resource.createBinResource( resType,binary ) ;
            table.getConnectTable().startConnect( address,port,cb32Word,bin ) ;
            
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,String cb32Word,BinResource binary )
        throws InputException,AccessException {
        
        if( address == null || port < 0 || port > 65535 ||
            binary == null || binary.size() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.isOpen() == false ) {
            throw new AccessException( "バインドされていません" ) ;
        }
        
        try {
            
            MaachangConnectTableFactory table = null ;
            
            synchronized( sync.get() ) {
                
                table = factory.getTableFactory() ;
                
                if( table.isCb32Flag() == true && cb32Word != null &&
                    cb32Word.length() > 0 ) {
                    Cb32Table cb32Table = table.getCb32Table() ;
                    if( cb32Table.isWord( cb32Word ) == false ) {
                        cb32Word = null ;
                    }
                }
                else {
                    cb32Word = null ;
                }
                
            }
            
            table.getConnectTable().startConnect(
                address,port,cb32Word,binary ) ;
            
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out )
        throws AccessException {
        return this.receive( out,null ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限タイムアウト待ちです.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out,int timeout )
        throws AccessException {
        return receive( out,null,timeout ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @param cb32Word 対象の暗号ワード情報が格納されます.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out,String[] cb32Word )
        throws AccessException {
        
        if( this.isOpen() == false ) {
            throw new AccessException( "バインドされていません" ) ;
        }
        
        CompletionBean bean = null ;
        
        try {
            CompletionTable table = null ;
            synchronized( sync.get() ) {
                 table = factory.getTableFactory().
                    getCompletionTable() ;
            }
            bean = table.get() ;
        } catch( Exception e ) {
            bean = null ;
        }
        
        if( bean != null ) {
            
            if( out != null ) {
                out.create( bean.getAddress(),bean.getPort() ) ;
            }
            if( cb32Word != null && cb32Word.length == 1 ) {
                cb32Word[ 0 ] = bean.getCb32Word() ;
            }
            
            return bean.getReceiveData() ;
            
        }
        
        return null ;
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限タイムアウト待ちです.
     * @param cb32Word 対象の暗号ワード情報が格納されます.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out,String[] cb32Word,int timeout )
        throws AccessException {
        
        if( this.isOpen() == false ) {
            throw new AccessException( "バインドされていません" ) ;
        }
        
        if( timeout <= 0 ) {
            timeout = Integer.MAX_VALUE ;
        }
        
        long time = ( long )( timeout & 0x00000000ffffffffL ) +
            System.currentTimeMillis() ;
        
        CompletionBean bean = null ;
        CompletionTable table = null ;
        
        try {
            synchronized( sync.get() ) {
                 table = factory.getTableFactory().
                    getCompletionTable() ;
            }
            
            for( ;; ) {
                
                bean = table.get() ;
                
                if( bean != null ) {
                    break ;
                }
                else if( time <= System.currentTimeMillis() ) {
                    throw new TimeoutException( "タイムアウト例外" ) ;
                }
                
                UtilCom.idleTime() ;
                
            }
            
            if( out != null ) {
                out.create( bean.getAddress(),bean.getPort() ) ;
            }
            if( cb32Word != null && cb32Word.length == 1 ) {
                cb32Word[ 0 ] = bean.getCb32Word() ;
            }
            
            return bean.getReceiveData() ;
        
        } catch( Exception e ) {
            return null ;
        }
        
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     * @exception AccessException アクセス例外.
     */
    public InetAddress getBindAddress()
        throws AccessException {
        
        InetAddress ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.getUdp().getLocalAddress() ;
            }
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * バインドポート番号を取得.
     * <BR><BR>
     * バインドポート番号を取得します.
     * <BR>
     * @return int バインドポート番号が返されます.
     * @exception AccessException アクセス例外.
     */
    public int getBindPort()
        throws AccessException {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.getUdp().getLocalPort() ;
            }
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * バッファサイズを取得.
     * <BR><BR>
     * バッファサイズを取得します.
     * <BR>
     * @return int バッファサイズが返されます.
     * @exception AccessException アクセス例外.
     */
    public int getBuffer()
        throws AccessException {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.getUdp().getBuffer() ;
            }
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * 暗号ワード条件を設定.
     * <BR><BR>
     * 暗号ワード条件を設定します.
     * <BR>
     * @param cb32 暗号ワード条件を追加する条件を設定します.
     */
    public void setCb32( boolean cb32 ) {
        
        if( this.isOpen() == false ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                Cb32Table tbl = factory.getTableFactory().getCb32Table() ;
                
                if( cb32 == true ) {
                    if( tbl == null ) {
                        factory.getTableFactory().setCb32Table( new Cb32Table() ) ;
                    }
                }
                else if( tbl != null ) {
                    factory.getTableFactory().setCb32Table( null ) ;
                }
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 暗号ワードを追加.
     * <BR><BR>
     * 暗号ワードを追加します.
     * <BR>
     * @param word 追加対象の暗号ワードを設定します.
     */
    public void addWord( String word ) {
        
        if( this.isOpen() == false ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                Cb32Table tbl = factory.getTableFactory().getCb32Table() ;
                if( tbl != null ) {
                    tbl.add( word ) ;
                }
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 暗号ワードを削除.
     * <BR><BR>
     * 対象の暗号ワードを削除します.
     * <BR>
     * @param word 削除対象の暗号ワードを設定します.
     */
    public void removeWord( String word ) {
        
        if( this.isOpen() == false ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                Cb32Table tbl = factory.getTableFactory().getCb32Table() ;
                if( tbl != null ) {
                    tbl.remove( word ) ;
                }
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 暗号ワード一覧を取得.
     * <BR><BR>
     * 暗号ワード一覧が返されます.
     * <BR>
     * @return String[] 暗号ワード一覧が返されます.
     */
    public String[] getWords() {
        
        String[] ret = null ;
        
        if( this.isOpen() == false ) {
            return null ;
        }
        
        try {
            synchronized( sync.get() ) {
                Cb32Table tbl = factory.getTableFactory().getCb32Table() ;
                if( tbl != null ) {
                    ret = tbl.getWords() ;
                }
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 暗号ワード数を取得.
     * <BR><BR>
     * 設定されている暗号ワード数が返されます.
     * <BR>
     * @return int 暗号ワード数が返されます.
     */
    public int getWordSize() {
        
        int ret = 0 ;
        
        if( this.isOpen() == false ) {
            return 0 ;
        }
        
        try {
            synchronized( sync.get() ) {
                Cb32Table tbl = factory.getTableFactory().getCb32Table() ;
                if( tbl != null ) {
                    ret = tbl.size() ;
                }
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 暗号モードを取得.
     * <BR><BR>
     * 暗号モードを取得します.
     * <BR>
     * @return boolean 暗号モードが返されます.<BR>
     *                 [true]が返された場合、暗号モードはONです.<BR>
     *                 [false]が返された場合、暗号モードはOFFです.
     */
    public boolean isCb32() {
        
        if( this.isOpen() == false ) {
            return false ;
        }
        
        try {
            synchronized( sync.get() ) {
                return factory.getTableFactory().isCb32Flag() ;
            }
        } catch( Exception e ) {
        }
        
        return false ;
        
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープンされているか返されます.<BR>
     *                 [true]が返された場合、オープン中です.<BR>
     *                 [false]が返された場合、クローズされています.
     */
    public boolean isOpen() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                if( factory == null || factory.getUdp().isOpen() == false ||
                    factory.getTableFactory().isExitThreadFlag() == true ) {
                    ret = false ;
                }
                ret = true ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * UDPオブジェクトを取得.
     */
    private BaseUdpProtocol getUdp()
        throws AccessException {
        
        if( this.isOpen() == false ) {
            throw new AccessException( "バインドされていません" ) ;
        }
        
        return factory.getUdp() ;
    }
    
}

