/*
 * @(#)CB32Table.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import org.maachang.commons.conv.CodeBase32;
import org.maachang.commons.conv.SHA1;
import org.maachang.commons.util.array.ObjectArray;

/**
 * 暗号管理テーブル.
 *
 * @version 2007/01/13
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class Cb32Table {
    
    /**
     * デフォルトワード.
     */
    public static final String DEFAULT_WORD = "%default@word-maachang-connect-!!?" ;
    
    /**
     * 暗号情報管理テーブル.
     */
    private ObjectArray table = null ;
    
    /**
     * コンストラクタ.
     */
    public Cb32Table() {
        this.create() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param words 対象のワード群を設定します.
     */
    public Cb32Table( String[] words ) {
        this.create( words ) ;
    }
    
    /**
     * 情報破棄.
     * <BR><BR>
     * 情報を破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     */
    public synchronized void create() {
        table = new ObjectArray() ;
        this.add( DEFAULT_WORD ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param words 対象のワード群を設定します.
     */
    public synchronized void create( String[] words ) {
        table = new ObjectArray() ;
        this.add( DEFAULT_WORD ) ;
        if( words != null && words.length > 0 ) {
            int len = words.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( words[ i ] != null && words[ i ].length() > 0 ) {
                    this.add( words[ i ] ) ;
                }
            }
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        if( table != null ) {
            table.clear() ;
        }
        table = null ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 新しい暗号ワード情報を追加します.
     * <BR>
     * @param word 対象のワード情報を追加します.
     */
    public synchronized void add( String word ) {
        
        if( word == null || ( word = word.trim() ).length() <= 0 ) {
            return ;
        }
        
        if( this.searchWord( word ) == -1 ) {
            Cb32PrivateValue val = new Cb32PrivateValue( word ) ;
            table.add( val ) ;
        }
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 追加されている暗号ワードを削除します.
     * <BR>
     * @param word 削除対象のワード情報を設定します.
     */
    public synchronized void remove( String word ) {
        
        if( word == null || ( word = word.trim() ).length() <= 0 ) {
            return ;
        }
        
        int no = -1 ;
        
        if( DEFAULT_WORD.equals( word ) == false &&
            ( no = this.searchWord( word ) ) != -1 ) {
            
            table.remove( no ) ;
        }
        
    }
    
    /**
     * デフォルトSHA1を取得.
     * <BR><BR>
     * デフォルトのSHA1を取得します.
     * <BR>
     * @return byte[] デフォルトのSHA1が返されます.
     */
    public synchronized byte[] defaultSha1() {
        if( table != null && table.size() > 0 ) {
            return ( ( Cb32PrivateValue )table.get( 0 ) ).getSha1() ;
        }
        return null ;
    }
    
    /**
     * デフォルトプライベートKeyを取得.
     * <BR><BR>
     * デフォルトのプライベートKeyを取得します.
     * <BR>
     * @return byte[] デフォルトのプライベートKeyが返されます.
     */
    public synchronized byte[] defaultPrivateKey() {
        if( table != null && table.size() > 0 ) {
            return ( ( Cb32PrivateValue )table.get( 0 ) ).getPrivateKey() ;
        }
        return null ;
    }
    
    /**
     * 対象ワードからSHA1を取得.
     * <BR><BR>
     * 対象ワードからSHA1を取得します.
     * <BR>
     * @param word 対象のワード情報を設定します.
     * @return byte[] SHA1情報が返されます.
     */
    public synchronized byte[] getSha1( String word ) {
        if( word == null || word.length() <= 0 ) {
            return null ;
        }
        
        int no = this.searchWord( word ) ;
        if( no != -1 ) {
            return ( ( Cb32PrivateValue )table.get( no ) ).getSha1() ;
        }
        return null ;
    }
    
    /**
     * 対象ワードからプライベートKeyを取得.
     * <BR><BR>
     * 対象ワードからプライベートKeyを取得します.
     * <BR>
     * @param word 対象のワード情報を設定します.
     * @return byte[] プライベートKey情報が返されます.
     */
    public synchronized byte[] getPrivateKey( String word ) {
        if( word == null || word.length() <= 0 ) {
            return null ;
        }
        
        int no = this.searchWord( word ) ;
        if( no != -1 ) {
            return ( ( Cb32PrivateValue )table.get( no ) ).getPrivateKey() ;
        }
        return null ;
    }
    
    /**
     * 対象SHA1からプライベートKeyを取得.
     * <BR><BR>
     * 対象SHA1からプライベートKeyを取得します.
     * <BR>
     * @param binary 対象のSHA1情報を設定します.
     * @return byte[] プライベートKey情報が返されます.
     */
    public synchronized byte[] getPrivateKey( byte[] binary ) {
        return this.getPrivateKey( binary,0 ) ;
    }
    
    /**
     * 対象SHA1からプライベートKeyを取得.
     * <BR><BR>
     * 対象SHA1からプライベートKeyを取得します.
     * <BR>
     * @param binary 対象のSHA1情報を設定します.
     * @param offset 対象の検索開始位置を設定します.
     * @return byte[] プライベートKey情報が返されます.
     */
    public synchronized byte[] getPrivateKey( byte[] binary,int offset ) {
        if( binary == null || binary.length <= 0 || offset < 0 ) {
            return null ;
        }
        
        int no = this.searchSha1( binary,offset ) ;
        if( no != -1 ) {
            return ( ( Cb32PrivateValue )table.get( no ) ).getPrivateKey() ;
        }
        return null ;
    }
    
    /**
     * 対象SHA1から暗号ワードを取得.
     * <BR><BR>
     * 対象SHA1から暗号ワードを取得します.
     * <BR>
     * @param binary 対象のSHA1情報を設定します.
     * @return String 暗号ワード情報が返されます.
     */
    public synchronized String getCb32Word( byte[] binary ) {
        return this.getCb32Word( binary,0 ) ;
    }
    
    /**
     * 対象SHA1から暗号ワードを取得.
     * <BR><BR>
     * 対象SHA1から暗号ワードを取得します.
     * <BR>
     * @param binary 対象のSHA1情報を設定します.
     * @param offset 対象の検索開始位置を設定します.
     * @return String 暗号ワード情報が返されます.
     */
    public synchronized String getCb32Word( byte[] binary,int offset ) {
        if( binary == null || binary.length <= 0 || offset < 0 ) {
            return null ;
        }
        
        int no = this.searchSha1( binary,offset ) ;
        if( no != -1 ) {
            return ( ( Cb32PrivateValue )table.get( no ) ).getCb32String() ;
        }
        return null ;
    }
    
    /**
     * 格納ワード一覧を取得.
     * <BR><BR>
     * 格納ワード一覧が返されます.
     * <BR>
     * @retrn String[] 格納ワード一覧が返されます.
     */
    public synchronized String[] getWords() {
        if( table != null ) {
            int len = table.size() ;
            if( len > 0 ) {
                String[] ret = new String[ len ] ;
                Cb32PrivateValue val = null ;
                for( int i = 0 ; i < len ; i ++ ) {
                    val = ( Cb32PrivateValue )table.get( i ) ;
                    ret[ i ] = val.getCb32String() ;
                }
                
                return ret ;
            }
        }
        
        return null ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR><BR>
     * 格納されている情報数が返されます.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public synchronized int size() {
        if( table != null ) {
            return table.size() ;
        }
        return 0 ;
    }
    
    /**
     * 利用ワードが存在するかチェック.
     * <BR><BR>
     * 利用ワードが存在するかチェックします.
     * <BR>
     * @param word チェック対象のワード情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象ワードの情報は存在します.<BR>
     *                 [false]が返された場合、対象ワードの情報は存在しません.
     */
    public synchronized boolean isWord( String word ) {
        return ( this.searchWord( word ) == -1 ) ? false : true ;
    }
    
    /**
     * ワード存在位置を取得.
     */
    private int searchWord( String word ) {
        if( word != null && word.length() > 0 && table != null ) {
            int len = table.size() ;
            Cb32PrivateValue val = null ;
            for( int i = 0 ; i < len ; i ++ ) {
                val = ( Cb32PrivateValue )table.get( i ) ;
                if( word.equals( val.getCb32String() ) == true ) {
                    return i ;
                }
            }
        }
        
        return -1 ;
    }
    
    /**
     * SHA1存在位置を取得.
     */
    private int searchSha1( byte[] binary,int offset ) {
        if( binary != null && table != null ) {
            boolean flag = true ;
            int len = table.size() ;
            Cb32PrivateValue val = null ;
            byte[] s1 = null ;
            for( int i = 0 ; i < len ; i ++ ) {
                val = ( Cb32PrivateValue )table.get( i ) ;
                s1 = val.getSha1() ;
                flag = true ;
                for( int j = 0,k = offset ; j < 16 ; j ++,k ++ ) {
                    if( binary[ k ] != s1[ j ] ) {
                        flag = false ;
                        break ;
                    }
                }
                if( flag == true ) {
                    return i ;
                }
            }
        }
        
        return -1 ;
    }
}

/**
 * 内部キー要素.
 */
class Cb32PrivateValue {
    
    /**
     * 暗号ワード.
     */
    private String cb32String = null ;
    
    /**
     * SHA1コード.
     */
    private byte[] sha1 = null ;
    
    /**
     * 暗号コード.
     */
    private byte[] privateKey = null ;
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ワード情報を設定して、オブジェクトを生成します.
     * <BR>
     * @param word 対象のワード情報を設定します.
     */
    public Cb32PrivateValue( String word ) {
        try {
            cb32String = word ;
            privateKey = CodeBase32.convertStringByCode32Key( word ) ;
            //sha1 = SHA1.convertBinary( word.getBytes( BaseDef.UTF8 ) ) ;
            sha1 = SHA1.convertBinary( privateKey ) ;
        } catch( Exception e ) {
            e.printStackTrace() ;
        }
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        cb32String = null ;
        privateKey = null ;
        sha1 = null ;
    }

    /**
     * cb32String を取得.
     * <BR><BR>
     * @return cb32String が返されます.
     */
    public String getCb32String() {
        return cb32String;
    }

    /**
     * cb32String を設定.
     * <BR><BR>
     * @param cb32String cb32String を設定します.
     */
    public void setCb32String(String cb32String) {
        this.cb32String = cb32String;
    }

    /**
     * privateKey を取得.
     * <BR><BR>
     * @return privateKey が返されます.
     */
    public byte[] getPrivateKey() {
        return privateKey;
    }

    /**
     * privateKey を設定.
     * <BR><BR>
     * @param privateKey privateKey を設定します.
     */
    public void setPrivateKey(byte[] privateKey) {
        this.privateKey = privateKey;
    }

    /**
     * sha1 を取得.
     * <BR><BR>
     * @return sha1 が返されます.
     */
    public byte[] getSha1() {
        return sha1;
    }

    /**
     * sha1 を設定.
     * <BR><BR>
     * @param sha1 sha1 を設定します.
     */
    public void setSha1(byte[] sha1) {
        this.sha1 = sha1;
    }
    
}

