/*
 * @(#)ConnectBean.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import java.net.InetAddress;

import org.maachang.connect.thread.ProtocolDef;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.util.ConvertParam;

/**
 * 1つのコネクションを表すBean.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ConnectBean implements BaseBean {
    
    /**
     * コネクションID.
     */
    private long id = -1L ;
    
    /**
     * 生成時間.
     */
    private long createTime = -1L ;
    
    /**
     * 前回コネクション送信時間.
     */
    private long beforeSendTime = -1L ;
    
    /**
     * コネクション先.
     */
    private InetAddress address = null ;
    
    /**
     * ポート番号.
     */
    private int port = -1 ;
    
    /**
     * 暗号ワード.
     */
    private String cb32Word = null ;
    
    /**
     * 送信データ電文.
     */
    private BinResource sendData = null ;
    
    /**
     * 送信開始フラグ
     */
    private boolean startSendFlag = false ;
    
    /**
     * 送信終了フラグ.
     */
    private boolean endSendFlag = false ;
    
    /**
     * 全体パケットサイズ.
     */
    private int allPacketLength = -1 ;
    
    /**
     * コンストラクタ.
     */
    private ConnectBean() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * コネクションBeanを生成します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @param address 送信先コネクションアドレスを設定します.
     * @param port 送信先ポート番号を設定します.
     * @param cb32Word 暗号ワードを設定します.
     * @param sendData 送信対象のデータを設定します.
     * @exception InputException 入力例外.
     */
    public ConnectBean( long id,InetAddress address,int port,String cb32Word,BinResource sendData )
        throws InputException {
        if( address == null || sendData == null || sendData.size() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        int size = sendData.size() ;
        
        this.id = id ;
        this.createTime = System.currentTimeMillis() ;
        this.beforeSendTime = -1L ;
        this.address = address ;
        this.port = port ;
        this.cb32Word = cb32Word ;
        this.sendData = sendData ;
        this.startSendFlag = false ;
        this.endSendFlag = false ;
        this.allPacketLength = ( size / ProtocolDef.SEND_PACKET_DATA_LENGTH ) +
            ( ( ( size % ProtocolDef.SEND_PACKET_DATA_LENGTH ) != 0 ) ? 1 : 0 ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     */
    protected void finalize() throws Exception {
        id = -1L ;
        createTime = -1L ;
        beforeSendTime = -1L ;
        address = null ;
        port = -1 ;
        cb32Word = null ;
        sendData = null ;
        startSendFlag = false ;
        endSendFlag = false ;
        allPacketLength = -1 ;
    }
    
    /**
     * 指定パケット項番に対するパケットデータを生成.
     * <BR><BR>
     * 指定パケット項番に対するパケットデータを生成します.
     * <BR>
     * @param tableFactory 対象のテーブルファクトリを設定します.
     * @param no 対象のパケット項番を設定します.
     * @return byte[] 対象のパケット情報が返されます.
     */
    public synchronized byte[] getPacketData(
        MaachangConnectTableFactory tableFactory,int no ) {
        
        if( no < 0 || no >= allPacketLength ) {
            return null ;
        }
        
        int point = no * ProtocolDef.SEND_PACKET_DATA_LENGTH ;
        int packetLength = sendData.size() - point ;
        
        if( packetLength >= ProtocolDef.SEND_PACKET_DATA_LENGTH ) {
            packetLength = ProtocolDef.SEND_PACKET_DATA_LENGTH ;
        }
        
        int pnt = 0 ;
        int off = 0 ;
        byte[] ret = null ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            ret = ProtocolDef.getCb32Binary(
                packetLength + ProtocolDef.SEND_PACKET_HEADER_LENGTH ) ;
            pnt = ProtocolDef.CB32_LENGTH_PLUS ;
            off = ProtocolDef.CB32_LENGTH_PLUS ;
        }
        // 非暗号条件の場合.
        else {
            ret = new byte[
                packetLength + ProtocolDef.SEND_PACKET_HEADER_LENGTH ] ;
            pnt = 0 ;
        }
        
        
        // プロトコルヘッダ情報を設定.
        ConvertParam.convertShort(
            ret,pnt,( short )( ProtocolDef.HEADER_SEND_PACKET & 0x0000ffff ) ) ;
        pnt += 2 ;
        
        // コネクションIDを設定.
        ConvertParam.convertLong( ret,pnt,id ) ;
        pnt += 8 ;
        
        // パケットデータ長.
        ConvertParam.convertShort( ret,pnt,( short )( packetLength & 0x0000ffff ) ) ;
        pnt += 2+2 ;
        
        // パケット項番.
        ConvertParam.convertInt( ret,pnt,no ) ;
        pnt += 4 ;
        
        // パケットデータを設定.
        sendData.getBinary( ret,point,pnt,packetLength ) ;
        
        // チェックコードを取得.
        int check = ProtocolDef.checkCode( ret,off ) ;
        pnt = off + 2 + 8 + 2 ;
        ConvertParam.convertShort( ret,pnt,( short )( check & 0x0000ffff ) ) ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            // 暗号処理.
            ProtocolDef.encryption( tableFactory.getCb32Table(),cb32Word,ret ) ;
        }
        
        // 更新時間を更新.
        this.createTime = System.currentTimeMillis() ;
        
        return ret ;
    }
    
    /**
     * 生成時間を更新.
     * <BR><BR>
     * 生成時間を更新します.
     */
    public synchronized void updateTime() {
        createTime = System.currentTimeMillis() ;
    }
    
    /**
     * 前回送信時間を更新.
     * <BR><BR>
     * 前回更新時間を更新します.
     */
    public synchronized void updateBeforeSendTime() {
        beforeSendTime = System.currentTimeMillis() ;
    }

    /**
     * createTime を取得.
     * <BR><BR>
     * @return createTime が返されます.
     */
    public synchronized long getCreateTime() {
        return createTime;
    }

    /**
     * beforeSendTime を取得.
     * <BR><BR>
     * @return beforeSendTime が返されます.
     */
    public long getBeforeSendTime() {
        return beforeSendTime;
    }

    /**
     * id を取得.
     * <BR><BR>
     * @return id が返されます.
     */
    public synchronized long getId() {
        return id;
    }

    /**
     * address を取得.
     * <BR><BR>
     * @return address が返されます.
     */
    public InetAddress getAddress() {
        return address;
    }

    /**
     * port を取得.
     * <BR><BR>
     * @return port が返されます.
     */
    public int getPort() {
        return port;
    }

    /**
     * cb32Word を取得.
     * <BR><BR>
     * @return cb32Word が返されます.
     */
    public String getCb32Word() {
        return cb32Word;
    }

    /**
     * sendData を取得.
     * <BR><BR>
     * @return sendData が返されます.
     */
    public synchronized BinResource getSendData() {
        return sendData;
    }

    /**
     * startSendFlag を取得.
     * <BR><BR>
     * @return startSendFlag が返されます.
     */
    public synchronized boolean isStartSendFlag() {
        return startSendFlag;
    }

    /**
     * startSendFlag を設定.
     * <BR><BR>
     * @param startSendFlag startSendFlag を設定します.
     */
    public synchronized void setStartSendFlag(boolean startSendFlag) {
        this.startSendFlag = startSendFlag;
    }

    /**
     * endSendFlag を取得.
     * <BR><BR>
     * @return endSendFlag が返されます.
     */
    public boolean isEndSendFlag() {
        return endSendFlag;
    }

    /**
     * endSendFlag を設定.
     * <BR><BR>
     * @param endSendFlag endSendFlag を設定します.
     */
    public void setEndSendFlag(boolean endSendFlag) {
        this.endSendFlag = endSendFlag;
    }

    /**
     * allPacketLength を取得.
     * <BR><BR>
     * @return allPacketLength が返されます.
     */
    public int getAllPacketLength() {
        return allPacketLength;
    }
    
}

