/*
 * @(#)ConnectTable.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.util.BigTable;
import org.maachang.commons.util.SequenceIDEx;
import org.maachang.commons.util.array.ObjectArray;

/**
 * コネクション管理テーブル.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ConnectTable extends BaseTable {
    
    /**
     * デフォルトコネクションタイムアウト.
     * 3.5sec.
     */
    private static final long DEF_TIMEOUT = 2500L ;
    
    /**
     * 最小コネクションタイムアウト.
     * 1.5sec.
     */
    private static final long MIN_TIMEOUT = 1500L ;
    
    /**
     * 最大コネクションタイムアウト.
     * 60sec.
     */
    private static final long MAX_TIMEOUT = 60000L ;
    
    /**
     * デフォルトコネクション送信パケットタイムアウト.
     * 500msec.
     */
    private static final long DEF_SEND_PACKET_TIMEOUT = 500L ;
    
    /**
     * 最小コネクション送信パケットタイムアウト.
     * 250msec.
     */
    private static final long MIN_SEND_PACKET_TIMEOUT = 250L ;
    
    /**
     * 最大コネクション送信パケットタイムアウト.
     * 5sec.
     */
    private static final long MAX_SEND_PACKET_TIMEOUT = 5000L ;
    
    /**
     * インデックステーブル.
     */
    private BigTable indexTable = null ;
    
    /**
     * シーケンスID発行.
     */
    private SequenceIDEx sequence = null ;
    
    /**
     * コネクションタイムアウト値.
     */
    private long timeout = -1L ;
    
    /**
     * コネクションパケット送信時間.
     */
    private long sendPacketTime = -1L ;
    
    /**
     * コンストラクタ.
     */
    public ConnectTable() {
        this( DEF_TIMEOUT,DEF_SEND_PACKET_TIMEOUT ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト値を設定して、コネクションテーブルを生成します.
     * <BR>
     * @param timeout コネクションタイムアウト値を設定します.<BR>
     *                 設定可能な最小値は[1500]ミリ秒です.<BR>
     *                 設定可能な最大値は[60000]ミリ秒です.
     * @param sendPacketTime 送信パケットタイムアウト値を設定します.<BR>
     *                        設定可能な最小値は[250]ミリ秒です.<BR>
     *                        設定可能な最大値は[5000]ミリ秒です.
     */
    public ConnectTable( long timeout,long sendPacketTime ) {
        
        if( timeout == -1L ) {
            timeout = DEF_TIMEOUT ;
        }
        if( timeout <= MIN_TIMEOUT ) {
            timeout = MIN_TIMEOUT ;
        }
        else if( timeout >= MAX_TIMEOUT ) {
            timeout = MAX_TIMEOUT ;
        }
        
        if( sendPacketTime == -1L ) {
            sendPacketTime = DEF_SEND_PACKET_TIMEOUT ;
        }
        if( sendPacketTime <= MIN_SEND_PACKET_TIMEOUT ) {
            sendPacketTime = MIN_SEND_PACKET_TIMEOUT ;
        }
        else if( sendPacketTime >= MAX_SEND_PACKET_TIMEOUT ) {
            sendPacketTime = MAX_SEND_PACKET_TIMEOUT ;
        }
        
        this.table = new ObjectArray() ;
        this.indexTable = new BigTable() ;
        this.sequence = new SequenceIDEx() ;
        this.timeout = timeout ;
        this.sendPacketTime = sendPacketTime ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理です.
     */
    protected void finalize() throws Exception {
        this.table = null ;
        this.indexTable = null ;
        this.sequence = null ;
        this.timeout = -1L ;
        this.sendPacketTime = -1L ;
    }
    
    /**
     * コネクションを開始.
     * <BR><BR>
     * コネクションを開始します.
     * <BR>
     * @param address 送信先コネクションアドレスを設定します.
     * @param port 送信先ポート番号を設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param sendData 送信対象のデータを設定します.
     * @return long 開始されたコネクションIDが返されます.
     * @exception InputException 入力例外.
     */
    public synchronized long startConnect(
        InetAddress address,int port,String cb32Word,BinResource sendData )
        throws InputException {
        
        long id = sequence.getID() ;
        ConnectBean bean = new ConnectBean( id,address,port,cb32Word,sendData ) ;
        table.add( bean ) ;
        indexTable.add( id,bean ) ;
        return id ;
        
    }
    
    /**
     * コネクションを終了.
     * <BR><BR>
     * コネクションを終了します.
     * <BR>
     * @param id 終了対象のコネクションIDを設定します.
     * @return ConnectBean 削除されたコネクションBeanが返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean exitConnect( long id ) {
        int num = this.getNumber( id ) ;
        if( num != -1 ) {
            ConnectBean bean = ( ConnectBean )table.remove( num ) ;
            if( bean != null ) {
                indexTable.remove( id ) ;
                return bean ;
            }
        }
        return null ;
    }
    
    /**
     * 指定条件がタイムアウトしている場合、コネクションを終了.
     * <BR><BR>
     * 指定条件がタイムアウトしている場合、コネクションを終了します.
     * <BR>
     * @param num 項番を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、タイムアウトしています.
     *                  [false]が返された場合、タイムアウトしていません.
     */
    public synchronized boolean exitConnectByTimeout( int num ) {
        if( num < 0 || table.size() <= num ) {
            return false ;
        }
        
        ConnectBean bean = ( ConnectBean )table.get( num ) ;
        if( bean != null ) {
            if( bean.getCreateTime() + timeout <= System.currentTimeMillis() ) {
                table.remove( num ) ;
                indexTable.remove( bean.getId() ) ;
                return true ;
            }
        }
        
        return false ;
    }
    
    /**
     * コネクションタイムアウト値を取得.
     * <BR><BR>
     * 指定されているコネクションタイムアウト値を取得します.
     * <BR>
     * @return long コネクションタイムアウト値が返されます.
     */
    public synchronized long getTimeout() {
        return timeout ;
    }
    
    /**
     * コネクション送信パケットタイムアウト値を取得.
     * <BR><BR>
     * 指定されているコネクション送信パケットタイムアウト値を取得します.
     * <BR>
     * @return long コネクション送信パケットタイムアウト値が返されます.
     */
    public synchronized long getSendPacketTimeout() {
        return sendPacketTime ;
    }
    
    /**
     * コネクションIDを取得.
     * <BR><BR>
     * 項番を設定して、コネクションIDを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return long コネクションIDが返されます.<BR>
     *               [-1L]が返された場合、情報は存在しません.
     */
    public synchronized long getConnectId( int num ) {
        if( num < 0 || table.size() <= num ) {
            return -1L ;
        }
        
        ConnectBean bean = ( ConnectBean )table.get( num ) ;
        if( bean != null ) {
            return bean.getId() ;
        }
        
        return -1L ;
    }
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * コネクションIDを設定して、コネクションBeanを取得します.
     * <BR>
     * @param id コネクションIDを設定します.
     * @return ConnectBean コネクションBeanが返されます.<BR>
     *                      [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean getConnectBean( long id ) {
        return ( ConnectBean )indexTable.get( id ) ;
        //return this.getConnectBean( this.getNumber( id ) ) ;
    }
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * 項番を設定して、コネクションBeanを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return ConnectBean コネクションBeanが返されます.<BR>
     *                      [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean getConnectBean( int num ) {
        if( num < 0 || table.size() <= num ) {
            return null ;
        }
        
        return ( ConnectBean )table.get( num ) ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 情報数が返されます.
     */
    public synchronized int size() {
        return table.size() ;
    }
    
    /**
     * 格納コネクションID群を取得.
     * <BR><BR>
     * 格納コネクションID群を取得します.
     * <BR>
     * @return long[] 格納されているコネクションID群が返されます.
     */
    public synchronized long[] getConnectIds() {
        return indexTable.getNumbers() ;
    }
    
    /**
     * 対象コネクションIDが存在するかチェック.
     * <BR><BR>
     * 指定したコネクションIDが存在するかチェックします.
     * <BR>
     * @param id 対象のコネクションIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、情報は存在します.<BR>
     *                  [false]が返された場合、情報は存在しません.
     */
    public synchronized boolean isConnectId( long id ) {
        return indexTable.isData( id ) ;
        //int num = this.getNumber( id ) ;
        //if( num != -1 ) {
        //    return true ;
        //}
        //return false ;
    }
    
}

