/*
 * @(#)MaachangConnectBaseExecution.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread ;

import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.thread.execution.ExecutionAnalysis;
import org.maachang.connect.thread.execution.ExecutionConnect;
import org.maachang.connect.thread.execution.ExecutionReceive;
import org.maachang.connect.thread.execution.ExecutionSend;
import org.maachang.connect.thread.execution.ExecutionUnderRetry;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.BaseUdpProtocol;

/**
 * MaachangConnectプロトコルスレッドファクトリ.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class MaachangConnectThreadFactory {
    
    /**
     * 最小スレッド数.
     */
    private static final int MIN_THREAD = 1 ;
    
    /**
     * 最大スレッド数.
     */
    private static final int MAX_THREAD = 4 ;
    
    /**
     * 電文解析スレッドサイズデフォルト値.
     */
    private static final int DEF_ANALYSIS_THREAD = 2 ;
    
    /**
     * 受信再送待ちスレッドサイズデフォルト値.
     */
    private static final int DEF_UNDER_RETRY_THREAD = 2 ;
    
    /**
     * 送信スレッドサイズデフォルト値.
     */
    private static final int DEF_SEND_THREAD = 2 ;
    
    /**
     * 受信スレッドサイズデフォルト値.
     */
    private static final int DEF_RECEIVE_THREAD = 3 ;
    
    
    
    /**
     * UDPプロトコル.
     */
    private BaseUdpProtocol udp = null ;
    
    /**
     * MaachangConnectテーブルファクトリ.
     */
    private MaachangConnectTableFactory tableFactory = null ;
    
    /**
     * タイムアウト監視スレッド.
     */
    private TableTimeoutThread timeoutThread = null ;
    
    /**
     * コネクションスレッド.
     */
    private MaachangConnectBaseThread executionConnect = null ;
    
    /**
     * 電文解析スレッド.
     */
    private MaachangConnectBaseThread[] executionAnalysys = null ;
    
    /**
     * パケット受信待ちスレッド.
     */
    private MaachangConnectBaseThread[] executionUnderRetry = null ;
    
    /**
     * 電文送信スレッド.
     */
    private MaachangConnectBaseThread[] executionSend = null ;
    
    /**
     * 電文受信スレッド.
     */
    private MaachangConnectBaseThread[] executionReceive = null ;
    
    /**
     * MaachangConnect同期.
     */
    private MaachangConnectSync maachangSync = new MaachangConnectSync() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private MaachangConnectThreadFactory() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param udp 対象のUDPプロトコルを設定します.
     * @param tableFactory 対象のテーブルファクトリを設定します.
     * @exception InputException 入力例外.
     */
    public MaachangConnectThreadFactory( BaseUdpProtocol udp,MaachangConnectTableFactory tableFactory )
        throws InputException {
        
        this( udp,tableFactory,
            DEF_ANALYSIS_THREAD,DEF_UNDER_RETRY_THREAD,
            DEF_SEND_THREAD,DEF_RECEIVE_THREAD ) ;
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param udp 対象のUDPプロトコルを設定します.
     * @param tableFactory 対象のテーブルファクトリを設定します.
     * @param executionAnalysisLength 電文解析スレッド数を設定します.
     * @param executionUnderRetryLength 電文パケット受信待ちスレッド数を設定します.
     * @param executionSendLength 電文送信スレッド数を設定します.
     * @param executionReceiveLength 電文受信スレッド数を設定します.
     * @exception InputException 入力例外.
     */
    public MaachangConnectThreadFactory( BaseUdpProtocol udp,MaachangConnectTableFactory tableFactory,
        int executionAnalysisLength,int executionUnderRetryLength,int executionSendLength,
        int executionReceiveLength ) throws InputException {
        
        int i ;
        int len ;
        
        if( udp == null || udp.isOpen() == false || tableFactory == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( executionAnalysisLength == -1 ) {
            executionAnalysisLength = DEF_ANALYSIS_THREAD ;
        }
        else if( executionAnalysisLength <= MIN_THREAD ) {
            executionAnalysisLength = MIN_THREAD ;
        }
        else if( executionAnalysisLength >= MAX_THREAD ) {
            executionAnalysisLength = MAX_THREAD ;
        }
        
        if( executionUnderRetryLength == -1 ) {
            executionUnderRetryLength = DEF_UNDER_RETRY_THREAD ;
        }
        else if( executionUnderRetryLength <= MIN_THREAD ) {
            executionUnderRetryLength = MIN_THREAD ;
        }
        else if( executionUnderRetryLength >= MAX_THREAD ) {
            executionUnderRetryLength = MAX_THREAD ;
        }
        
        if( executionSendLength == -1 ) {
            executionSendLength = DEF_SEND_THREAD ;
        }
        else if( executionSendLength <= MIN_THREAD ) {
            executionSendLength = MIN_THREAD ;
        }
        else if( executionSendLength >= MAX_THREAD ) {
            executionSendLength = MAX_THREAD ;
        }
        
        if( executionReceiveLength == -1 ) {
            executionReceiveLength = DEF_RECEIVE_THREAD ;
        }
        else if( executionReceiveLength <= MIN_THREAD ) {
            executionReceiveLength = MIN_THREAD ;
        }
        else if( executionReceiveLength >= MAX_THREAD ) {
            executionReceiveLength = MAX_THREAD ;
        }
        
        this.udp = udp ;
        this.tableFactory = tableFactory ;
        this.maachangSync.create() ;
        
        // タイムアウト監視スレッドを生成.
        this.timeoutThread = new TableTimeoutThread( tableFactory,maachangSync ) ;
        
        // コネクション実行スレッドを生成.
        ExecutionConnect execConn = new ExecutionConnect() ;
        this.executionConnect = new MaachangConnectBaseThread(
            0,execConn,udp,tableFactory,maachangSync ) ;
        
        // 電文解析実行スレッドを生成.
        ExecutionAnalysis execAna = new ExecutionAnalysis() ;
        len = executionAnalysisLength ;
        this.executionAnalysys = new MaachangConnectBaseThread[ len ] ;
        for( i = 0 ; i < len ; i ++ ) {
            this.executionAnalysys[ i ] = new MaachangConnectBaseThread(
                i,execAna,udp,tableFactory,maachangSync ) ;
        }
        
        // パケット受信待ち実行スレッドを生成.
        ExecutionUnderRetry execUnder = new ExecutionUnderRetry() ;
        len = executionUnderRetryLength ;
        this.executionUnderRetry = new MaachangConnectBaseThread[ len ] ;
        for( i = 0 ; i < len ; i ++ ) {
            this.executionUnderRetry[ i ] = new MaachangConnectBaseThread(
                i,execUnder,udp,tableFactory,maachangSync ) ;
        }
        
        // 電文送信実行スレッドを生成.
        ExecutionSend execSend = new ExecutionSend() ;
        len = executionSendLength ;
        this.executionSend = new MaachangConnectBaseThread[ len ] ;
        for( i = 0 ; i < len ; i ++ ) {
            this.executionSend[ i ] = new MaachangConnectBaseThread(
                i,execSend,udp,tableFactory,maachangSync ) ;
        }
        
        // 電文受信実行スレッドを生成.
        ExecutionReceive execRecv = new ExecutionReceive() ;
        len = executionReceiveLength ;
        this.executionReceive = new MaachangConnectBaseThread[ len ] ;
        for( i = 0 ; i < len ; i ++ ) {
            this.executionReceive[ i ] = new MaachangConnectBaseThread(
                i,execRecv,udp,tableFactory,maachangSync ) ;
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理です.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        if( timeoutThread != null ) {
            timeoutThread.destroy() ;
            timeoutThread = null ;
        }
        
        if( executionConnect != null ) {
            executionConnect.destroy() ;
            executionConnect = null ;
        }
        
        int i ;
        int len ;
        
        if( executionAnalysys != null ) {
            len = executionAnalysys.length ;
            for( i = 0 ; i < len ; i ++ ) {
                executionAnalysys[ i ].destroy() ;
            }
            executionAnalysys = null ;
        }
        
        if( executionUnderRetry != null ) {
            len = executionUnderRetry.length ;
            for( i = 0 ; i < len ; i ++ ) {
                executionUnderRetry[ i ].destroy() ;
            }
            executionUnderRetry = null ;
        }
        
        if( executionSend != null ) {
            len = executionSend.length ;
            for( i = 0 ; i < len ; i ++ ) {
                executionSend[ i ].destroy() ;
            }
            executionSend = null ;
        }
        
        if( executionReceive != null ) {
            len = executionReceive.length ;
            for( i = 0 ; i < len ; i ++ ) {
                executionReceive[ i ].destroy() ;
            }
            executionReceive = null ;
        }
        
        if( udp != null ) {
            udp.close() ;
            udp = null ;
        }
        
        tableFactory = null ;
        maachangSync.clear() ;
        
    }

    /**
     * maachangSync を取得.
     * <BR><BR>
     * @return maachangSync が返されます.
     */
    public MaachangConnectSync getMaachangSync() {
        return maachangSync;
    }

    /**
     * tableFactory を取得.
     * <BR><BR>
     * @return tableFactory が返されます.
     */
    public MaachangConnectTableFactory getTableFactory() {
        return tableFactory;
    }

    /**
     * udp を取得.
     * <BR><BR>
     * @return udp が返されます.
     */
    public BaseUdpProtocol getUdp() {
        return udp;
    }

}

