/*
 * @(#)ExecutionAnalysis.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread.execution ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.connect.table.ConnectBean;
import org.maachang.connect.table.ConnectTable;
import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.table.SendReceiveBean;
import org.maachang.connect.table.SendTable;
import org.maachang.connect.table.UnderReceiveBean;
import org.maachang.connect.table.UnderReceiveTable;
import org.maachang.connect.thread.MaachangConnectBaseExecution;
import org.maachang.connect.thread.ProtocolDef;

import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.util.ConvertParam;
import org.maachang.commons.util.UtilCom;

/**
 * 受信電文解析処理.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ExecutionAnalysis implements MaachangConnectBaseExecution {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionAnalysis.class ) ;
    
    /**
     * コネクション送信結果電文送信カウント.
     */
    private static final int SEND_CONNECT_RETURN_COUNT = 2 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ExecutionAnalysis() {
        
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * MaachangConnectプロトコル実行処理.
     * <BR>
     * @param threadNum スレッド項番を設定されます.
     * @param udp 対象のUDPプロトコルを設定されます.
     * @param tableFactory テーブルFactoryを設定されます.
     * @param sync 同期オブジェクトが設定されます.
     * @param connectAddress 受信処理などに利用するコネクションアドレス.
     * @param cb32Word 暗号解析結果の暗号ワードを格納する情報が設定されます.
     * @exception Exception 処理例外.
     */
    public void execution( int threadNum,BaseUdpProtocol udp,
        MaachangConnectTableFactory tableFactory,MaachangConnectSync sync,
        ConnectAddress connectAddress,String[] cb32Word ) throws Exception {
        
        try {
            
            // 受信データを取得.
            SendReceiveBean bean = tableFactory.getSendReceiveTable().getReceive() ;
            
            // 受信電文が存在しない場合.
            if( bean == null || bean.getTelegram() == null ||
                bean.getTelegram().length <= 0 ||
                bean.getAddress() == null ) {
                // 処理しない.
                UtilCom.idleTime() ;
                return ;
            }
            
            // 対象電文が暗号電文の場合.
            int type = ProtocolDef.getProtocolType( bean.getTelegram() ) ;
            if( type == ProtocolDef.HEADER_CB32 ) {
                
                // 暗号を解析.
                bean.setTelegram(
                    ProtocolDef.analysis(
                        cb32Word,tableFactory.getCb32Table(),
                        bean.getTelegram() ) ) ;
                
                // 処理解析結果、失敗した場合.
                if( bean.getTelegram() == null ) {
                    LOG.warn( "暗号受信解析に失敗(IP:" +
                        bean.getAddress().getHostName() +
                        " port:" + bean.getPort() + ")" ) ;
                    return ;
                }
                
                // タイプをもう一度取得.
                type = ProtocolDef.getProtocolType( bean.getTelegram() ) ;
                
            }
            
            //System.out.println( "受信電文タイプ:" + Integer.toHexString( type ) ) ;
            
            // 電文ヘッダから、処理解析.
            switch( type ) {
                case ProtocolDef.HEADER_SEND_CONNECT :  // コネクション送信.
                    this.receiveSendConnect( tableFactory,bean,sync,cb32Word[ 0 ] ) ;
                    break ;
                case ProtocolDef.HEADER_RECV_CONNECT :  // コネクション受信.
                    this.receiveRecvConnect( tableFactory,bean,sync,cb32Word[ 0 ] ) ;
                    break ;
                case ProtocolDef.HEADER_SEND_PACKET :   // パケット送信.
                    this.receiveSendPacket( tableFactory,bean,sync,cb32Word[ 0 ] ) ;
                    break ;
                case ProtocolDef.HEADER_RETRY_PACKET :  // パケット再送依頼.
                    this.receiveResendPacket( tableFactory,bean,sync,cb32Word[ 0 ] ) ;
                    break ;
                case ProtocolDef.HEADER_EXIT_PACKET :   // パケット完了通知.
                    this.receiveExitPacket( tableFactory,bean,sync,cb32Word[ 0 ] ) ;
                    break ;
            }
            
        } catch( Exception e ) {
            LOG.warn( "パケット電文解析例外",e ) ;
            UtilCom.idleTime() ;
        }
        
    }
    
    /**
     * 受信処理 : コネクション送信処理.
     */
    private void receiveSendConnect(
        MaachangConnectTableFactory tableFactory,SendReceiveBean bean,
        MaachangConnectSync sync,String cb32Word ) {
        
        LOG.debug( "[ExecutionAnalysis](受信処理)コネクション送信処理" ) ;
        
        //////////////////////////////////////////////////////////////////
        // 受信待ちテーブルに対して、受信コネクションIDが存在しない場合は、
        // 新規で受信待ちテーブルに、受信枠を設定し、コネクション応答電文を
        // 受信先に送信.
        //////////////////////////////////////////////////////////////////
        
        byte[] bin = bean.getTelegram() ;
        if( bin == null || bin.length != ProtocolDef.SEND_CONNECT_LENGTH ) {
            return ;
        }
        
        int pnt = 2 ;
        // コネクションIDを取得.
        long connectId = ConvertParam.convertLong( pnt,bin ) ;
        pnt += 8 ;
        
        // 電文長を取得.
        int dataLength = ConvertParam.convertInt( pnt,bin ) ;
        pnt += 4 ;
        
        // 取得条件が不正な場合.
        if( connectId <= -1L || dataLength <= 0 ) {
            // 何もしない.
            return ;
        }
        
        UnderReceiveTable table = tableFactory.getUnderReceiveTable() ;
        
        synchronized( sync.getUnderReceiveSync().get() ) {
            
            // 対象IDの条件が既に受信待ちテーブルに存在する場合.
            if( table.isConnectId( connectId ) == true ) {
                // 何もしない.
                return ;
            }
            
            // テーブルに受信待ちテーブルを追加.
            table.createUnderReceive(
                connectId,bean.getAddress(),bean.getPort(),
                cb32Word,dataLength ) ;
            
        }
        
        //////////////////////////////////////
        // コネクション送信に対して結果を送信.
        //////////////////////////////////////
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            bin = ProtocolDef.getCb32Binary( ProtocolDef.RECV_CONNECT_LENGTH ) ;
            pnt = ProtocolDef.CB32_LENGTH_PLUS ;
        }
        // 非暗号条件の場合.
        else {
            bin = new byte[ ProtocolDef.RECV_CONNECT_LENGTH ] ;
            pnt = 0 ;
        }
        
        // コネクション送信ヘッダ.
        ConvertParam.convertShort(
            bin,pnt,( short )( ProtocolDef.HEADER_RECV_CONNECT & 0x0000ffff ) ) ;
        pnt += 2 ;
        
        // コネクションID.
        ConvertParam.convertLong( bin,pnt,connectId ) ;
        pnt += 8 ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            // 暗号処理.
            ProtocolDef.encryption( tableFactory.getCb32Table(),cb32Word,bin ) ;
        }
        
        // 送信条件としてセット.
        tableFactory.getSendReceiveTable().addSend(
            bean.getAddress(),bean.getPort(),
            bin,SEND_CONNECT_RETURN_COUNT ) ;
        
    }
    
    /**
     * 受信処理 : コネクション受信処理.
     */
    private void receiveRecvConnect(
        MaachangConnectTableFactory tableFactory,SendReceiveBean bean,
        MaachangConnectSync sync,String cb32Word ) {
        
        LOG.debug( "[ExecutionAnalysis](受信処理)コネクション受信処理" ) ;
        
        //////////////////////////////////////////////////////////////////
        // コネクションテーブルに受信されたコネクションIDが存在する場合、
        // その情報を、送信テーブルに移動し、送信電文を、パケット単位で、
        // 送信先に送信.
        //////////////////////////////////////////////////////////////////
        
        byte[] bin = bean.getTelegram() ;
        if( bin == null || bin.length != ProtocolDef.RECV_CONNECT_LENGTH ) {
            return ;
        }
        
        int pnt = 2 ;
        // コネクションIDを取得.
        long connectId = ConvertParam.convertLong( pnt,bin ) ;
        pnt += 8 ;
        
        // 取得条件が不正な場合.
        if( connectId <= -1L ) {
            // 何もしない.
            return ;
        }
        
        ConnectTable table = null ;
        ConnectBean connBean = null ;
        
        synchronized( sync.getConnectSync().get() ) {
            
            // コネクションテーブルを取得.
            table = tableFactory.getConnectTable() ;
            connBean = table.exitConnect( connectId ) ;
            
            // コネクションテーブルに存在しない場合.
            if( connBean == null ) {
                // 何もしない.
                return ;
            }
            
            synchronized( sync.getSendSync().get() ) {
                
                // 取得されたコネクションBeanを送信テーブルに移動.
                SendTable sndTable = tableFactory.getSendTable() ;
                sndTable.putSendData( connBean ) ;
                
            }
            
        }
        
        // パケット単位で情報を送信.
        int len = connBean.getAllPacketLength() ;
        for( int i = 0 ; i < len ; i ++ ) {
            bin = connBean.getPacketData( tableFactory,i ) ;
            tableFactory.getSendReceiveTable().addSend(
                bean.getAddress(),bean.getPort(),bin ) ;
        }
        
    }
    
    /**
     * 受信処理 : パケット送信処理.
     */
    private void receiveSendPacket(
        MaachangConnectTableFactory tableFactory,SendReceiveBean bean,
        MaachangConnectSync sync,String cb32Word ) {
        
        LOG.debug( "[ExecutionAnalysis](受信処理)パケット送信処理" ) ;
        
        //////////////////////////////////////////////////////////////////
        // 受信したパケットデータを、受信待ちテーブル内の指定コネションID
        // に対して、情報追加.
        //////////////////////////////////////////////////////////////////
        
        byte[] bin = bean.getTelegram() ;
        if( bin == null || bin.length <= ProtocolDef.SEND_PACKET_HEADER_LENGTH ) {
            return ;
        }
        
        int pnt = 2 ;
        // コネクションIDを取得.
        long connectId = ConvertParam.convertLong( pnt,bin ) ;
        pnt += 8 ;
        
        // パケット長を取得.
        int packetLength = ( int )( ConvertParam.convertShort( pnt,bin ) & 0x0000ffff ) ;
        pnt += 2 ;
        
        // チェックコード.
        int checkCode = ( int )( ConvertParam.convertShort( pnt,bin ) & 0x0000ffff ) ;
        pnt += 2 ;
        
        // パケット項番を取得.
        int packetNo = ConvertParam.convertInt( pnt,bin ) ;
        pnt += 4 ;
        
        // 取得条件が不正な場合.
        if( connectId <= -1L || packetLength <= 0 || packetNo < 0 ) {
            // 何もしない.
            return ;
        }
        // チェックコードとバイナリコードの条件が一致しない場合.
        else if( ProtocolDef.checkCode( bin,0 ) != checkCode ) {
            // 何もしない.
            return ;
        }
        
        UnderReceiveTable table = tableFactory.getUnderReceiveTable() ;
        UnderReceiveBean recvBean = null ;
        
        synchronized( sync.getUnderReceiveSync().get() ) {
            
            // 受信コネクションIDに対して受信中Bean情報を取得.
            recvBean = table.getUnderReceiveBean( connectId ) ;
            
            // 受信中テーブルに存在しない場合.
            if( recvBean == null ) {
                // 何もしない.
                return ;
            }
            
            // 更新時間を更新.
            recvBean.updateTime() ;
        }
        
        // 取得受信中Beanにパケット情報を追加.
        recvBean.putPacketData( packetNo,packetLength,bin ) ;
        
    }
    
    /**
     * 受信処理 : パケット再送依頼.
     */
    private void receiveResendPacket(
        MaachangConnectTableFactory tableFactory,SendReceiveBean bean,
        MaachangConnectSync sync,String cb32Word ) {
        
        LOG.debug( "[ExecutionAnalysis](受信処理)パケット再送依頼" ) ;
        
        //////////////////////////////////////////////////////////////////
        // 受信したパケット内の再送パケット内容を送信する.
        //////////////////////////////////////////////////////////////////
        
        byte[] bin = bean.getTelegram() ;
        if( bin == null || bin.length <= ProtocolDef.RETRY_PACKET_HEADER_LENGTH ) {
            return ;
        }
        
        int pnt = 2 ;
        // コネクションIDを取得.
        long connectId = ConvertParam.convertLong( pnt,bin ) ;
        pnt += 8 ;
        
        // 再送パケット長を取得.
        int packetLength = ConvertParam.convertInt( pnt,bin ) ;
        pnt += 4 ;
        
        // 取得条件が不正な場合.
        if( connectId <= -1L || packetLength <= 0 ) {
            // 何もしない.
            return ;
        }
        
        SendTable sndTable = tableFactory.getSendTable() ;
        ConnectBean connBean = null ;
        
        synchronized( sync.getSendSync().get() ) {
            
            // 送信テーブルから、コネクションBeanを取得.
            connBean = sndTable.getConnectBean( connectId ) ;
            
            // コネクションBeanが存在しない場合.
            if( connBean == null ) {
                // 何もしない.
                return ;
            }
            
            // 更新時間を更新.
            connBean.updateTime() ;
            
        }
        
        byte[] sendBin = null ;
        
        // 要求パケット情報を送信.
        for( int i = 0 ; i < packetLength ; i ++ ) {
            int packetNo = ConvertParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            sendBin = connBean.getPacketData( tableFactory,packetNo ) ;
            if( sendBin != null ) {
                tableFactory.getSendReceiveTable().addSend(
                    bean.getAddress(),bean.getPort(),sendBin ) ;
            }
        }
    }
    
    /**
     * 受信処理 : パケット完了通知.
     */
    private void receiveExitPacket(
        MaachangConnectTableFactory tableFactory,SendReceiveBean bean,
        MaachangConnectSync sync,String cb32Word ) {
        
        LOG.debug( "[ExecutionAnalysis](受信処理)パケット完了通知" ) ;
        
        //////////////////////////////////////////////////////////////////
        // 受信されたパケット内のコネクションIDのデータに対して、
        // 完了フラグをセット.
        //////////////////////////////////////////////////////////////////
        
        byte[] bin = bean.getTelegram() ;
        if( bin == null || bin.length != ProtocolDef.EXIT_PACKET_LENGTH ) {
            return ;
        }
        
        int pnt = 2 ;
        // コネクションIDを取得.
        long connectId = ConvertParam.convertLong( pnt,bin ) ;
        pnt += 8 ;
        
        // 取得条件が不正な場合.
        if( connectId <= -1L ) {
            // 何もしない.
            return ;
        }
        
        SendTable sndTable = tableFactory.getSendTable() ;
        
        synchronized( sync.getSendSync().get() ) {
            
            // 送信テーブルから、コネクションBeanを取得.
            ConnectBean connBean = sndTable.getConnectBean( connectId ) ;
            
            // コネクションBeanが存在しない場合.
            if( connBean == null ) {
                // 何もしない.
                return ;
            }
            
            // 完了フラグを設定.
            connBean.updateTime() ;
            connBean.setEndSendFlag( true ) ;
            
        }
        
    }
    
}

