/*
 * @(#)ExecutionConnect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread.execution ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.connect.table.ConnectBean;
import org.maachang.connect.table.ConnectTable;
import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.thread.MaachangConnectBaseExecution;
import org.maachang.connect.thread.ProtocolDef;

import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.util.ConvertParam;
import org.maachang.commons.util.UtilCom;

/**
 * コネクションパケット送信処理.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ExecutionConnect implements MaachangConnectBaseExecution {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionConnect.class ) ;
    
    /**
     * コネクションパケット送信回数.
     */
    private static final int CONNECT_SEND_COUNT = 2 ;
    
    /**
     * コンストラクタ.
     */
    public ExecutionConnect() {
        
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * MaachangConnectプロトコル実行処理.
     * <BR>
     * @param threadNum スレッド項番を設定されます.
     * @param udp 対象のUDPプロトコルを設定されます.
     * @param tableFactory テーブルFactoryを設定されます.
     * @param sync 同期オブジェクトが設定されます.
     * @param connectAddress 受信処理などに利用するコネクションアドレス.
     * @param cb32Word 暗号解析結果の暗号ワードを格納する情報が設定されます.
     * @exception Exception 処理例外.
     */
    public void execution( int threadNum,BaseUdpProtocol udp,
        MaachangConnectTableFactory tableFactory,MaachangConnectSync sync,
        ConnectAddress connectAddress,String[] cb32Word ) throws Exception {
        
        try {
            
            UtilCom.idleTime() ;
            
            ConnectBean bean = null ;
            ConnectTable connectTable = tableFactory.getConnectTable() ;
            
            int len ;
            long[] ids = connectTable.getConnectIds() ;
            
            if( ids != null && ( len = ids.length ) > 0 ) {
                
                // コネクションテーブルに設定されている内容を処理.
                for( int i = 0 ; i < len ; i ++ ) {
                    
                    UtilCom.idleTime() ;
                    
                    synchronized( sync.getConnectSync().get() ) {
                        
                        // １つのコネクション条件を取得.
                        if( ( bean = connectTable.getConnectBean( ids[ i ] ) ) == null ) {
                            continue ;
                        }
                        
                    }
                    
                    // コネクション送信可能時間を計算.
                    long sendTimeout = bean.getBeforeSendTime() + 
                        connectTable.getSendPacketTimeout() ;
                    
                    // コネクション送信可能な場合.
                    if( sendTimeout <= System.currentTimeMillis() ) {
                        
                        // コネクションデータ送信.
                        this.sendConnectPacket( tableFactory,bean ) ;
                        
                    }
                    
                }
                
            }
            
        } catch( Exception e ) {
            LOG.warn( "コネクション送信処理で例外",e ) ;
        }
        
    }
    
    /**
     * コネクションパケット送信処理.
     */
    private void sendConnectPacket( MaachangConnectTableFactory tableFactory,ConnectBean bean ) {
        
        LOG.debug( "[ExecutionConnect]コネクション送信" ) ;
        
        int pnt = 0 ;
        byte[] bin = null ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            bin = ProtocolDef.getCb32Binary( ProtocolDef.SEND_CONNECT_LENGTH ) ;
            pnt = ProtocolDef.CB32_LENGTH_PLUS ;
        }
        // 非暗号条件の場合.
        else {
            bin = new byte[ ProtocolDef.SEND_CONNECT_LENGTH ] ;
            pnt = 0 ;
        }
        
        // コネクション送信ヘッダ.
        ConvertParam.convertShort(
            bin,pnt,( short )( ProtocolDef.HEADER_SEND_CONNECT & 0x0000ffff ) ) ;
        pnt += 2 ;
        
        // コネクションID.
        ConvertParam.convertLong( bin,pnt,bean.getId() ) ;
        pnt += 8 ;
        
        // 送信データサイズ.
        ConvertParam.convertInt( bin,pnt,bean.getSendData().size() ) ;
        pnt += 4 ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            // 暗号処理.
            ProtocolDef.encryption(
                tableFactory.getCb32Table(),bean.getCb32Word(),bin ) ;
        }
        
        // 送信条件としてセット.
        tableFactory.getSendReceiveTable().addSend(
            bean.getAddress(),bean.getPort(),
            bin,CONNECT_SEND_COUNT ) ;
        
        // 次回送信時間を更新.
        bean.updateBeforeSendTime() ;
    }
    
}

