/*
 * @(#)MqConfig.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.config ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;

/**
 * MaachangQコンフィグファイル読み込む処理.
 *
 * @version 2006/12/31
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public interface MqConfig {
    
    /**
     * キャッシュディレクトリを取得.
     * <BR><BR>
     * キャッシュディレクトリを取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[cache-directory]情報を取得します.
     * <BR>
     * @return String キャッシュディレクトリ名が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getCacheDirectory() ;
    
    /**
     * 構成情報セーブAndロードディレクトリ名を取得.
     * <BR><BR>
     * 構成情報セーブAndロードディレクトリ名を取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[save-and-load-directory]情報を取得します.
     * <BR>
     * @return String 構成情報セーブAndロードディレクトリ名が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getSaveAndLoadDirectory() ;
    
    /**
     * オートセーブ間隔を取得.
     * <BR><BR>
     * オートセーブ間隔を取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[auto-save-timing]情報を取得します.
     * <BR>
     * @return long オートセーブ間隔が返されます.<BR>
     *              情報が存在しない場合[-1]が返されます.
     */
    public long getAutoSaveTiming() ;
    
    /**
     * 通信バインドポートを取得.
     * <BR><BR>
     * 通信バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[bind-port]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return int バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getBindPort( int no ) ;
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[bind-address]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return InetAddress バインドアドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getBindAddress( int no ) ;
    
    /**
     * SSL通信バインドポートを取得.
     * <BR><BR>
     * SSL通信バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-bind-port]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return int バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSSLBindPort( int no ) ;
    
    /**
     * SSLバインドアドレスを取得.
     * <BR><BR>
     * SSLバインドアドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-bind-address]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return InetAddress バインドアドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getSSLBindAddress( int no ) ;
    
    /**
     * バインド情報数を取得.
     * <BR><BR>
     * 設定されているバインド情報数が返されます.
     * <BR>
     * @return int バインド情報数が返されます.
     */
    public int getBindSize() ;
    
    /**
     * 受付コネクション最大数を取得.
     * <BR><BR>
     * 受付コネクション最大数を取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[max-access]情報を取得します.
     * <BR>
     * @return int 受付コネクション最大数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getMaxAccess() ;
    
    /**
     * アドミニ用バインドポートを取得.
     * <BR><BR>
     * アドミニ用バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[admin-port]情報を取得します.
     * <BR>
     * @return int アドミニ用バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getAdminPort() ;
    
    /**
     * アドミニ用アドレスを取得.
     * <BR><BR>
     * アドミニ用アドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[admin-address]情報を取得します.
     * <BR>
     * @return InetAddress アドミニ用アドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getAdminAddress() ;
    
    /**
     * SSLアドミニ用バインドポートを取得.
     * <BR><BR>
     * SSLアドミニ用バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-admin-port]情報を取得します.
     * <BR>
     * @return int アドミニ用バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSSLAdminPort() ;
    
    /**
     * SSLアドミニ用アドレスを取得.
     * <BR><BR>
     * SSLアドミニ用アドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-admin-address]情報を取得します.
     * <BR>
     * @return InetAddress アドミニ用アドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getSSLAdminAddress() ;
    
    /**
     * 送信スレッド数を取得.
     * <BR><BR>
     * 送信スレッド数を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[thread]情報を取得します.
     * <BR>
     * @return int 送信スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSendThread() ;
    
    /**
     * ハートビート間隔を取得.
     * <BR><BR>
     * ハートビート間隔を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[heartbeat-timing]情報を取得します.
     * <BR>
     * @return long ハートビート間隔が返されます.<BR>
     *              情報が存在しない場合[-1]が返されます.
     */
    public long getHearbeatTiming() ;
    
    /**
     * ハートビートコネクション不能判定値を取得.
     * <BR><BR>
     * ハートビートコネクション不能判定値を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[heartbeat-close-count]情報を取得します.
     * <BR>
     * @return int ハートビートコネクション不能判定値が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getHeartBeatCloseCount() ;
    
    /**
     * 受信コアスレッド数を取得.
     * <BR><BR>
     * 受信コアスレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[core-thread]情報を取得します.
     * <BR>
     * @return int 受信コアスレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getCoreThread() ;
    
    /**
     * 受信電文判別スレッド数を取得.
     * <BR><BR>
     * 受信電文判別スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[switch-thread]情報を取得します.
     * <BR>
     * @return int 受信電文判別スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSwitchThread() ;
    
    /**
     * 受信データ解析スレッド数を取得.
     * <BR><BR>
     * 受信データ解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[data-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信データ解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getDataAnalysisThread() ;
    
    /**
     * 受信ハートビート解析スレッド数を取得.
     * <BR><BR>
     * 受信ハートビート解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[heartbeat-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信ハートビート解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getHeartBeatAnalysisThread() ;
    
    /**
     * 受信正常結果解析スレッド数を取得.
     * <BR><BR>
     * 受信正常結果解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[success-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信正常結果解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSuccessAnalysisThread() ;
    
    /**
     * 受信エラー結果解析スレッド数を取得.
     * <BR><BR>
     * 受信エラー結果解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[error-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信エラー結果解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getErrorAnalysisThread() ;
    
    /**
     * 対象の情報を文字列で取得.
     * <BR><BR>
     * 対象の情報を文字列で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return String 対象の条件をStringで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    public String get( String section,String key,int no ) ;
    
    /**
     * 対象条件数を取得.
     * <BR><BR>
     * 対象の条件の定義数を取得します.
     * <BR>
     * @return int 定義数が返されます.
     */
    public int size( String section,String key ) ;
    
    /**
     * 定義ファイルで必要なセクション名群が有効であるかチェック.
     * <BR><BR>
     * 定義ファイルで必要なセクション名群が有効であるかチェックします.
     * <BR>
     * @return boolean 有効であるかチェックします.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、有効ではありません.
     */
    public boolean isUseSection() ;
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * @param section 対象のセクション名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、セクション名は存在します.<BR>
     *                 [false]が返された場合、セクション名は存在しません.
     */
    public boolean isSection( String section ) ;
    
}
