/*
 * @(#)SSLConfigImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.config.service ;

import org.maachang.commons.conf.BaseConfig;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.io.FileAccessException;
import org.maachang.queue.config.SSLConfig;

/**
 * サーバSSLコンフィグファイル読み込む処理.
 *
 * @version 2007/01/13
 * @author  Masahito Suzuki
 * @since   MaachangQ-1.00
 */
class SSLConfigImple extends BaseConfig implements SSLConfig {
    
    /**
     * コンフィグセクション : SSL定義.
     */
    private static final String SECTION_SSL = "ssl" ;
    
    /**
     * コンフィグセクション : キーストア定義.
     */
    private static final String SECTION_KEY_STORE = "keystore" ;
    
    /**
     * コンフィグセクション : トラストストア定義.
     */
    private static final String SECTION_TRUST_STORE = "truststore" ;
    
    /**
     * SSL定義 : SSLプロトコル利用条件.
     */
    private static final String IS_SSL = "ssl" ;
    
    /**
     * SSL定義 : SSSLプロトコル.
     */
    private static final String SSL_PROTOCOL = "protocol" ;
    
    /**
     * SSL定義 : 乱数アルゴリズム(PRNG).
     */
    private static final String RANDOM_ALGORITHM = "random-algorithm" ;
    
    /**
     * SSL定義 : クライアント認証モード.
     */
    private static final String CLIENT_AUTH = "client-auth" ;
    
    /**
     * キーストア定義 : キーストアタイプ.
     */
    private static final String KEY_STORE_TYPE = "key-store" ;
    
    /**
     * キーストア定義 : キーストアマネージャアルゴリズム.
     */
    private static final String KEY_MANAGER_ALGORITHM = "key-manager-algorithm" ;
    
    /**
     * キーストア定義 : キーストアパスワード.
     */
    private static final String KEY_PASSWD = "key-passwd" ;
    
    /**
     * キーストア定義 : キーストアファイル.
     */
    private static final String KEY_FILE = "key-file" ;
    
    /**
     * トラストストア定義 : トラストストア利用条件.
     */
    private static final String IS_TRUST = "trust" ;
    
    /**
     * トラストストア定義 : トラストストアタイプ.
     */
    private static final String TRUST_STORE_TYPE = "trust-store" ;
    
    /**
     * トラストストア定義 : トラストストアマネージャアルゴリズム.
     */
    private static final String TRUST_MANAGER_ALGORITHM = "trust-manager-algorithm" ;
    
    /**
     * トラストストア定義 : トラストストアパスワード.
     */
    private static final String TRUST_PASSWD = "trust-passwd" ;
    
    /**
     * トラストストア定義 : トラストストアファイル.
     */
    private static final String TRUST_FILE = "trust-file" ;
    
    /**
     * コンストラクタ.
     */
    public SSLConfigImple() {
        super() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * MQ定義ファイルを指定して生成.
     * <BR><BR>
     * MQ定義ファイルを指定して生成します.
     * <BR>
     * @param name 生成対象のファイル名を設定します.
     * @exception ExecutionException 実行例外.
     */
    public void create( String name )
        throws ExecutionException {
        
        try{
            
            super.open( name ) ;
            
        }catch( BaseException be ){
            this.clear() ;
            throw new ExecutionException( be ) ;
        }catch( Exception e ){
            this.clear() ;
            throw new ExecutionException( e ) ;
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクト情報をクリアします.
     */
    public void clear() {
        
        super.close() ;
    }
    
    /**
     * SSLプロトコル利用条件を取得.
     * <BR><BR>
     * SSLプロトコル利用条件を取得します.<BR>
     * この情報はセクション名[ssl]以下に
     * 設定されている[ssl]情報を取得します.
     * <BR>
     * @return boolean SSLプロトコル利用条件が返されます.
     */
    public boolean isSSL() {
        
        boolean ret = false ;
        
        try{
            
            ret = this.getBoolean(
                SSLConfigImple.SECTION_SSL,
                SSLConfigImple.IS_SSL,
                0
            ) ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * SSLプロトコルを取得.
     * <BR><BR>
     * SSLプロトコルを取得します.<BR>
     * この情報はセクション名[ssl]以下に
     * 設定されている[protocol]情報を取得します.
     * <BR>
     * @return String SSLプロトコル名が返されます.
     */
    public String getSSlProtocol() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_SSL,
                SSLConfigImple.SSL_PROTOCOL,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 乱数アルゴリズム(PRNG)を取得.
     * <BR><BR>
     * 乱数アルゴリズム(PRNG)を取得します.<BR>
     * この情報はセクション名[ssl]以下に
     * 設定されている[random-algorithm]情報を取得します.
     * <BR>
     * @return String 乱数アルゴリズム(PRNG)が返されます.
     */
    public String getRandomAlgorithm() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_SSL,
                SSLConfigImple.RANDOM_ALGORITHM,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * クライアント認証モードを取得.
     * <BR><BR>
     * クライアント認証モードを取得します.<BR>
     * この情報はセクション名[ssl]以下に
     * 設定されている[client-auth]情報を取得します.
     * <BR>
     * @return boolean クライアント認証モードが返されます.
     */
    public boolean isClientAuth() {
        
        boolean ret = false ;
        
        try{
            
            ret = this.getBoolean(
                SSLConfigImple.SECTION_SSL,
                SSLConfigImple.CLIENT_AUTH,
                0
            ) ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * キーストアタイプを取得.
     * <BR><BR>
     * キーストアタイプを取得します.<BR>
     * この情報はセクション名[keystore]以下に
     * 設定されている[key-store]情報を取得します.
     * <BR>
     * @return String キーストアタイプが返されます.
     */
    public String getKeyStore() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_KEY_STORE,
                SSLConfigImple.KEY_STORE_TYPE,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キーストアマネージャアルゴリズムを取得.
     * <BR><BR>
     * キーストアマネージャアルゴリズムを取得します.<BR>
     * この情報はセクション名[keystore]以下に
     * 設定されている[key-manager-algorithm]情報を取得します.
     * <BR>
     * @return String キーストアマネージャアルゴリズムが返されます.
     */
    public String getKeyManagerAlgorithm() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_KEY_STORE,
                SSLConfigImple.KEY_MANAGER_ALGORITHM,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キーストアパスワードを取得.
     * <BR><BR>
     * キーストアパスワードを取得します.<BR>
     * この情報はセクション名[keystore]以下に
     * 設定されている[key-passwd]情報を取得します.
     * <BR>
     * @return String キーストアパスワードが返されます.
     */
    public String getKeyPasswd() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_KEY_STORE,
                SSLConfigImple.KEY_PASSWD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キーストアファイルを取得.
     * <BR><BR>
     * キーストアファイルを取得します.<BR>
     * この情報はセクション名[keystore]以下に
     * 設定されている[key-file]情報を取得します.
     * <BR>
     * @return String キーストアファイルが返されます.
     */
    public String getKeyFile() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_KEY_STORE,
                SSLConfigImple.KEY_FILE,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * トラストストア利用条件を取得.
     * <BR><BR>
     * トラストストア利用条件を取得します.<BR>
     * この情報はセクション名[truststore]以下に
     * 設定されている[trust]情報を取得します.
     * <BR>
     * @return boolean トラストストア利用条件が返されます.
     */
    public boolean isTrust() {
        
        boolean ret = false ;
        
        try{
            
            ret = this.getBoolean(
                SSLConfigImple.SECTION_TRUST_STORE,
                SSLConfigImple.IS_TRUST,
                0
            ) ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * トラストストアタイプを取得.
     * <BR><BR>
     * トラストストアタイプを取得します.<BR>
     * この情報はセクション名[truststore]以下に
     * 設定されている[trust-store]情報を取得します.
     * <BR>
     * @return String トラストストアタイプが返されます.
     */
    public String getTrustStore() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_TRUST_STORE,
                SSLConfigImple.TRUST_STORE_TYPE,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * トラストストアマネージャアルゴリズムを取得.
     * <BR><BR>
     * トラストストアマネージャアルゴリズムを取得します.<BR>
     * この情報はセクション名[truststore]以下に
     * 設定されている[trust-manager-algorithm]情報を取得します.
     * <BR>
     * @return String トラストストアマネージャアルゴリズムが返されます.
     */
    public String getTrustManagerAlgorithm() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_TRUST_STORE,
                SSLConfigImple.TRUST_MANAGER_ALGORITHM,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * トラストストアパスワードを取得.
     * <BR><BR>
     * トラストストアパスワードを取得します.<BR>
     * この情報はセクション名[truststore]以下に
     * 設定されている[trust-passwd]情報を取得します.
     * <BR>
     * @return String トラストストアパスワードが返されます.
     */
    public String getTrustPasswd() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_TRUST_STORE,
                SSLConfigImple.TRUST_PASSWD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * トラストストアファイルを取得.
     * <BR><BR>
     * トラストストアファイルを取得します.<BR>
     * この情報はセクション名[truststore]以下に
     * 設定されている[trust-file]情報を取得します.
     * <BR>
     * @return String トラストストアファイルが返されます.
     */
    public String getTrustFile() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                SSLConfigImple.SECTION_TRUST_STORE,
                SSLConfigImple.TRUST_FILE,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象の情報を文字列で取得.
     * <BR><BR>
     * 対象の情報を文字列で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return String 対象の条件をStringで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    public String get( String section,String key,int no ) {
        
        String ret = null ;
        
        try{
            
            ret = this.getString( section,key,no ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象条件数を取得.
     * <BR><BR>
     * 対象の条件の定義数を取得します.
     * <BR>
     * @return int 定義数が返されます.
     */
    public int size( String section,String key ) {
        return this.getLength( section,key ) ;
    }
    
    /**
     * 定義ファイルで必要なセクション名群が有効であるかチェック.
     * <BR><BR>
     * 定義ファイルで必要なセクション名群が有効であるかチェックします.
     * <BR>
     * @return boolean 有効であるかチェックします.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、有効ではありません.
     */
    public boolean isUseSection() {
        return (
            super.isSection( SSLConfigImple.SECTION_SSL ) == false ||
            super.isSection( SSLConfigImple.SECTION_KEY_STORE ) == false ||
            super.isSection( SSLConfigImple.SECTION_TRUST_STORE ) == false
        ) ? false : true ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * @param section 対象のセクション名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、セクション名は存在します.<BR>
     *                 [false]が返された場合、セクション名は存在しません.
     */
    public boolean isSection( String section ) {
        return super.isSection( section ) ;
    }
    
    
    
    
    /**
     * 対象コンフィグ情報を取得.
     * <BR><BR>
     * 対象のコンフィグ情報を取得します.
     * <BR>
     * @param name 対象のコンフィグ名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    protected final void newInit( String name )
        throws InputException,FileAccessException,ReadSyntaxException {
        
        try{
            synchronized( m_sync.get() ){
                m_conf.create( name ) ;
                m_name = name ;
                m_isRead = true ;
            }
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }
    }
    
}
