/*
 * @(#)ExecutionServerMessage.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect ;

import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.MessageBean;
import org.maachang.queue.access.protocol.MessageProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.access.status.StatusDefine;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.ExecutionReceiveClient;
import org.maachang.queue.connect.common.ServerConnectThread;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.connect.common.TransactionData;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.Mq;
import org.maachang.queue.main.queue.MqManager;
import org.maachang.queue.main.queue.Queue;
import org.maachang.queue.main.queue.QueueDef;
import org.maachang.queue.main.queue.QueueValue;
import org.maachang.queue.main.queue.ReceiveMq;
import org.maachang.queue.main.queue.SendMqInterface;

/**
 * メッセージ情報受信実行処理.
 *
 * @version 2007/01/14
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionServerMessage implements ExecutionReceiveClient {
    
    /**
     * 受信データなしの場合の待機時間.
     */
    private static final int WAIT_NOT_RECEIVE = 500 ;
    
    /**
     * トランザクション取得失敗の場合の待機時間.
     */
    private static final int WAIT_NOT_TRANSACTION = 1500 ;
    
    
    /**
     * コンストラクタ.
     */
    public ExecutionServerMessage() {
        
    }
    
    /**
     * スレッド終了処理.
     * <BR><BR>
     * スレッドが終了するときに呼び出されます.
     * <BR>
     * @param session 対象のログインセッションが設定されます.
     * @param tranMan 対象のトランザクションマネージャが設定されます.
     */
    public void exit( LoginSession session,CharTable tranMan ) {
        
        // トランザクション中の条件を強制的にロールバック.
        try {
            
            String[] names = tranMan.getNames() ;
            
            if( names != null && names.length > 0 ) {
                
                int len = names.length ;
                TransactionData data = null ;
                
                for( int i = 0 ; i < len ; i ++ ) {
                    
                    data = ( TransactionData )tranMan.get( names[ i ] ) ;
                    
                    if( data.getTranId() > 0 ) {
                       
                        try {
                            
                            Mq mq = MqManager.get(
                                data.getManager(),
                                data.getQueue(),
                                data.getQueueType() ) ;
                            
                            if( mq != null ) {
                                //mq.rollback( data.getTranId() ) ;
                                mq.commit( data.getTranId() ) ;
                            }
                            
                        } catch( Exception ee ) {
                        }
                        
                    }
                    
                }
                
                tranMan.clear() ;
                
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * クライアント接続実行処理.
     * <BR><BR>
     * クライアント接続実行処理を行います.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param tranMan 対象のトランザクションマネージャが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param type 対象の電文処理タイプが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @exception Exception 例外.
     */
    public void execution( int[] telegramType,LoginSession session,
        CharTable tranMan,ConnectObject connect,int id,int type,
        Telegram telegram )
        throws Exception {
        
        // 受信タイプが「メッセージ電文」の場合.
        if( type == MessageProtocol.CATEGORY_TYPE_MESSAGE ) {
            
            byte[] binary = telegram.getTelegram() ;
            telegram.clear() ;
            
            // 受信バイナリを解析.
            MessageBean bean = new MessageBean() ;
            MessageProtocol.analysisProtocol( bean,binary ) ;
            bean.setId( id ) ;
            binary = null ;
            
            // 実行区分が異常条件の場合.
            if( bean.getType() != MessageProtocol.TYPE_CONNECT_MANAGER &&
                bean.getType() != MessageProtocol.TYPE_CONNECT_QUEUE &&
                bean.getType() != MessageProtocol.TYPE_SEND &&
                bean.getType() != MessageProtocol.TYPE_RECEIVE &&
                bean.getType() != MessageProtocol.TYPE_TRANSACTION &&
                bean.getType() != MessageProtocol.TYPE_COMMIT &&
                bean.getType() != MessageProtocol.TYPE_ROLLBACK ) {
                return ;
            }
            
            // 解析された実行区分を設定.
            telegramType[ 0 ] = bean.getType() ;
            
            // 受信メッセージの処理区分から、個別の処理を行う.
            switch( bean.getType() ) {
                
                // キューマネージャ存在確認.
                case MessageProtocol.TYPE_CONNECT_MANAGER :
                    this.isQueueManager( connect,bean ) ;
                    break ;
                
                // キュー存在確認.
                case MessageProtocol.TYPE_CONNECT_QUEUE :
                    this.isQueue( connect,bean ) ;
                    break ;
                
                // 電文送信.
                case MessageProtocol.TYPE_SEND :
                    this.sendMessage( tranMan,connect,bean ) ;
                    break ;
                
                // 電文受信.
                case MessageProtocol.TYPE_RECEIVE :
                    this.receiveMessage( tranMan,connect,bean ) ;
                    break ;
                
                // トランザクション開始.
                case MessageProtocol.TYPE_TRANSACTION :
                    this.transaction( tranMan,connect,bean ) ;
                    break ;
                
                // コミット処理.
                case MessageProtocol.TYPE_COMMIT :
                    this.commit( tranMan,connect,bean ) ;
                    break ;
                
                // ロールバック処理.
                case MessageProtocol.TYPE_ROLLBACK :
                    this.rollback( tranMan,connect,bean ) ;
                    break ;
                
            }
            
        }
        
    }
    
    /**
     * キューマネージャ存在確認.
     */
    private void isQueueManager( ConnectObject connect,MessageBean bean )
        throws Exception {
        
        int errorCode = MaachangQErrorCode.SUCCESS ;
        String message = null ;
        
        // キューマネージャ取得.
        QueueManager man = QueueManagerFactory.get( bean.getQueueManager() ) ;
        
        // キューマネージャが存在しない.
        if( man == null ) {
            errorCode = MaachangQErrorCode.ERROR_NOT_QUEUE_MANAGER ;
            message = "キューマネージャ(" + bean.getQueueManager() +
                ")は存在しません" ;
        }
        
        // キューマネージャのステータスは利用不可能.
        if( ( man.getState() & StatusDefine.STATE_MASK_ERROR ) == StatusDefine.STATE_MASK_ERROR ) {
            errorCode = MaachangQErrorCode.ERROR_QUEUE_MANAGER_NOT_SUCCESS ;
            message = "キューマネージャ(" + bean.getQueueManager() +
                ")のステータス(" +
                QueueManagerStatus.getStateByString( man.getState() ) +
                ")は、利用不可です" ;
        }
        else {
            message = "キューマネージャは存在" ;
        }
        
        // 処理結果を戻す.
        this.sendResult( connect,bean,errorCode,message ) ;
        
    }
    
    /**
     * キュー存在確認.
     */
    private void isQueue( ConnectObject connect,MessageBean bean )
        throws Exception {
        
        int errorCode = MaachangQErrorCode.SUCCESS ;
        String message = null ;
        
        // MQ取得.
        Mq mq = MqManager.get(
            bean.getQueueManager(),
            bean.getQueue(),
            bean.getQueueType() ) ;
            
        // オートコミット情報を設定.
        message = String.valueOf( mq.isAutoCommit() ) ;
        
        // 処理結果を戻す.
        this.sendResult( connect,bean,errorCode,message ) ;
        
    }
    
    /**
     * クライアントから送信電文が送られた.
     */
    private void sendMessage( CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
        
        // MQ取得.
        SendMqInterface send = ( SendMqInterface )MqManager.get(
            bean.getQueueManager(),
            bean.getQueue(),
            bean.getQueueType() ) ;
        
        // トランザクションデータを取得.
        TransactionData tran = this.getTransaction(
            tranMan,( Mq )send,bean ) ;
        int tranId = Queue.TRANSACTION_BY_AUTOCOMMIT ;
        if( tran != null ) {
            tranId = tran.getTranId() ;
        }
        
        // キュー同期を設定.
        synchronized( ( ( Mq )send ).getSynchronized() ) {
            
            // メッセージをキューに設定.
            send.put( tranId,bean.getKey(),bean.getProcessId(),
                bean.getMessage(),bean.getPriority(),
                bean.getExpire() ) ;
            bean.setMessage( null ) ;
            
            // 処理結果を戻す.
            this.sendResult( connect,bean,
                MaachangQErrorCode.SUCCESS,"メッセージを設定" ) ;
        }
        
    }
    
    /**
     * クライアントから対象キューのメッセージ取得を要求された.
     */
    private void receiveMessage( CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
        
        int errorCode = MaachangQErrorCode.SUCCESS ;
        String message = null ;
        
        // 指定キュータイプは、受信キューでない場合.
        if( bean.getQueueType() != QueueDef.TYPE_RECEIVE ) {
            errorCode = MaachangQErrorCode.ERROR_QUEUE_TYPE ;
            message = "指定キュータイプは不正です" ;
        }
        else {
            
            // 受信MQ取得.
            ReceiveMq mq = ( ReceiveMq )MqManager.get(
                bean.getQueueManager(),
                bean.getQueue(),
                bean.getQueueType() ) ;
            
            QueueValue result = new QueueValue() ;
            
            // トランザクションデータを取得.
            TransactionData tran = this.getTransaction(
                tranMan,mq,bean ) ;
            int[] tranId = null ;
            if( tran != null ) {
                tranId = tran.getTranIds() ;
            }
            
            // キュー同期.
            synchronized( mq.getSynchronized() ) {
                
                // 受信処理.
                if( mq.get( result,tranId,bean.getKey() ) == false ) {
                    
                    // 情報が存在しない場合.
                    UtilCom.sleep( WAIT_NOT_RECEIVE ) ;
                    errorCode = MaachangQErrorCode.WARNING_NOT_MESSAGE ;
                    message = "データなし" ;
                    
                }
                else {
                    
                    // 情報が存在する場合.
                    bean.setKey( result.getKey() ) ;
                    bean.setProcessId( result.getProcessId() ) ;
                    bean.setMessage( result.getBinaryMessage() ) ;
                    bean.setExpire( ( int )result.getExpire() ) ;
                    bean.setPriority( result.getPriority() ) ;
                    
                    // 電文をコンバート.
                    byte[] bin = MessageProtocol.createProtocol(
                        bean.getId(),bean.getType(),bean ) ;
                    bean.clear() ;
                    
                    // 電文を送信.
                    connect.send( bin ) ;
                    
                    // 送信成功の場合は、対象送信データを
                    // キューから削除.
                    mq.successGet( tranId,result ) ;
                    
                    return ;
                }
                
            }
            
        }
        
        // 処理結果を戻す.
        this.sendResult( connect,bean,errorCode,message ) ;
        
    }
    
    /**
     * クライアントからトランザクション開始を要求.
     */
    private void transaction( CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
        
        int errorCode = MaachangQErrorCode.SUCCESS ;
        String message = null ;
        
        // MQ取得.
        Mq mq = MqManager.get(
            bean.getQueueManager(),
            bean.getQueue(),
            bean.getQueueType() ) ;
        
         // トランザクションデータを取得.
        TransactionData tran = this.getTransaction(
            tranMan,mq,bean ) ;
        
        // このコネクションがトランザクション開始でない場合.
        if( tran != null && tran.getTranId() == -1 ) {
            
            // 他のコネクションがトランザクションを開始している場合.
            if( mq.getTransactionId() > 0 ) {
                // 少し待機.
                UtilCom.sleep( WAIT_NOT_TRANSACTION ) ;
            }
            
            // トランザクションを取得.
            int id = mq.transaction() ;
            
            // トランザクションが取得できた場合.
            if( id > 0 ) {
                tran.setTranId( id ) ;
                message = "トランザクション取得" ;
            }
            
        }
        else if( tran == null ) {
            message = "オートコミット" ;
        }
        else if( tran.getTranId() > 0 ) {
            message = "トランザクション取得済み" ;
        }
        
        // 処理結果を戻す.
        this.sendResult( connect,bean,errorCode,message ) ;
        
    }
    
    /**
     * クライアントからコミットを要求.
     */
    private void commit( CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
       this.commitOrRollback( true,tranMan,connect,bean ) ;
    }
    
    /**
     * クライアントからロールバックを要求.
     */
    private void rollback( CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
       this.commitOrRollback( false,tranMan,connect,bean ) ;
    }
    
    /**
     * コミットorロールバック.
     */
    private void commitOrRollback(
        boolean mode,CharTable tranMan,ConnectObject connect,MessageBean bean )
        throws Exception {
        
        int errorCode = MaachangQErrorCode.SUCCESS ;
        String message = null ;
        
        // MQ取得.
        Mq mq = MqManager.get(
            bean.getQueueManager(),
            bean.getQueue(),
            bean.getQueueType() ) ;
        
         // トランザクションデータを取得.
        TransactionData tran = this.getTransaction(
            tranMan,mq,bean ) ;
        
        // このコネクションがトランザクション開始中の場合.
        if( tran != null && tran.getTranId() > 0 ) {
            
            if( mode == true ) {
                mq.commit( tran.getTranId() ) ;
                
                // トランザクションを削除.
                this.removeTransaction( tranMan,bean ) ;
                
                message = "コミット完了" ;
            }
            else {
                mq.rollback( tran.getTranId() ) ;
                
                // トランザクションを削除.
                this.removeTransaction( tranMan,bean ) ;
                
                message = "ロールバック完了" ;
            }
        }
        // トランザクション開始中でないか、オートコミットの場合.
        else if( mode == true ) {
            message = "コミット不要" ;
        }
        else {
            message = "ロールバック不要" ;
        }
        
        // 処理結果を戻す.
        this.sendResult( connect,bean,errorCode,message ) ;
        
    }
    
    /**
     * メッセージの送信.
     */
    private void sendResult( ConnectObject connect,MessageBean bean,
        int errorCode,String message ) {
        
        int errType ;
        if( errorCode == MaachangQErrorCode.SUCCESS ) {
            errType = ResultProtocol.TYPE_SUCCESS ;
        }
        else {
            errType = ResultProtocol.TYPE_ERROR ;
        }
        
        ServerConnectThread.sendRsult(
            bean.getId(),errType,bean.getType(),bean.getQueueManager(),
            bean.getQueue(),bean.getQueueType(),connect,
            message,errorCode ) ;
    }
    
    /**
     * 対象トランザクションを取得.
     */
    private final TransactionData getTransaction( CharTable tranMan,Mq mq,MessageBean bean ) {
        
        if( mq.isAutoCommit() == false ) {
            
            String index = ExecutionServerMessage.getIndex( bean ) ;
            TransactionData data = ( TransactionData )tranMan.get( index ) ;
            
            if( data == null ) {
                data = new TransactionData() ;
                data.setManager( bean.getQueueManager() ) ;
                data.setQueue( bean.getQueue() ) ;
                data.setQueueType( bean.getType() ) ;
                tranMan.add( index,data ) ;
            }
            
            return data ;
            
        }
        
        return null ;
    }
    
    /**
     * 対象トランザクションを削除.
     */
    private final void removeTransaction( CharTable tranMan,MessageBean bean ) {
        String index = ExecutionServerMessage.getIndex( bean ) ;
        tranMan.remove( index ) ;
    }
    
    /**
     * 指定トランザクション名を生成.
     */
    private static final String getIndex( MessageBean bean ) {
        return new StringBuffer().
            append( bean.getQueue() ).
            append( "@" ).
            append( bean.getQueueType() ).
            append( "-" ).
            append( bean.getQueueManager() ).
            toString() ;
    }
    
}

