/*
 * @(#)ExecutionAdminConnect.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import java.util.HashSet;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminConnectBean;
import org.maachang.queue.access.protocol.admin.AdminConnectProtocol;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;
import org.maachang.queue.main.service.SaveAndLoadService;


/**
 * コネクション電文処理.
 *
 * @version 2007/01/22
 * @author  Masahito Suzuki
 * @since   FastConnect 1.00
 */
public class ExecutionAdminConnect
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminConnect() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminConnectProtocol.CATEGORY_TYPE_ADMIN_CONNECT ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminConnectBean bean = new AdminConnectBean() ;
        AdminConnectProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminConnectProtocol.TYPE_ADD_CONNECT :
                this.addConnect( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_SET_CONNECT :
                this.setConnect( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_REMOVE_CONNECT :
                this.removeConnect( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_GET_CONNECT :
                this.getConnect( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_GET_CONNECT_NAMES :
                this.getConnectNames( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_GET_CONNECT_SIZE :
                this.getConnectSize( session,connect,bean ) ;
                return true ;
            case AdminConnectProtocol.TYPE_IS_CONNECT :
                this.isConnect( session,connect,bean ) ;
                return true ;
        }
        
        return false ;
        
    }
    
    /**
     * コネクション追加.
     */
    public final void addConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        try {
            synchronized( ConnectFactory.getSync() ) {
                
                if( ConnectFactory.get( bean.getName() ) != null ) {
                    throw new MaachangQException(
                        "コネクション(" + bean.getName() +
                        ")は既に存在します",
                        MaachangQErrorCode.ERROR_USE_CONNECT ) ;
                }
                
                String cacheName = bean.getCacheName() ;
                if( cacheName == null ||
                    ( cacheName = cacheName.trim().toLowerCase() ).length() <= 0 ) {
                    cacheName = null ;
                }
                
                if( cacheName != null && MqCacheFactory.get( cacheName ) == null ) {
                    throw new MaachangQException(
                        "指定キャッシュ名(" + cacheName + ")は存在しません",
                        MaachangQErrorCode.ERROR_NOT_SET_CACHE_CONNECT ) ;
                }
                ConnectFactory.create(
                    bean.getName(),cacheName,bean.getPort(),
                    bean.getInetAddress(),bean.getBuffer(),
                    bean.isCb32(),bean.getParams() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CONNECT ) ;
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "コネクション追加に失敗しました",
                MaachangQErrorCode.ERROR_ADD_CONNECT ) ;
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"コネクション追加" ) ;
        
    }
    
    /**
     * コネクション再設定.
     */
    public final void setConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        try {
            synchronized( ConnectFactory.getSync() ) {
                
                Connect conn = this.getConnect( bean ) ;
                conn.setCb32( bean.isCb32() ) ;
                
                // 暗号条件がOFFの場合.
                if( bean.isCb32() == false ) {
                    // 何もしない.
                    return ;
                }
                
                // 削除対象の条件準備.
                HashSet set = new HashSet() ;
                if( conn.getCb32WordSize() > 0 ) {
                    String[] names = conn.getCb32Word() ;
                    int len = names.length ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        set.add( names[ i ] ) ;
                    }
                }
                
                String[] words = bean.getParams() ;
                
                // 対象条件が存在する場合は、削除準備から除外.
                if( words != null && words.length > 0 ) {
                    int len = words.length ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        if( words[ i ] != null ) {
                            words[ i ] = words[ i ].trim() ;
                            if( words[ i ].length() > 0 ) {
                                if( set.remove( words[ i ] ) == true ) {
                                    words[ i ] = null ;
                                }
                            }
                        }
                    }
                }
                
                // 削除準備に残った内容を現在のコネクションから削除.
                if( set.size() > 0 ) {
                    Object[] objs = set.toArray() ;
                    int len = objs.length ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        conn.removeCb32Word( ( String )objs[ i ] ) ;
                    }
                }
                
                // 現在条件として存在しないものを追加.
                if( words != null && words.length > 0 ) {
                    int len = words.length ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        if( words[ i ] != null && words[ i ].length() > 0 ) {
                            conn.addCb32Word( words[ i ] ) ;
                        }
                    }
                }
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CONNECT ) ;
                
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "コネクション設定に失敗しました",
                MaachangQErrorCode.ERROR_SET_CONNECT ) ;
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"コネクション設定" ) ;
        
    }
    
    /**
     * コネクション削除.
     */
    public final void removeConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ConnectFactory.getSync() ) {
            
            this.getConnect( bean ) ;
            
            try {
                // コネクション削除.
                ConnectFactory.remove( bean.getName() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CONNECT ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "コネクションの削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_CONNECT ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"コネクション削除" ) ;
        
    }
    
    /**
     * コネクション情報取得.
     */
    public final void getConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( ConnectFactory.getSync() ) {
                
                Connect conn = this.getConnect( bean ) ;
                
                bean.setAddress( conn.getBindAddress().getHostName() ) ;
                bean.setPort( conn.getPort() ) ;
                bean.setCacheName( conn.getCacheName() ) ;
                bean.setBuffer( conn.getBuffer() ) ;
                bean.setCb32( conn.isCb32() ) ;
                bean.setParams( conn.getCb32Word() ) ;
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "コネクション取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_CONNECT ) ;
        }
        
        this.sendConnect( session,connect,bean ) ;
        
    }
    
    /**
     * コネクション名群取得.
     */
    public final void getConnectNames(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( ConnectFactory.getSync() ) {
                
                String[] names = ConnectFactory.getNames() ;
                bean.setParams( names ) ;
                
            }
            
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "コネクション名一覧に失敗しました",
                MaachangQErrorCode.ERROR_GET_LIST_CONNECT ) ;
        }
        
        this.sendConnect( session,connect,bean ) ;
        
    }
    
    /**
     * コネクション数取得.
     */
    public final void getConnectSize(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( ConnectFactory.getSync() ) {
                
                int len = ConnectFactory.size() ;
                bean.setSize( len ) ;
                
            }
            
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "コネクション数取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_SIZE_CONNECT ) ;
        }
        
        this.sendConnect( session,connect,bean ) ;
        
    }
    
    /**
     * コネクション存在チェック.
     */
    public final void isConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( ConnectFactory.getSync() ) {
            
            try {
                this.getConnect( bean ) ;
            } catch( MaachangQException fa ) {
                throw fa ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "コネクション存在確認に失敗",
                    MaachangQErrorCode.ERROR_IS_CONNECT ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"コネクション存在" ) ;
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendConnect(
        LoginSession session,ConnectObject connect,AdminConnectBean bean )
        throws Exception {
        
        byte[] binary = AdminConnectProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
    
    /**
     * コネクション取得.
     */
    private final Connect getConnect( AdminConnectBean bean )
        throws Exception {
        
        Connect conn =  ConnectFactory.get( bean.getName() ) ;
        if( conn == null ) {
            throw new MaachangQException(
                "コネクション(" + bean.getName() +
                ")は存在しません",
                MaachangQErrorCode.ERROR_NOT_CONNECT ) ;
        }
        
        return conn ;
    }
}

