/*
 * @(#)ExecutionAdminUser.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.UserInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminUserBean;
import org.maachang.queue.access.protocol.admin.AdminUserProtocol;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginManager;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;


/**
 * ユーザ電文処理.
 *
 * @version 2007/01/15
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionAdminUser
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminUser() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminUserProtocol.CATEGORY_TYPE_ADMIN_USER ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminUserBean bean = new AdminUserBean() ;
        AdminUserProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminUserProtocol.TYPE_ADD_USER :
                this.addUser( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_REMOVE_USER :
                this.removeUser( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_SET_USER_CHANGE_PASSWD :
                this.changePasswd( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_THIS_USER_CHANGE_PASSWD :
                this.changeThisPasswd( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_CHANGE_OWNER :
                this.changeOwner( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_GET_USERS :
                this.getUsers( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_USER_SIZE :
                this.getUserSize( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_USER_BY_ROOT_OWNER :
                this.getRootOwner( session,connect,bean ) ;
                return true ;
            case AdminUserProtocol.TYPE_IS_USER :
                this.isUser( session,connect,bean ) ;
                return true ;
                
        }
        
        return false ;
        
    }
    
    /**
     * ユーザ追加.
     */
    public final void addUser(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        LoginManager.add(
            session,bean.getName(),bean.getParams()[ 0 ],
            bean.getFlags()[ 0 ] ) ;
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"ユーザ追加" ) ;
        
    }
    
    /**
     * ユーザ削除
     */
    public final void removeUser(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        LoginManager.remove( session,bean.getName() ) ;
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"ユーザ削除" ) ;
        
    }
    
    /**
     * パスワード変更.
     */
    public final void changePasswd(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        LoginManager.changePasswd( session,
            bean.getName(),bean.getParams()[ 0 ] ) ;
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"パスワード変更" ) ;
        
    }
    
    /**
     * 現在ログインユーザパスワード変更.
     */
    public final void changeThisPasswd(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        LoginManager.changeThisPasswd( session,
            bean.getParams()[ 1 ],bean.getParams()[ 0 ] ) ;
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"自パスワード変更" ) ;
        
    }
    
    /**
     * 指定権限変更.
     */
    public final void changeOwner(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        LoginManager.changeOwner( session,
            bean.getName(),bean.getFlags()[ 0 ] ) ;
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"権限変更" ) ;
        
    }
    
    /**
     * ユーザ一覧を取得.
     */
    public final void getUsers(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        UserInfo[] info = LoginManager.getUsers( session ) ;
        if( info != null && info.length > 0 ) {
                
            int len = info.length ;
            boolean[] flags = new boolean[ len ] ;
            String[] params = new String[ len ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                flags[ i ] = info[ i ].isRootOwner() ;
                params[ i ] = info[ i ].getUser() ;
            }
            
            bean.setFlags( flags ) ;
            bean.setParams( params ) ;
            
        }
        else {
            bean.setFlags( null ) ;
            bean.setParams( null ) ;
        }
        
        this.sendUser( session,connect,bean ) ;
    }
    
    /**
     * 現在ログインユーザ数取得.
     */
    public final void getConnectUser(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        int size = LoginManager.getLoginUserCount( session,
            bean.getName() ) ;
        bean.setSize( size ) ;
        
        this.sendUser( session,connect,bean ) ;
    }
    
    /**
     * ユーザサイズ取得.
     */
    public final void getUserSize(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        int size = LoginManager.getSize( session ) ;
        bean.setSize( size ) ;
        
        this.sendUser( session,connect,bean ) ;
    }
    
    /**
     * 指定ユーザ権限取得.
     */
    public final void getRootOwner(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        boolean use = LoginManager.getOwner( session,bean.getName() ) ;
        bean.setFlags( new boolean[] { use } ) ;
        
        this.sendUser( session,connect,bean ) ;
        
    }
    
    /**
     * ユーザ存在確認.
     */
    public final void isUser(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        super.checkLogin( session,bean ) ;
        boolean use = LoginManager.isUser( session,bean.getName() ) ;
        
        if( use == true ) {
            super.sendResult( connect,bean,
                MaachangQErrorCode.SUCCESS,"権限変更" ) ;
        }
        else {
            throw new MaachangQException(
                "ユーザは存在しません",
                MaachangQErrorCode.ERROR_NOT_USER) ;
        }
        
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendUser(
        LoginSession session,ConnectObject connect,AdminUserBean bean )
        throws Exception {
        
        byte[] binary = AdminUserProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
}

